"""Parser to deal with GenBank files and parse them into a useful format.

This uses the Martel-based regular expression in the file genbank_format.py
to describe the GenBank file format. This module contains classes to
utilize this format description to parse things.

classes:
Iterator - Iterate through a file of GenBank entries
ParserCustomizer - A Base class providing attributes for customizing
a GenBank parse.
ErrorFeatureParser - Catch errors caused during parsing.
FeatureParser - Parse a GenBank record in a Seq and SeqFeature object.

_FeatureConsumer - Create SeqFeature objects from info generated by the Scanner
_PrintingConsumer - A debugging consumer.

_Scanner - Set up a Martel based GenBank parser to parse a record.
_EventGenerator - SAX handler to generate biopython-events from XML tags
produced by Martel.

ParserFailureError - Exception indicating a failure in the parser (ie.
scanner or consumer).
"""
# standard library
import string

# XML from python 2.0
from xml.sax import handler

# Martel
import Martel
from Martel import RecordReader

# other Biopython stuff
from Bio.SeqRecord import SeqRecord
from Bio.Alphabet import IUPAC
from Bio.Seq import Seq
from Bio import File
from Bio.ParserSupport import AbstractConsumer

import genbank_format
from Bio.PGML.SeqFeature.Reference import Reference
from Bio.PGML.SeqFeature import Location
from Bio.PGML.SeqFeature import SeqFeature

class Iterator:
    """Iterator interface to move over a file of GenBank entries one at a time.
    """
    def __init__(self, handle, parser = None):
        """Initialize the iterator.

        Arguments:
        o handle - A handle with GenBank entries to iterate through.
        o parser - An optional parser to pass the entries through before
        returning them. If None, then the raw entry will be returned.
        """
        self._reader = RecordReader.StartsWith(handle, "LOCUS")
        # self._reader = RecordReader.EndsWith(handle, "//\n")
        self._parser = parser

    def next(self):
        """Return the next GenBank record from the handle.

        Will return None if we ran out of records.
        """
        data = self._reader.next()

        if self._parser is not None:
            if data:
                return self._parser.parse(File.StringHandle(data))

        return data

class ParserCustomizer:
    """Base class for Parsers which provides some attributes for customization.

    Attributes:
    o debug_level - Specify the amount of debugging information Martel
    should spit out. Default is 0 (no debugging info). Ranges from 0 to 2.
    o use_fuzziness - Specify whether or not to use fuzzy representations.
    The default is 1 (use fuzziness).
    o error_strictness - The level of strictness to use in reporting errors.
    Valid values are:
      * 2 - Raise errors for every kind of error
      * 1 - Print warnings for every kind of error
      * 0 - Ignore errors without saying anything.
    This is set at 2 by default.
    XXX Right now this probably won't work unless set at 2. I'm not sure
    how I'm going to deal with continuing parsing a record when we've got
    a problem. Maybe I should dump this idea... Okay -- I'm going to
    dump it for now
    """
    def __init__(self):
        self.debug_level = 0
        self.use_fuzziness = 1
        # self.error_strictness = 2

    def report_error(self, error_type, error_message):
        """Report errors based on the specified error strictness.

        XXX Not useful unless I bring this back -- maybe I'll just
        trash it...

        Arguments:
        o error_type - An appropriate python error type to raise. This
        error_type will be printed if the strictness is set at 1.
        o error_message - The message associated with the error.
        """
        if self.error_strictness == 2:
            raise error_type(error_message)
        elif self.error_strictiness == 1:
            print "Warning: %s : %s" % error_type, error_message
        else:
            pass

class ParserFailureError(Exception):
    """Failure caused by some kind of problem in the parser.
    """
    pass

class ErrorFeatureParser:
    """Parse GenBank files into Seq + Feature objects, trying to catch errors.

    This is just a small wrapper class around the FeatureParser which
    catches errors that occur in the parser.

    When errors occur, we'll raise a ParserFailureError, which returns
    the accession number of the record that caused the error.
    """
    def __init__(self, debug_level = 0, bad_file_handle = None):
        """Initialize an ErrorFeatureParser.

        Arguments:
        o debug_level - The Martel debugging level.
        o bad_file_handle - A handle to write problem GenBank files to
        If None, the files will not be saved.
        """
        self._bad_file_handle = bad_file_handle

        self._parser = FeatureParser(debug_level)

    def parse(self, handle):
        """Parse the specified handle.
        """
        record = handle.read()

        try:
            return self._parser.parse(File.StringHandle(record))
        # deal with any problems as Parser Errors
        except:
            if self._bad_file_handle:
                self._bad_file_handle.write(record)

            raise ParserFailureError("Could not parse record %s" %
                                     self._parser._consumer.data.id)
                                                          
class FeatureParser(ParserCustomizer):
    """Parse GenBank files into Seq + Feature objects.
    """
    def __init__(self, debug_level = 0):
        """Initialize a GenBank parser and Feature consumer.

        Arguments:
        o debug_level - An optional argument that species the amount of
        debugging information Martel should spit out. By default we have
        no debugging info (the fastest way to do things), but if you want
        you can set this as high as two and see exactly where a parse fails.
        """
        ParserCustomizer.__init__(self)
        self.debug_level = debug_level
        self._scanner = _Scanner()

    def parse(self, handle):
        """Parse the specified handle.
        """
        self._consumer = _FeatureConsumer(self)
        self._scanner.feed(handle, self._consumer, self.debug_level)
        return self._consumer.data

class _FeatureConsumer(AbstractConsumer):
    """Create a SeqRecord object with Features to return.

    Attributes:
    o parser - The parser using this consumer. This allows us to get at
    customization values. If not specified, we'll just use the default
    parser customizer.
    """
    def __init__(self, parser = None):
        self.data = SeqRecord(None)

        if parser:
            self._parser = parser
        else:
            self._parser = ParserCustomizer()

        self._seq_type = ''
        self._seq_data = ''
        self._current_ref = None
        self._cur_feature = None
        self._cur_qualifier_key = None
        self._cur_qualifier_value = None

    def locus(self, locus_name):
        """Set the locus name is set as the name of the Sequence.
        """
        self.data.name = locus_name

    def size(self, content):
        pass

    def residue_type(self, type):
        """Record the sequence type so we can choose an appropriate alphabet.
        """
        self._seq_type = type

    def data_file_division(self, division):
        self.data.annotations['data_file_division'] = division

    def date(self, submit_date):
        self.data.annotations['date'] = submit_date 

    def definition(self, definition):
        """Set the definition as the description of the sequence.
        """
        self.data.description = definition

    def accession(self, acc_num):
        """Set the accession number as the id of the sequence.

        If we have multiple accession numbers, the first one passed is
        used.
        """
        if self.data.id == '<unknown id>':
            self.data.id = acc_num

    def nid(self, content):
        self.data.annotations['nid'] = content

    def version(self, version_id):
        """Set the version to overwrite the id.

        Since the verison provides the same information as the accession
        number, plus some extra info, we set this as the id if we have
        a version.
        """
        self.data.id = version_id

    def gi(self, content):
        self.data.annotations['gi'] = content

    def keywords(self, content):
        # process the keywords into a python list
        if content[-1] == '.':
            keywords = content[:-1]
        else:
            keywords = content
        keyword_list = string.split(keywords, ';')
        self.data.annotations['keywords'] = keyword_list

    def source(self, content):
        if content[-1] == '.':
            source_info = content[:-1]
        else:
            source_info = content
        self.data.annotations['source'] = source_info

    def organism(self, content):
        self.data.annotations['organism'] = content

    def taxonomy(self, content):
        if content[-1] == '.':
            tax_info = content[:-1]
        else:
            tax_info = content
        tax_list = string.split(tax_info, ';')
        self.data.annotations['taxonomy'] = tax_list
        
    def reference_num(self, content):
        """Signal the beginning of a new reference object.
        """
        # if we have a current reference that hasn't been added to
        # the list of references, add it.
        if self._current_ref:
            self.data.annotations['references'].append(self._current_ref)
        else:
            self.data.annotations['references'] = []

        self._current_ref = Reference()

    def reference_bases(self, content):
        """Attempt to determine the sequence region the reference entails.

        Possible types of information we may have to deal with:
        
        (bases 1 to 86436)
        (sites)
        (bases 1 to 105654; 110423 to 111122)
        """
        # first remove the parentheses
        ref_base_info = content[1:-1]

        all_locations = []
        # only attempt to get out information if we find the words
        # 'bases' and 'to'
        if (string.find(ref_base_info, 'bases') != -1 and
            string.find(ref_base_info, 'to') != -1):
            # get rid of the beginning 'bases'
            ref_base_info = ref_base_info[5:]
            # split possibly multiple locations using the ';'
            all_base_info = string.split(ref_base_info, ';')

            for base_info in all_base_info:
                start, end = string.split(base_info, 'to')
                this_location = \
                  Location.FeatureLocation(int(string.strip(start)),
                                           int(string.strip(end)))
                all_locations.append(this_location)

        # make sure if we are not finding information then we have
        # the string 'sites' or the string 'bases'
        elif (ref_base_info == 'sites' or
              string.strip(ref_base_info == 'bases')):
            pass
        # otherwise raise an error
        else:
            raise ValueError(
              "Could not parse base info %s in record %s" %
              (ref_base_info, self.data.id))

        self._current_ref.location = all_locations
                
    def authors(self, content):
        self._current_ref.authors = content

    def title(self, content):
        self._current_ref.title = content

    def journal(self, content):
        self._current_ref.journal = content

    def medline_id(self, content):
        self._current_ref.medline_id = content

    def pubmed_id(self, content):
        self._current_ref.pubmed_id = content

    def remark(self, content):
        self._current_ref.comment = content

    def comment(self, content):
        self.data.annotations['comment'] = content

    def start_feature_table(self):
        """Indicate we've got to the start of the feature table.
        """
        # make sure we've added on our last reference object
        if self._current_ref:
            self.data.annotations['references'].append(self._current_ref)
            self._current_ref = None

    def _add_feature(self):
        """Utility function to add a feature to the SeqRecord.

        This does all of the appropriate checking to make sure we haven't
        left any info behind, and that we are only adding info if it
        exists.
        """
        if self._cur_feature:
            # if we have a left over qualifier, add it to the qualifiers
            # on the current feature
            if self._cur_qualifier_key:
                self._cur_feature.qualifiers[self._cur_qualifier_key] = \
                                             self._cur_qualifier_value

            self._cur_qualifier_key = ''
            self._cur_qualifier_value = ''
            self.data.features.append(self._cur_feature)
            
    def feature_key(self, content):
        # if we already have a feature, add it on
        self._add_feature()

        # start a new feature
        self._cur_feature = SeqFeature.SeqFeature()
        self._cur_feature.type = content

        # assume positive strand to start with if we have DNA. The
        # complement in the location will change this later.
        if self._seq_type == "DNA":
            self._cur_feature.strand = 1

    def location(self, content):
        """Parse out location information from the location string.

        This is going to be hard. GenBank features can have a lot of
        crazy stuff like:

        110678..112039
        join(110678..110887,111045..111408,111486..111556,
                     111723..111821,111941..112039)
        complement(117185..119637)
        complement(join(117185..117774,117879..118720,
                     119131..119232,119342..119637))

        We're going to try to deal with all of it, and will raise an
        error if we can't deal with some location.
        """
        self._parse_location_info(content, self._cur_feature)

    def _parse_location_info(self, info, feature):
        """Parse location information and try to extract information from it.

        This will extract one information item from the location info and
        process it. It will then recursively call itself to parse inner
        location information.

        Arguments:
        o info - The information string to parse.
        o feature - The feature to add the information to
        """
        # strip off all extraneous whitespace
        info = string.strip(info)
        
        # a listing of all of the characters that are considered to
        # be in a location
        location_chars = string.digits + '<' + '>'
        
        # base case -- we are out of information
        if info == '':
            return
        # parse a location -- this is another base_case -- we assume
        # we have no information after a single location
        elif info[0] in location_chars:
            self._parse_location(info, feature)
            return
        elif info[:10] == "complement":
            self._parse_complement(info, feature)
        elif info[:4] == "join":
            self._parse_join(info, feature)
        # forget replace -- it is no longer present in newer GenBank files
        # as far as I can tell.
        elif info[:7] == "replace":
            raise ValueError("We cannot parse replace.")
        # parse a cross reference in the location
        # ie. something like A00001:(1..10)
        # this is also a base case -- we return after getting this
        elif info[0] not in location_chars:
            self._parse_cross_ref(info, feature)
        # otherwise we are stuck and should raise an error
        else:
            raise ValueError("Could not parse location info: %s"
                             % info)

    def _parse_location(self, info, feature):
        """Retrieve information from a location and add it to the feature.
        """
        location_info = self._get_location(info)
        feature.location = location_info

    def _parse_cross_ref(self, info, feature):
        """Retrieve cross reference info and add it to the current feature.

        The cross ref looks like:
        A0001:(1..100)
        """
        # first find the reference value
        colon_break = string.find(info, ':')
        if colon_break == -1:
            raise ValueError("Could not parse location info: %s" % info)
        ref_info = info[:colon_break]
        feature.ref = ref_info

        # now parse anything that is inside the reference, getting
        # rid of the parenthesese
        self._parse_location_ref(info[(colon_break + 2):-1])
        
    def _parse_complement(self, info, feature):
        """Mark a complement, and recursively parse the info inside of it.
        """
        # mark the current feature as being on the opposite strand
        feature.strand = -1

        # recursively call ourselves to parse anything inside the
        # complement
        right_paren = string.rfind(info, ")")
        if right_paren == -1:
            raise ValueError("Cannot parse location %s" % info)
        
        self._parse_location_info(string.strip
                                  (info[11:right_paren]),
                                  feature)
        # recursively call ourselves to parse anything after the
        # complement, if there is anything left
        self._parse_location_info(string.strip
                                  (info[(right_paren + 1):]),
                                  feature)
  
    def _parse_join(self, info, feature):
        """Parse a join and all of the information in it.
        """
        # find the second parenthesis
        second_paren = string.rfind(info, ")")
        if second_paren == -1:
            raise ValueError("Cannot parse location %s" % info)

        # split out all of the individual elements of the join
        join_elements = string.split(info[5:second_paren], ',')

        # for each join element, create a sub SeqFeature within the
        # current feature, then parse the information for this feature
        for join_element in join_elements:
            new_sub_feature = SeqFeature.SeqFeature()
            # add _span to the name, following the lead of Bioperl
            new_sub_feature.type = feature.type + '_span'
            # inherit references and strand from the parent feature
            new_sub_feature.ref = feature.ref
            new_sub_feature.strand = feature.strand

            # parse the information in the join
            self._parse_location_info(join_element, new_sub_feature)

            # now add the feature to the sub_features
            feature.sub_features.append(new_sub_feature)

        # set the location of the top -- this should be a combination of
        # the start position of the first sub_feature and the end position
        # of the last sub_feature
        feature_start = feature.sub_features[0].location.start
        feature_end = feature.sub_features[-1].location.end
        feature.location = Location.FeatureLocation(feature_start,
                                                    feature_end)

        # recursively call ourselves to parse whatever is left
        self._parse_location_info(string.strip
                                  (info[(second_paren + 1):]),
                                  feature)
            
    def _get_location(self, location_range):
        """Return a (possibly fuzzy) location from start and end coordinates.

        This returns a FeatureLocation object, depending on the info
        in location_range. location_range should just be straight out of
        a feature listing, so they can be things like
        110..120
        >110..120
        100..<120
        100^101
        (100.110)..120
        100

        If parser.use_fuzziness is set at one, the positions for the
        end points will possibly be fuzzy.
        """
        # first try to split the first and last by '..'
        try:
            start, end = string.split(location_range, '..')
        except ValueError:
            # we might just have a single base (ie. 100)
            try:
                location = int(location_range)
                pos = self._get_position(str(location))
                return Location.FeatureLocation(pos, pos)
            except ValueError:
                raise ValueError("Could not parse location %s" %
                                 location_range)

        # now get *Position objects for the start and end
        start_pos = self._get_position(start)
        end_pos = self._get_position(end)

        return Location.FeatureLocation(start_pos, end_pos)

    def _get_position(self, position):
        """Return a (possibly fuzzy) position for a single coordinate.

        This is used with _get_location to parse out a location of any
        end_point of arbitrary fuzziness.
        """
        final_pos = None
        # case 1 -- we've got a > sign
        if string.find(position, '>') != -1:
            # remove the >
            try:
                int_pos = int(string.replace(position, '>', ''))
                final_pos = Location.AfterPosition(int_pos)
            except ValueError:
                raise ValueError("Could not parse endpoint %s" %
                                 position)
        # case 2 -- we've got a < sign
        elif string.find(position, '<') != -1:
            # remove the <
            try:
                int_pos = int(string.replace(position, '<', ''))
                final_pos = Location.BeforePosition(int_pos)
            except ValueError:
                raise ValueError("Could not parse endpoint %s" %
                                 position)
        # case 3 -- we've got 100^101
        elif string.find(position, '^') != -1:
            try:
                start, end = string.split(position, '^')
                final_pos = Location.BetweenPosition(int(start),
                                                     int(end) - int(start))
            except ValueError:
                raise ValueError("Could not parse endpoint %s" %
                                 position)
        # case 4 -- we've got (100.101)
        elif string.find(position, '.') != -1:
            try:
                # move the first and last parentheses
                position = position[1:-1]
                start, end = string.split(position, '.')
                final_pos = Location.WithinPosition(int(start),
                                                    int(end) - int(start))
            except ValueError:
                raise ValueError("Could not parse endpoint %s" %
                                 position)
        # case 5 -- just a normal number!
        else:
            try:
                final_pos = Location.ExactPosition(int(position))
            except ValueError:
                raise ValueError("Could not parse endpoint %s" %
                                 position)

        # if we are using fuzziness return what we've got
        if self._parser.use_fuzziness:
            return final_pos
        # otherwise return an ExactPosition equivalent
        else:
            return Location.ExactPosition(final_pos.location)
                
        
    def qualifier_key(self, content):
        """When we get a qualifier key, use it as a dictionary key.

        XXX There is a bug here -- we lose qualifier information if
        there are multiple qualifier keys of the same type in a feature.
        What should we do? Add them like qualifier_key1, qualifier_key2?
        Don't worry about it? Hmmm, maybe we should look at what
        Bioperl and Biojava do...
        """
        # if we've got a key from before, add it to the dictionary of
        # qualifiers
        if self._cur_qualifier_key:
            self._cur_feature.qualifiers[self._cur_qualifier_key] = \
                                         self._cur_qualifier_value

        # remove the / and = from the qualifier
        qual_key = string.replace(content, '/', '')
        qual_key = string.replace(qual_key, '=', '')
        
        self._cur_qualifier_key = qual_key
        self._cur_qualifier_value = ''
        
    def qualifier_value(self, content):
        # get rid of the quotes surrounding the qualifier if we've got 'em
        qual_value = string.replace(content, '"', '')
        
        self._cur_qualifier_value = qual_value

    def base_count(self, content):
        pass

    def base_number(self, content):
        pass

    def sequence(self, content):
        """Add up sequence information as we get it.
        """
        new_seq = string.replace(content, ' ', '')
        new_seq = string.upper(new_seq)

        self._seq_data += new_seq

    def record_end(self):
        """Clean up when we've finished the record.
        """
        # add the last feature in the table which hasn't been added yet
        self._add_feature()

        # add the sequence information
        # first, determine the alphabet -- we'll default to a DNA alphabet
        # if no type information was specified (this is probably unlikely,
        # I guess).
        seq_alphabet = IUPAC.ambiguous_dna

        if self._seq_type:
            if self._seq_type == "DNA":
                seq_alphabet = IUPAC.ambiguous_dna
            elif string.find(self._seq_type, 'RNA') != -1:
                seq_alphabet = IUPAC.ambiguous_rna
            elif self._seq_type == "PROTEIN":
                seq_alphabet = IUPAC.protein
            else:
                raise ValueError("Did not expect sequence type %s" %
                                 self._seq_type)

        # now set the sequence
        self.data.seq = Seq(self._seq_data, seq_alphabet)

class _Scanner:
    """Start up Martel to do the scanning of the file.

    This initialzes the Martel based parser and connects it to a handler
    that will generate events for a Feature Consumer.
    """
    def feed(self, handle, consumer, debug = 0):
        """Feeed a set of data into the scanner.

        Arguments:
        o handle - A handle with the information to parse.
        o consumer - The consumer that should be informed of events.
        o debug - Specify how much debugging info for Martel to
        generate. See the comments for the parsers for more info.
        """
        parser = genbank_format.record.make_parser(debug_level = debug)
        parser.setContentHandler(_EventGenerator(consumer))
        parser.setErrorHandler(handler.ErrorHandler())

        parser.parseFile(handle)

class _EventGenerator(handler.ContentHandler):
    """Handler to generate events associated with the GenBank files.

    This acts like a normal SAX handler, and accepts XML generated by
    Martel during parsing. These events are then converted into
    'Biopython events', which can then be caught by a standard
    biopython consumer
    """
    def __init__(self, consumer):
        """Initialize to begin catching and firing off events.

        Arguments:
        o consumer - The consumer that we'll send Biopython events to.
        """
        self._consumer = consumer

        # a listing of all tags we are interested in
        self.interest_tags = ["locus", "size", "residue_type",
                              "data_file_division", "date",
                              "definition", "accession", "nid", "version",
                              "gi", "keywords", "source", "organism",
                              "taxonomy", "reference_num",
                              "reference_bases", "authors", "title",
                              "journal", "medline_id", "pubmed_id",
                              "remark", "comment",
                              "features_line", "feature_key",
                              "location", "qualifier_key",
                              "qualifier_value",
                              "base_count", "base_number",
                              "sequence", "record_end"]

        # a dictionary of flags to recognize when we are in different
        # info items
        self.flags = {}
        for tag in self.interest_tags:
            self.flags[tag] = 0

        # a dictionary of content for each tag of interest
        self.info = {}
        for tag in self.interest_tags:
            self.info[tag] = ''

    def _get_set_flags(self):
        """Return a listing of all of the flags which are set as positive.
        """
        set_flags = []
        for tag in self.flags.keys():
            if self.flags[tag] == 1:
                set_flags.append(tag)

        return set_flags
                      
    def startElement(self, name, attrs):
        """Recognize when we are recieving different items from Martel.

        We want to recognize when Martel is passing us different items
        of interest, so that we can collect the information we want from
        the characters passed.
        """
        # set the appropriate flag if we are keeping track of these flags
        if name in self.flags.keys():
            # make sure that all of the flags are being properly unset
            # and the information is being cleared.
            assert self.flags[name] == 0, "Flag % not unset" % name
            assert self.info[name] == '', "Info %s not properly reset" % name
            
            self.flags[name] = 1

    def characters(self, content):
        """Extract the information.

        Using the flags that are set, put the character information in
        the appropriate place.
        """
        set_flags = self._get_set_flags()

        # deal with each flag in the set flags
        for flag in set_flags:
            if (flag == "reference" or flag == "record_end" or
                flag == "features_line"):
                pass
            # default -- add up the character information we are getting
            else:
                self.info[flag] += content

    def endElement(self, name):
        """Send the information to the consumer.

        Once we've got the end element we've collected up all of the
        character information we need, and we need to send this on to
        the consumer to do something with it.
        """
        # only deal with the tag if it is something we are
        # interested in and potentially have information for
        if name in self._get_set_flags():
            # specific points in the parse where we want to inform
            # the consumer about where we are at
            if name == "record_end":
                self._consumer.record_end()
            elif name == "features_line":
                self._consumer.start_feature_table()
            # the default is to strip off whitespace and call the
            # consumer
            else:
                fun_to_call = eval("self._consumer" + "." + name)
                info_to_pass = string.strip(self.info[name])
                apply(fun_to_call, (info_to_pass,))

            # reset everything
            self.flags[name] = 0
            self.info[name] = ''

class _PrintConsumer:
    """Just a debugging consumer.

    Want to make sure I've got all of the appropriate return functions
    and that all of the data is being set properly.
    """
    def __init__(self):
        self.data = 'blah'

    def locus(self, content):
        print "locus", content

    def size(self, content):
        print "size", content

    def residue_type(self, content):
        print "residue_type:", content

    def data_file_division(self, content):
        print "data_file_division:", content

    def date(self, content):
        print "date:", content

    def definition(self, content):
        print "definition:", content

    def accession(self, content):
        print "accession:", content

    def nid(self, content):
        print "nid:", content

    def version(self, content):
        print "version:", content

    def gi(self, content):
        print "gi:", content

    def keywords(self, content):
        print "keywords:", content

    def source(self, content):
        print "source:", content

    def organism(self, content):
        print "organism", content

    def taxonomy(self, content):
        print "taxonomy:", content

    def reference_num(self, content):
        print "reference_num:", content

    def reference_bases(self, content):
        print "reference_bases:", content

    def authors(self, content):
        print "authors:", content

    def title(self, content):
        print "title", content

    def journal(self, content):
        print "journal", content

    def medline_id(self, content):
        print "medline_id:", content

    def pubmed_id(self, content):
        print "pubmed_id:", content

    def remark(self, content):
        print "remark:", content

    def comment(self, content):
        print "comment:", content

    def start_feature_table(self):
        print "Starting feature table"

    def feature_key(self, content):
        print "Feature key:", content

    def location(self, content):
        print "Location:", content

    def qualifier_key(self, content):
        print "qualifier key:", content

    def qualifier_value(self, content):
        print "qualifier_value:", content

    def base_count(self, content):
        print "base_count:", content

    def base_number(self, content):
        print "base_count:", content

    def sequence(self, content):
        print "sequence:", content

    def record_end(self):
        print "End of Record!"
            
            
        
        
    
    
    
