"""Specify locations of a feature on a Sequence.

This aims to handle, in Ewan's words, 'the dreaded fuzziness issue' in
much the same way as Biocorba. This has the advantages of allowing us
to handle fuzzy stuff in case anyone needs it, and also be compatible
with Biocorba.

classes:
o FeatureLocation - Specify the start and end location of a feature.

o ExactPosition - Specify the position as being exact.
o WithinPosition - Specify a position occuring within some range.
o BetweenPosition - Specify a position occuring between a range.
o BeforePosition - Specify the position as being found before some base.
o AfterPosition - Specify the position as being found after some base.
"""

class FeatureLocation:
    """Specify the location of a feature along a sequence.

    This attempts to deal with fuzziness of position ends, but also
    make it easy to get the start and end in the 'normal' case (no
    fuzziness).

    You should access the start and end attributes with
    your_location.start and your_location.end. If the start and
    end are exact, this will return the positions, if not, we'll return
    the approriate Fuzzy class with info about the position and fuzziness.
    """
    def __init__(self, start, end):
        """Specify the start and end of a sequence feature.

        start and end arguments specify the values where the feature begins
        and ends. These can either by any of the *Position objects that
        inherit from AbstractPosition, or can just be integers specifying the
        position. In the case of integers, the values are assumed to be
        exact and are converted in ExactPosition arguments. This is meant
        to make it easy to deal with non-fuzzy ends.
        """
        if isinstance(start, AbstractPosition):
            self._start = start
        else:
            self._start = ExactPosition(start)

        if isinstance(end, AbstractPosition):
            self._end = end
        else:
            self._end = ExactPosition(end)

    def __str__(self):
        return "(%s..%s)" % (self._start, self._end)

    def __getattr__(self, attr):
        """Make it easy to get non-fuzzy starts and ends.

        We override get attribute here so that in non-fuzzy cases we
        can just return the start and end position without any hassle.
        """
        if attr == 'start':
            if isinstance(self._start, ExactPosition):
                return self._start.position
            else:
                return self._start
        elif attr == 'end':
            if isinstance(self._end, ExactPosition):
                return self._end.position
            else:
                return self._end
        else:
            raise AttributeError("Cannot evaluate attribute %s." % attr)

class AbstractPosition:
    """Abstract base class representing a position.
    """
    def __init__(self, position):
        self.position = position
            
class ExactPosition(AbstractPosition):
    """Specify the specific position of a boundary.

    In this case, there is no fuzziness associated with the position.
    """
    def __init__(self, position):
        AbstractPosition.__init__(self, position)

    def __str__(self):
        return str(self.position)

class WithinPosition(AbstractPosition):
    """Specify the position of a boundary within some coordinates.

    Arguments:
    o position - The start position of the boundary
    o extension - The range to which the boundary can extend.

    This allows dealing with a position like ((1.4)..100). This
    indicates that the start of the sequence is somewhere between 1
    and 4. To represent that with this class we would set position as
    1 and extension as 3.
    """
    def __init__(self, position, extension):
        AbstractPosition.__init__(self, position)
        self.extension = extension

    def __str__(self):
        return "(%s.%s)" % (self.position, self.position + self.extension)

class BetweenPosition(AbstractPosition):
    """Specify the position of a boundary between two coordinates.

    Arguments:
    o position - The start position of the boundary.
    o extension - The range to the other position of a boundary.

    This specifies a coordinate which is found between the two positions.
    So this allows us to deal with a position like ((1^2)..100). To
    represent that with this class we set position as 1 and the
    extension as 1.
    """
    def __init__(self, position, extension):
        AbstractPosition.__init__(self, position)
        self.extension = extension

    def __str__(self):
        return "(%s^%s)" % (self.position, self.position + self.extension)

class BeforePosition(AbstractPosition):
    """Specify a position where the actual location occurs before it.

    Arguments:
    o position - The upper boundary of where the location can occur.

    This is used to specify positions like (<10..100) where the location
    occurs somewhere before position 10.
    """
    def __init__(self, position):
        AbstractPosition.__init__(self, position)

    def __str__(self):
        return "<%s" % self.position

class AfterPosition(AbstractPosition):
    """Specify a position where the actual location is found after it.

    Arguments:
    o position - The lower boundary of where the location can occur.

    This is used to specify positions like (>10..100) where the location
    occurs somewhere after position 10.
    """
    def __init__(self, position):
        AbstractPosition.__init__(self, position)

    def __str__(self):
        return ">%s" % self.position
    
