"""Represent a Sequence Feature holding info about a part of a sequence.

This is heavily modeled after the Biocorba SeqFeature objects, and
may be pretty biased towards GenBank stuff since I'm writing it
for the GenBank parser output...
"""
from Bio.PGML.SeqFeature import Location

class SeqFeature:
    """Represent a Sequence Feature on an object.

    Attributes:
    o location - the location of the feature on the sequence
    o type - the specified type of the feature (ie. CDS, exon, repeat...)
    o ref - A reference to another sequence. This could be an accession
    number for some different sequence.
    o qualifier - A dictionary of qualifiers on the feature. These are
    analagous to the qualifiers from a GenBank feature table. The keys of
    the dictionary are qualifier names, the values are the qualifier
    values.
    o sub_features - Additional SeqFeatures which fall under this 'parent'
    feature. For instance, if we having something like:

    CDS    join(1..10,30..40,50..60)

    The the top level feature would be a CDS from 1 to 60, and the sub
    features would be of 'CDS_span' type and would be from 1 to 10, 30 to
    40 and 50 to 60, respectively.
    """
    def __init__(self):
        """Initialize a SeqFeature on a Sequence.
        """
        self.location = None

        self.type = ''
        self.ref = ''
        self.strand = 0
        self.qualifiers = {}
        self.sub_features = []

    def __str__(self):
        """Make it easier to debug features.
        """
        out = "type: %s\n" % self.type
        out += "location: %s\n" % self.location
        out += "ref: %s\n" % self.ref
        out += "strand: %s\n" % self.strand
        out += "qualifiers: %s\n" % self.qualifiers
        if len(self.sub_features) != 0:
            out += "Sub-Features\n"
            for sub_feature in self.sub_features:
                out +="%s\n" % sub_feature

        return out
