/*
  3D vector routines.

*/

#include "vector3.h"

#include <math.h>
#include <glib.h>


/*------------------------------------------------------------*/
void
v3_initialize (vector3 *dest, const double x, const double y, const double z)
{
  /* pre */
  g_assert (dest);

  dest->x = x;
  dest->y = y;
  dest->z = z;
}

void
v3_eq (vector3 *dest, const vector3 *v)
{
  g_return_if_fail (dest != NULL);
  g_return_if_fail (v != NULL);
  
  dest->x = v->x;
  dest->y = v->y;
  dest->z = v->z;
}


/*------------------------------------------------------------*/
void
v3_sum (vector3 *dest, const vector3 *v1, const vector3 *v2)
     /*
       dest = v1 + v2
     */
{
  /* pre */
  g_assert (dest);
  g_assert (v1);
  g_assert (v2);

  dest->x = v1->x + v2->x;
  dest->y = v1->y + v2->y;
  dest->z = v1->z + v2->z;
}


/*------------------------------------------------------------*/
void
v3_difference (vector3 *dest, const vector3 *v1, const vector3 *v2)
     /*
       dest = v1 - v2
     */
{
  /* pre */
  g_assert (dest);
  g_assert (v1);
  g_assert (v2);

  dest->x = v1->x - v2->x;
  dest->y = v1->y - v2->y;
  dest->z = v1->z - v2->z;
}


/*------------------------------------------------------------*/
void
v3_scaled (vector3 *dest, const double s, const vector3 *src)
     /*
       dest = s * src
     */
{
  /* pre */
  g_assert (dest);
  g_assert (src);

  dest->x = s * src->x;
  dest->y = s * src->y;
  dest->z = s * src->z;
}


/*------------------------------------------------------------*/
void
v3_add (vector3 *dest, const vector3 *v)
     /*
       dest += v
     */
{
  /* pre */
  g_assert (dest);
  g_assert (v);

  dest->x += v->x;
  dest->y += v->y;
  dest->z += v->z;
}


/*------------------------------------------------------------*/
void
v3_subtract (vector3 *dest, const vector3 *v)
     /*
       dest -= v
     */
{
  /* pre */
  g_assert (dest);
  g_assert (v);

  dest->x -= v->x;
  dest->y -= v->y;
  dest->z -= v->z;
}


/*------------------------------------------------------------*/
void
v3_scale (vector3 *dest, const double s)
     /*
       dest *= s
     */
{
  /* pre */
  g_assert (dest);

  dest->x *= s;
  dest->y *= s;
  dest->z *= s;
}


/*------------------------------------------------------------*/
void
v3_invert (vector3 *v)
     /*
       v = (1/x, 1/y, 1/z)
     */
{
  /* pre */
  g_assert (v);
  g_assert (v->x != 0.0);
  g_assert (v->y != 0.0);
  g_assert (v->z != 0.0);

  v->x = 1.0 / v->x;
  v->y = 1.0 / v->y;
  v->z = 1.0 / v->z;
}


/*------------------------------------------------------------*/
void
v3_reverse (vector3 *v)
     /*
       v = -v
     */
{
  /* pre */
  g_assert (v);

  v->x = - v->x;
  v->y = - v->y;
  v->z = - v->z;
}


/*------------------------------------------------------------*/
void
v3_sum_scaled (vector3 *dest, const vector3 *v1,
	       const double s, const vector3 *v2)
     /*
       dest = v1 + s * v2
     */
{
  /* pre */
  g_assert (dest);
  g_assert (v1);
  g_assert (v2);

  dest->x = v1->x + s * v2->x;
  dest->y = v1->y + s * v2->y;
  dest->z = v1->z + s * v2->z;
}


/*------------------------------------------------------------*/
void
v3_add_scaled (vector3 *dest, const double s, const vector3 *v)
     /*
       dest += s * v
     */
{
  /* pre */
  g_assert (dest);
  g_assert (v);

  dest->x += s * v->x;
  dest->y += s * v->y;
  dest->z += s * v->z;
}


/*------------------------------------------------------------*/
double
v3_length (const vector3 *v)
{
  /* pre */
  g_assert (v);

  return sqrt (v->x * v->x + v->y * v->y + v->z * v->z);
}


/*------------------------------------------------------------*/
double
v3_angle (const vector3 *v1, const vector3 *v2)
     /*
       The angle (in radians) between the vectors v1 and v2.
     */
{
  /* pre */
  g_assert (v1);
  g_assert (v2);
  g_assert (v3_length (v1) > 0.0);
  g_assert (v3_length (v2) > 0.0);

  return acos ((v1->x * v2->x + v1->y * v2->y + v1->z * v2->z) /
	       sqrt ((v1->x * v1->x + v1->y * v1->y + v1->z * v1->z) *
		     (v2->x * v2->x + v2->y * v2->y + v2->z * v2->z)));
}


/*------------------------------------------------------------*/
double
v3_angle_points (const vector3 *p1, const vector3 *p2, const vector3 *p3)
     /*
       The angle (in radians) formed between the vector from p2 to p1
       and the vector from p2 to p3.
     */
{
  vector3 v1, v2;

  /* pre */
  g_assert (p1);
  g_assert (p2);
  g_assert (p3);

  v3_difference (&v1, p1, p2);
  v3_difference (&v2, p3, p2);
  return v3_angle (&v1, &v2);
}


/*------------------------------------------------------------*/
double
v3_torsion (const vector3 *p1, const vector3 *v1,
	    const vector3 *p2, const vector3 *v2)
     /*
       The torsion angle (in radians) between the vectors v1 and v2
       when viewed along the vector from p1 to p2.
     */
{
  double angle;
  vector3 dir, x1, x2;

  /* pre */
  g_assert (p1);
  g_assert (v1);
  g_assert (p2);
  g_assert (v2);

  v3_difference (&dir, p2, p1);
  v3_cross_product (&x1, &dir, v1);
  v3_cross_product (&x2, &dir, v2);
  angle = v3_angle (&x1, &x2);
  return (v3_dot_product (v1, &x2) > 0.0) ? angle : -angle;
}


/*------------------------------------------------------------*/
double
v3_torsion_points (const vector3 *p1, const vector3 *p2,
		   const vector3 *p3, const vector3 *p4)
     /*
       The torsion angle (in radians) formed between the points.
     */
{
  vector3 v1, v2;

  /* pre */
  g_assert (p1);
  g_assert (p2);
  g_assert (p3);
  g_assert (p4);
  g_assert (p1 != p2);
  g_assert (p2 != p3);
  g_assert (p3 != p4);

  v3_difference (&v1, p1, p2);
  v3_difference (&v2, p4, p3);
  return v3_torsion (p2, &v1, p3, &v2);
}


/*------------------------------------------------------------*/
double
v3_dot_product (const vector3 *v1, const vector3 *v2)
{
  /* pre */
  g_assert (v1);
  g_assert (v2);

  return v1->x * v2->x + v1->y * v2->y + v1->z * v2->z;
}


/*------------------------------------------------------------*/
void
v3_cross_product (vector3 *vz, const vector3 *vx, const vector3 *vy)
{
  /* pre */
  g_assert (vz);
  g_assert (vx);
  g_assert (vy);
  g_assert (vz != vx);
  g_assert (vz != vy);

  vz->x = vx->y * vy->z - vx->z * vy->y;
  vz->y = vx->z * vy->x - vx->x * vy->z;
  vz->z = vx->x * vy->y - vx->y * vy->x;
}


/*------------------------------------------------------------*/
void
v3_triangle_normal (vector3 *n,
		    const vector3 *p1, const vector3 *p2, const vector3 *p3)
     /*
       The normal vector n for the plane defined by the three points.
     */
{
  vector3 p12, p23;

  /* pre */
  g_assert (n);
  g_assert (p1);
  g_assert (p2);
  g_assert (p3);
  g_assert (v3_distance (p1, p2) > 0.0);
  g_assert (v3_distance (p2, p3) > 0.0);
  g_assert (v3_distance (p3, p1) > 0.0);

  v3_difference (&p12, p2, p1);
  v3_difference (&p23, p3, p2);
  v3_cross_product (n, &p12, &p23);
  v3_normalize (n);
}


/*------------------------------------------------------------*/
double
v3_sqdistance (const vector3 *p1, const vector3 *p2)
     /*
       The squared distance between the points.
     */
{
  register double xdiff, ydiff, zdiff;

  /* pre */
  g_assert (p1);
  g_assert (p2);

  xdiff = p1->x - p2->x;
  ydiff = p1->y - p2->y;
  zdiff = p1->z - p2->z;

  return xdiff * xdiff + ydiff * ydiff + zdiff * zdiff;
}


/*------------------------------------------------------------*/
double
v3_distance (const vector3 *p1, const vector3 *p2)
{
  register double xdiff, ydiff, zdiff;

  /* pre */
  g_assert (p1);
  g_assert (p2);

  xdiff = p1->x - p2->x;
  ydiff = p1->y - p2->y;
  zdiff = p1->z - p2->z;

  return sqrt (xdiff * xdiff + ydiff * ydiff + zdiff * zdiff);
}


/*------------------------------------------------------------*/
int
v3_close (const vector3 *v1, const vector3 *v2, const double sqdistance)
     /*
       Are the two points closer than the given squared distance?
     */
{
  register double xdiff;

  /* pre */
  g_assert (v1);
  g_assert (v2);

  xdiff = v1->x - v2->x;
  xdiff *= xdiff;
  if (xdiff > sqdistance) {
    return FALSE;
  } else {
    register double ydiff = v1->y - v2->y;
    ydiff *= ydiff;
    if (ydiff > sqdistance) {
      return FALSE;
    } else {
      register double zdiff = v1->z - v2->z;
      return (xdiff + ydiff + zdiff * zdiff) <= sqdistance;
    }
  }
}


/*------------------------------------------------------------*/
void
v3_normalize (vector3 *v)
     /*
       v /= length (v)
    */
{
  register double ir;

  /* pre */
  g_assert (v);
  g_assert (v3_length (v) > 0.0);

  ir = 1.0 / sqrt (v->x * v->x + v->y * v->y + v->z * v->z);
  v->x *= ir;
  v->y *= ir;
  v->z *= ir;
}


/*------------------------------------------------------------*/
void
v3_middle (vector3 *dest, const vector3 *p1, const vector3 *p2)
     /*
       dest = (p1 + p2) / 2
     */
{
  /* pre */
  g_assert (dest);
  g_assert (p1);
  g_assert (p2);

  dest->x = 0.5 * (p1->x + p2->x);
  dest->y = 0.5 * (p1->y + p2->y);
  dest->z = 0.5 * (p1->z + p2->z);
}


/*------------------------------------------------------------*/
void
v3_between (vector3 *dest,
	    const vector3 *p1, const vector3 *p2, const double fraction)
     /*
       dest = fraction * (p2 - p1) + p1
     */
{
  /* pre */
  g_assert (dest);
  g_assert (p1);
  g_assert (p2);

  dest->x = fraction * (p2->x - p1->x) + p1->x;
  dest->y = fraction * (p2->y - p1->y) + p1->y;
  dest->z = fraction * (p2->z - p1->z) + p1->z;
}


/*------------------------------------------------------------*/
int
v3_different (const vector3 *v1, const vector3 *v2)
{
  /* pre */
  g_assert (v1);
  g_assert (v2);

  return ((v1->x != v2->x) || (v1->y != v2->y) || (v1->z != v2->z));
}


/* -- Test -- * /

int
main ()
{
  vector3 v1;
  vector3 v2;
  
  v3_initialize (&v1, 2, -1, 3);
  v3_initialize (&v2, -1, 2, 3);
  
  v3_add (&v1, &v2);
  
  return 0;
}
*/
