#include <gnome.h>
#include "trackball.h"
#include "viewer.h"
#include "drawer.h"

// static int accum = 0;
// static jitter_point *jitter_points;

static GLuint pick_buffer[512];
static int    pick_atom = 0;

static GLuint fontbase;
static gboolean is_label_show = FALSE;

static GList *viewer_list = NULL;
static Viewer *cur_viewer = NULL;

static GtkWidget *note_book = NULL;


/* -- Mouse Cursor Shape -- */
static void
set_cursor_shape (int shape)
{
  static GdkCursor *cursor;
  static int prev_shape = -1;
  
  if (prev_shape != -1)
    gdk_cursor_destroy (cursor);
  
  cursor = gdk_cursor_new (shape);
  gdk_window_set_cursor (GTK_WIDGET (cur_viewer->glarea)->window, cursor);
  prev_shape = shape;
}

/* --- Pick --- */
static void
pick_start (void)
{
  glSelectBuffer (512, pick_buffer);
  glRenderMode (GL_SELECT);
  
  glInitNames ();
  glPushName (0);
  
}


static void
pick_end (void)
{
  int hits = glRenderMode (GL_RENDER);
  /*  
  g_print ("hits = %d \n", hits);
  */
  if (hits > 1)
    {
      is_label_show = TRUE;
      pick_atom = pick_buffer[(hits-1)*4 + 3];
    }
  
}


/* --- Render --- */

static void
gl_render_init (Viewer *viewer)
{
  int render_mode = viewer->render_mode;
  int mouse_option = viewer->mouse_option;
  
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
 
  if (render_mode == GL_SELECT && mouse_option == PICK_MODE)
    {
      GLint viewport[4];
      
      glGetIntegerv (GL_VIEWPORT, viewport);
      gluPickMatrix (viewer->mouse_x,
		     viewport[3] - viewer->mouse_y,
		     5.0, 5.0, viewport);
    }
  
  switch (viewer->project_option)
    {
    case PERSPECTIVE_MODE:
      {
	gluPerspective (viewer->fovy, 1.0, 1.0, 100.0);
	break;
      }
    case ORTHO_MODE:
      {
	glOrtho (-20 * viewer->scale,
		 20  * viewer->scale,
		 -20 * viewer->scale,
		 20  * viewer->scale,
		 0, 80);

	break;
      }
    }
    
  glMatrixMode(GL_MODELVIEW);

  /* clear background */
  glClearColor(0,0,0,1);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  glDepthFunc(GL_LESS);
  glEnable(GL_DEPTH_TEST);
}


static void
gl_render_lists (Viewer *viewer)
{
  GSList *slist;
  gint    num;
  
  for (slist = viewer->display_slist; slist; slist = slist->next)
    {
      num = *(int*)slist->data;
      glCallList (num);
    }
      
  
  if (g_slist_length (viewer->label_slist) > 0) 
    {
      glDisable (GL_LIGHTING);
      
      for (slist = viewer->label_slist; slist; slist = slist->next)
	{
	  num = *(int*) slist->data;
	  glCallList (num);
	}
      
      glEnable (GL_LIGHTING);
    }
  /*
  if (g_slist_length (alpha_display_slist) > 0)
    {
      glEnable (GL_BLEND);
      glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      
      for (slist = alpha_display_slist; slist; slist = slist->next)
	{
	  num = *(int*)slist->data;
	  glCallList (num);
	}
      glDisable (GL_BLEND);
    }
  */
}


static void
window_render (Viewer *viewer)
{
  int render_mode = viewer->render_mode;
  
  if (render_mode == GL_SELECT && viewer->mouse_option == PICK_MODE)
    pick_start ();
  
  /* -- view -- */
  gl_render_init (viewer);
  
  /* --  draw object -- */
  glLoadIdentity ();
  
  glTranslatef (viewer->trans_x, viewer->trans_y, -30);

  build_rotmatrix (viewer->xform, viewer->quat);
  glMultMatrixf (&(viewer->xform[0][0]));
  
  gl_render_lists (viewer);
  
  if (render_mode == GL_SELECT && viewer->mouse_option == PICK_MODE)
    pick_end ();
}

  
/* --- Draw -- */

void
viewer_draw (Viewer *viewer)
{
  if (gtk_gl_area_make_current (GTK_GL_AREA (viewer->glarea)))
    {
      /* (*(viewer->draw)) (viewer); */
      window_render (viewer);
      gtk_gl_area_swapbuffers (GTK_GL_AREA (viewer->glarea));
    }
}

	

/* -- Signal Connect --- */
static gint
viewer_expose (GtkWidget *widget, GdkEventExpose *event, Viewer *viewer)
{
  if (event->count > 0)
    {
      return FALSE;
    }
  
  viewer_draw (viewer);
  
  return TRUE;
}



static gint
viewer_reshape (GtkWidget * wid, GdkEventConfigure * event, Viewer * viewer)
{
  
  if (gtk_gl_area_make_current(GTK_GL_AREA (viewer->glarea)))
    {
      viewer->width  = wid->allocation.width;
      viewer->height = wid->allocation.height;
      viewer->aspect = (GLdouble) (wid->allocation.width / wid->allocation.height);
      
      glViewport (0, 0, viewer->width, viewer->height);
      gtk_gl_area_swapbuffers (viewer->glarea);
    }
  
  return TRUE;
}


static gint
viewer_init (GtkWidget * widget, Viewer * viewer)
{
  GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
  GLfloat mat_shininess[] = { 50.0 };
  GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };
  GdkFont *font;
  char *fontname = "-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*";

  if (gtk_gl_area_make_current (GTK_GL_AREA (viewer->glarea)))
    {
      glClearColor (0.0, 0.0, 0.0, 0.0);
      glShadeModel (GL_SMOOTH);

      glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
      glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
      glLightfv(GL_LIGHT0, GL_POSITION, light_position);

      glEnable(GL_LIGHTING);
      glEnable(GL_LIGHT0);
      glEnable(GL_DEPTH_TEST);

      /*  generate font display lists */
      font = gdk_font_load (fontname);
      
      if (!font)
	{
	  g_print("Can't load font '%s'\n", fontname);
	  gtk_exit(1);
	}

      gdk_gl_use_gdk_font (font, 0, 128, fontbase);
      /*      fontheight = font->ascent + font->descent; */
      gdk_font_unref (font);
      
    }
  
  return TRUE;
}


static gint
viewer_button_press (GtkWidget *wid, GdkEventButton *event, Viewer *viewer)
{
  cur_viewer = viewer;

  viewer->mouse_x = event->x;
  viewer->mouse_y = event->y;

  if (event->button == 1)
    viewer->mouse_option = PICK_MODE;
  else if (event->button == 2)
    {
      set_cursor_shape (GDK_FLEUR);
      viewer->mouse_option = TRANSLATE_MODE;
    }
  else if (event->button == 3)
    {
      if (event->state & GDK_SHIFT_MASK)
	{
	  set_cursor_shape (GDK_DOUBLE_ARROW);
	  viewer->mouse_option = ZOOM_MODE;
	}
      else
	{
	  set_cursor_shape (GDK_CIRCLE);
          viewer->mouse_option = ROTATE_MODE;
	}
    }
  
  return TRUE;
}

static gint
viewer_button_release (GtkWidget *wid, GdkEventButton *event, Viewer *viewer)
{
  gint button;
  
  button = event->button;
  
  if (button == 1)
    {
      is_label_show = FALSE;
      
      if (viewer->render_mode == GL_RENDER)
	{
	  viewer->render_mode = GL_SELECT;
	  display_pick (viewer);
	}
      gtk_widget_draw (GTK_WIDGET (viewer->glarea), NULL);
      
      if (is_label_show)
	{
	  vector3 *pos = (vector3 *) g_list_nth_data (viewer->mol->pos_list,
						      pick_atom);
	  AtomInfo *atom =
	    (AtomInfo *) g_list_nth_data (viewer->mol->atom_list, pick_atom);
	  display_label (viewer, pos, atom->name, fontbase);
	  
	  /* Tricks to show label on the OpenGL window */
	  viewer->render_mode = GL_RENDER;
	  gtk_widget_draw (GTK_WIDGET (viewer->glarea), NULL);

	  viewer->render_mode = GL_SELECT;
	}
    }

  set_cursor_shape (GDK_LEFT_PTR);
  
  return TRUE;
}


gint
viewer_motion (GtkWidget *widget, GdkEventMotion *event, Viewer *viewer)
{
  GdkModifierType state;
  GdkRectangle area;
  int x, y;
  GLfloat spin_quat[4];

  if (cur_viewer == NULL)
    return TRUE;
  
  if (event->is_hint)
    gdk_window_get_pointer (event->window, &x, &y, &state);
  else
    {
      x = event->x;
      y = event->y;
      state = event->state;
    }
  
  area.x = 0;
  area.y = 0;
  area.width  = widget->allocation.width;
  area.height = widget->allocation.height;

  if (state & GDK_BUTTON1_MASK)
    {
      
      
    }
  if (state & GDK_BUTTON2_MASK)
    {
      cur_viewer->trans_x += ((x - cur_viewer->mouse_x) / area.width) * 10;
      cur_viewer->trans_y -= ((y - cur_viewer->mouse_y) / area.height) * 10;
      gtk_widget_draw (widget, &area);
    }
  if (state & GDK_BUTTON3_MASK)
    {
      if (state & GDK_SHIFT_MASK)
	{
	  switch (cur_viewer->project_option)
	    {
	    case PERSPECTIVE_MODE:
	      cur_viewer->fovy += ((y - cur_viewer->mouse_y)/area.height) * 40;
	      if (cur_viewer->fovy < 5)
		cur_viewer->fovy = 5;
	      if (cur_viewer->fovy > 120)
		cur_viewer->fovy = 120;
	      break;
	    case ORTHO_MODE:
	      cur_viewer->scale += ((y - cur_viewer->mouse_y)/area.height)*2;
	      if (cur_viewer->scale < 1.0e-2)
		cur_viewer->scale = 1.0e-2;
	      break;
	    }
	}
      else
	{
	  trackball (spin_quat, 
		     (2.0*cur_viewer->mouse_x -       area.width) / area.width,
		     (     area.height - 2.0*cur_viewer->mouse_y) / area.height,
		     (           2.0*x -       area.width) / area.width,
		     (     area.height -            2.0*y) / area.height);
	  add_quats(spin_quat, cur_viewer->quat, cur_viewer->quat);
	}
      
      gtk_widget_draw (widget, &area);
    }

  cur_viewer->mouse_x = x;
  cur_viewer->mouse_y = y;
  
  return TRUE;
}



void
viewer_add (Viewer *viewer)
{
  GtkWidget *label_box;
  GtkGLArea *glarea;
  
  gint attrlist[] =
  {
    GDK_GL_RGBA,
    GDK_GL_DOUBLEBUFFER,
    GDK_GL_DEPTH_SIZE, 
    GDK_GL_USE_GL,
    GDK_GL_NONE
  };

  viewer->glarea = glarea = GTK_GL_AREA (gtk_gl_area_new (attrlist));
  viewer_list = g_list_append (viewer_list, viewer);

  gtk_widget_set_events (GTK_WIDGET (glarea),
			 GDK_EXPOSURE_MASK |
			 GDK_BUTTON_PRESS_MASK |
			 GDK_BUTTON_RELEASE_MASK |
			 GDK_POINTER_MOTION_MASK |
			 GDK_POINTER_MOTION_HINT_MASK);

  trackball (viewer->quat, 0.0, 0.0, 0.0, 0.0);
  /*
  glDeleteLists (1, (GLsizei) latest_display_list);
  */

  gtk_signal_connect (GTK_OBJECT (glarea), "expose_event",
		      GTK_SIGNAL_FUNC (viewer_expose), viewer);

  gtk_signal_connect (GTK_OBJECT (glarea), "configure_event",
		      GTK_SIGNAL_FUNC (viewer_reshape), viewer);

  gtk_signal_connect (GTK_OBJECT (glarea), "realize",
		      GTK_SIGNAL_FUNC (viewer_init), viewer);

  gtk_signal_connect (GTK_OBJECT (glarea), "button_press_event",
		      GTK_SIGNAL_FUNC (viewer_button_press), viewer);

  gtk_signal_connect (GTK_OBJECT (glarea), "button_release_event",
		      GTK_SIGNAL_FUNC (viewer_button_release), viewer);

  gtk_signal_connect (GTK_OBJECT (glarea), "motion_notify_event",
		      GTK_SIGNAL_FUNC (viewer_motion), NULL);
  
  gtk_widget_show (GTK_WIDGET (glarea));

  label_box = gtk_hbox_new (FALSE, 0);

  gtk_box_pack_start (GTK_BOX (label_box), viewer->label_name, FALSE, TRUE, 0);
  gtk_widget_show_all (label_box);
  
  gtk_notebook_prepend_page (GTK_NOTEBOOK (note_book),
			    GTK_WIDGET (glarea), label_box);

  gtk_notebook_set_page (GTK_NOTEBOOK (note_book), 0); /* show first page */
}

Viewer *
viewer_new_with_molecule (Molecule *mol, char *name)
{
  static  GtkWidget *window = NULL;
  GtkWidget *box;
  Viewer *viewer;
  
  if (!window)
    {
      window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
           
      gtk_signal_connect (GTK_OBJECT (window), "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &window);
     
      gtk_window_set_title (GTK_WINDOW (window), "OpenGL Viewer");
      gtk_container_set_border_width (GTK_CONTAINER (window), 0);
      gtk_widget_set_usize (window, 500, 550);
      gtk_widget_set_uposition (window, 0, 0);
      
      box = gtk_vbox_new (FALSE, 0);
      gtk_container_add (GTK_CONTAINER (window), box);
      
      note_book = gtk_notebook_new ();
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (note_book), GTK_POS_TOP);
      gtk_box_pack_start (GTK_BOX (box), note_book, TRUE, TRUE, 0);

      gtk_widget_realize (note_book);
      
    }
  
  cur_viewer = viewer = g_new0 (Viewer, 1);
  
  viewer_list = g_list_append (viewer_list, viewer);
  
  viewer->glarea = NULL;
  
  viewer->fovy = 45.0;
  viewer->slab = 40;
  viewer->trans_x = 0;
  viewer->trans_y = 0;
  viewer->scale = 1.;
  
  viewer->project_option = ORTHO_MODE;
  /*
  viewer->mouse_option = ROTATE_MODE;
  */
  viewer->render_mode = GL_RENDER;
  
  viewer->label_name = gtk_label_new (name);
  
  viewer->mol = mol;
  viewer->mid = g_list_length (viewer_list);
  viewer->is_drawed = FALSE;
  
  viewer_add (viewer);

  if (!GTK_WIDGET_VISIBLE (window))
    gtk_widget_show_all (window);
  /*
  else
    gtk_widget_destroy (window);
  */
  return viewer;
  
}


Viewer *
viewer_new (void)
{
  return viewer_new_with_molecule (NULL, "Model");
}


void
start_display_list (Viewer *viewer, gboolean is_label)
{
  GLint *num;
  
  num = g_new0 (GLint, 1);
  *num = glGenLists (1);
  
  glNewList (*num, GL_COMPILE);
    
  if (is_label)
    {
      viewer->label_slist
	= g_slist_append (viewer->label_slist, num);
    }
/*  else if (current_state->transparency != 0.0)
    {
      alpha_display_slist = g_slist_append (alpha_display_slist, num);
      }*/
  else
    {
      viewer->display_slist
	= g_slist_append (viewer->display_slist, num);
    }
}


void
delete_display_lists (Viewer *viewer)
{
  GLint *num;
  GSList *slist;
  
  for (slist = viewer->label_slist; slist; slist = slist->next)
    {
      num = slist->data;
      glDeleteLists (*num, 1);
    }
  g_slist_free (viewer->label_slist);
  viewer->label_slist = NULL;
  
  for (slist = viewer->display_slist; slist; slist = slist->next)
    {
      num = slist->data;
      glDeleteLists (*num, 1);
    }
  g_slist_free (viewer->display_slist);
  viewer->display_slist = NULL;
}

gint
get_current_mid ()
{
  return cur_viewer->mid;
}


Viewer *
get_viewer_by_mid (gint mid)
{
  GList *list;
  
  for (list = viewer_list; list; list = list->next)
    {
      Viewer *viewer = (Viewer *)list->data;
      
      if (viewer->mid == mid)
	return viewer;
    }
  
  return NULL;
}

	  
      
