#include <glib.h>
#include <math.h>
#include "md.h"
#include "cell.h"
#include "param.h"

void cell_print ();
static gboolean check_is_bond (int ib, int jb);

static int ncells = 0;     /* number of cells in the box */
static int cell_nx = 0;
static int cell_ny = 0;
static int cell_nz = 0;
static vector3 cell_size; 
static int *head_of_cell = NULL;
static int *linked_list = NULL;

typedef struct _cell_neighbor CellNeigh;

struct _cell_neighbor
{
  GList *n_list;
};


CellNeigh *cell_neigh = NULL;


/*
  cell_locate
  return the cell positon
*/

gint
cell_locate (vector3 *pos)
{
  gint ix, iy, iz;
  gint icell;
  
  ix = floor ((pos->x + hbox_x) / cell_size.x);
  iy = floor ((pos->y + hbox_y) / cell_size.y);
  iz = floor ((pos->z + hbox_z) / cell_size.z);
  
  icell = iz + (cell_nz) * (iy + (cell_ny) * ix);
  
  return icell;
}

void
cell_locate1 (int icell, int *ix, int *iy, int *iz)
{
  int tx, ty;
  int tcell;
  int nyz;
  
  tx = 0;
  ty = 0;
  tcell = icell;
  nyz = cell_ny * cell_nz;
  
  while (tcell >= nyz)
    {
      tx++;
      tcell -= nyz;
    }
  while (tcell >= cell_nz)
    {
      ty++;
      tcell -= cell_nz;
    }
  
  *iz = tcell;
  *iy = ty;
  *ix = tx;
}

/*
  Generate cell list 
  rc : cutoff distance
*/
void
cell_list_new (double rc)
{
  static gboolean need_init = TRUE;
  gint icell;
  gint i;
  
  if (need_init)
   {  
     need_init = FALSE;
  
     cell_nx = ceil (control.box_x/rc);
     cell_ny = ceil (control.box_y/rc);
     cell_nz = ceil (control.box_z/rc);
     cell_size.x = control.box_x / cell_nx;
     cell_size.y = control.box_y / cell_ny;
     cell_size.z = control.box_z / cell_nz;

     hbox_x = 0.5 * control.box_x;
     hbox_y = 0.5 * control.box_y;
     hbox_z = 0.5 * control.box_z;
  
     ncells = cell_nx * cell_ny * cell_nz;
  
     head_of_cell = g_new0 (int, ncells);
     linked_list  = g_new0 (int, natoms);
   }
  
  for (i = 0; i < ncells; i++)
    head_of_cell[i] = -1;
  
  for (i = 0; i < natoms; i++)
    {
      icell = cell_locate (r[i]);
      linked_list[i] = head_of_cell[icell];
      head_of_cell[icell] = i;
    }
  
}

void
cell_neighbor_list_new ()
{
  static gboolean need_init = TRUE;
  int i, j;
  int ix, iy, iz;
  int jx, jy, jz;
  double dx, dy, dz;
  gboolean is_mirror;
  
  if (!need_init)
    return;
  
  need_init = TRUE;

  cell_neigh = g_new0 (CellNeigh, ncells);
  
  for (i = 0; i < ncells-1; i++)
   {
     cell_locate1 (i, &ix, &iy, &iz);
     
     for (j = i+1; j < ncells; j++)
       {
	 gint *i_neigh;
	 is_mirror = FALSE;
	 
	 cell_locate1 (j, &jx, &jy, &jz);
	 
	 dx = ix - jx;
	 
	 if (dx > 0.5 * cell_nx)
	   {
	     dx -= cell_nx;
	     is_mirror = TRUE;
	   }
	 else if (dx < - 0.5 * cell_nx)
	   {
	     dx += cell_nx;
	     is_mirror = TRUE;
	   }
	 
	 if (fabs (dx) > 1.5)
	   continue;
	 
	 dy = iy - jy;
	 
	 if (dy > 0.5 * cell_ny)
	   {
	     dy -= cell_ny;
	     is_mirror = TRUE;
	   }
	 else if (dy < - 0.5 * cell_ny)
	   {
	     dy += cell_ny;
	     is_mirror = TRUE;
	   }
	 
	 if (fabs (dy) > 1.5)
	   continue;
	 
	 dz = iz - jz;
	 
	 if (dz > 0.5 * cell_nz)
	   {
	     dz -= cell_nz;
	     is_mirror = TRUE;
	   }
	 else if (dz < - 0.5 * cell_nz)
	   {
	     dz += cell_nz;
	     is_mirror = TRUE;
	   }
	 
	 if (fabs (dz) > 1.5)
	   continue;

	 i_neigh = g_new0 (int, 1);

	 if (is_mirror)
	   {
	     *i_neigh = i;
	     cell_neigh[j].n_list = g_list_prepend (cell_neigh[j].n_list, i_neigh);
	   }
	 else
	   {
	     *i_neigh = j;
	     cell_neigh[i].n_list =
	       g_list_prepend (cell_neigh[i].n_list, i_neigh);
	   }
       }
   }
}

 
void
verlet_list_new ()
{
  static gboolean need_init = TRUE;
  /* [FIXME] */
  gdouble rv = 5.0;
  gint i, j;
  gint icel, jcel;
  vector3 dr;
  double dr2;
  
  if (need_init)
    {
      need_init = FALSE;
      verlet_list = g_new0 (VerletList, natoms);
    }
      
  cell_list_new (rv);
  cell_neighbor_list_new ();

  cell_print ();
  
  for (i = 0; i < natoms; i++)
    {
      verlet_list[i].n = 0;
    }
  
  for (icel = 0; icel < ncells; icel++)
    {
      GList *list;

      i = head_of_cell [icel];
      
      while (i >= 0)
	{
	  for (list = cell_neigh[icel].n_list; list; list = list->next)
	    {
	      jcel = *((int*) list->data);
	      j = head_of_cell[jcel];
	      
	      while (j >= 0)
		{
		  
		  if (check_is_bond (i, j))
		    {
		      j = linked_list[j];
		      continue;
		    }
		  
		  if (i != j)
		    {
		      dr.x = r[i]->x - r[j]->x;
		      dr.y = r[i]->y - r[j]->y;
		      dr.z = r[i]->z - r[j]->z;
		      
		      if (dr.x > hbox_x)
			dr.x -= control.box_x;
		      else if (dr.x < -hbox_x)
			dr.x += control.box_x;
		      
		      if (dr.y > hbox_y)
			dr.y -= control.box_y;
		      else if (dr.y < -hbox_y)
			dr.y += control.box_y;
		      
		      if (dr.z > hbox_z)
			dr.z -= control.box_z;
		      else if (dr.z < -hbox_z)
			dr.z += control.box_z;
		      
		      dr2 = dr.x * dr.x + dr.y * dr.y + dr.z * dr.z;
		      
		      if (dr2 < rv*rv)
			{
			  if (verlet_list[i].n < MAX_VERLET)
			    {
			      verlet_list[i].list[verlet_list[i].n] = j;
			      verlet_list[i].n++;
			    }
			  else
			    {
			      verlet_list[j].list[verlet_list[j].n] = i;
			      verlet_list[j].n++;
			      
			      if (verlet_list[j].n >= MAX_VERLET)
				{
				  g_warning ("verlet_list_new : the number of "
					     "verlet_list element at (%d) is %d "
					     "over than MAX_VERLET %d\n",
					     j, verlet_list[j].n, MAX_VERLET);
				  exit (1);
				}
			    }
			  
			}
		    }
		  j = linked_list[j];
		}
	    }
	  i = linked_list[i];
	}
      
    }
  
}

static gboolean
check_is_bond (int ib, int jb)
{
  int i;
  int ia1, ia2;
  
  if (ib < jb)
    {
      ia1 = ib;
      ia2 = jb;
    }
  else
    {
      ia2 = ib;
      ia1 = jb;
    }
    
  for (i = 0; i < nrat; i++)
    {
      if (rat[i].ia1 == ia1 && rat[i].ia2 == ia2)
	return TRUE;
    }
  
  return FALSE;
}


void
cell_print ()
{
  int i, icel;
  int j, jcel;
  int sum_n;
  GList *list;
  FILE *fd;
  
  fd = fopen ("cell_test", "wt");
  
  for (icel = 0; icel < ncells; icel++)
    {
      fprintf (fd, "\n icel = %d ----\n", icel);
      
      i = head_of_cell[icel];
      
      while (i >= 0)
	{
	  fprintf (fd, "%5d", i);
	  i = linked_list[i];
	}
    }

  sum_n = 0;
  
  for (icel = 0; icel < ncells; icel++)
    {
      gint n;
      
      fprintf (fd, "\n icel = %d", icel);
      n = g_list_length (cell_neigh[icel].n_list);
      fprintf (fd, "  neighbor = %d ----\n", n);
      sum_n += n;
      
      for (list = cell_neigh[icel].n_list; list; list = list->next)
	{
	  i = *((int*) list->data);
	  
	  fprintf (fd, "%5d", i);
	}
    }
  
  fprintf (fd, "Sum = %d\n", sum_n);
  
  fclose (fd);
  
}

