#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "dnd-utils.h"
#include "md.h"

#define SFORM  "%127[^#]" /* Format for scanf to read strings safely */

typedef struct _match MD_Match;

struct _match 
{
  char *key;
  char *format;
  char *_default;
  gpointer ptr;
};

const MD_Match match [] = {
  {"title",            SFORM,  "Test Simulation",(gpointer)&control.title},
  {"nsteps",           "%ld",  "1000",          (gpointer)&control.nsteps},
  {"step",             "%lf",  "0.005",        (gpointer)&control.step},
  {"rtf-fname",        "%s",  "",             (gpointer)&control.rtf_fname},
  {"prm-fname",        "%s",  "",             (gpointer)&control.prm_fname},
  {"temperature",      "%lf",  "0.0",          (gpointer)&control.temp},
  {"box-x",            "%lf",  "0.0",          (gpointer)&control.box_x},
  {"box-y",            "%lf",  "0.0",          (gpointer)&control.box_y},
  {"box-z",            "%lf",  "0.0",          (gpointer)&control.box_z},
  {0,0,0,0}
}; 		/* Null termination essential.	*/

static void
default_control ()
{
  const MD_Match *match_p;
  char tmp[64];
  
  for( match_p = match; match_p->key; match_p++)
     {
       strncpy(tmp,match_p->_default, sizeof tmp);
       (void)sscanf(tmp, match_p->format, match_p->ptr);
     }
}

static int
assign (char *strval, char *fmt, gpointer ptr)
{
   int len = strlen(fmt);
   int code = fmt[MAX(0,len-1)];
   if( len > 2 && fmt[len-2] == 'l' ) code = toupper(code);

   switch(code)
   {
    case 's':
    case ']':
      return sscanf(strval, fmt, (char*)ptr);
    case 'd':
      return sscanf(strval, fmt, (int*)ptr);
    case 'D':
      return sscanf(strval, fmt, (long*)ptr);
    case 'f':
      return sscanf(strval, fmt, (float*)ptr);
    case 'F':
      return sscanf(strval, fmt, (double*)ptr);
    default:
      g_message ( "Scanf code \"%s\" not catered for", fmt);
   }
   return -1;		/* This statement is never reached		*/
}

static gboolean
print_control ()
{
  gboolean ok = TRUE;
  
  g_print ("Control.title : %s \n", control.title);
  g_print ("Control.nstep : %d \n", control.nsteps);
  g_print ("Control.step :  %f \n", control.step);
  g_print ("Control.rtf_fname : %s \n", control.rtf_fname);
  g_print ("Control.prm_fname : %s \n", control.prm_fname);
  g_print ("Control.temperature : %f \n", control.temp);
  g_print ("Control.box_x : %f \n", control.box_x);
  g_print ("Control.box_y : %f \n", control.box_y);
  g_print ("Control.box_z : %f \n", control.box_z);
  
  if (control.box_x < 1.0e-2 ||
      control.box_y < 1.0e-2 ||
      control.box_z < 1.0e-2 )
    {
      g_warning ("Box Size is not initiated");
      ok = FALSE;
    }
  
  return ok;
  
}

gboolean
md_read_control (char *fname)
{
  FILE *fd;
  char buffer[200];
  char name[120];
  char value[120];
  gint n_items;
  const MD_Match *match_p;
  gboolean ok = TRUE;
  
  default_control ();
  
  fd = fopen (fname, "rt");
  
  if (fd == NULL)
    {
      g_message ("md_read_control : Fail to open file %s ", fname);
      return FALSE;
    }
  
  
  while (dnd_read_line (fd, buffer))
    {
      n_items = sscanf (buffer, "%[^=] %127[^#]", name, value);
      
      if (str_eq_min (name, "end"))
	break;

      if (n_items < 1)
	g_message (" read_control : Error line : \n %s \n", buffer);
      else
	{
	  for (match_p = match; match_p->key; match_p++)
	    {
	      g_strdown (name);
	      if (str_eq_min (name, match_p->key))
		break;
	    }
	  
	  if (! match_p->key)
	    g_message (" read_control : Error line : \n %s \n", buffer);
	  else
	    {
	      if (n_items == 1)
		{
		  if (str_eq (match_p->format, SFORM))
		    /* name=<empty> - assign null */
		    *(char*)match_p->ptr = '\0';
		  else
		    g_message (" read_control : Error : %s \n", name);
		}
	      else
		{
		  n_items = assign (value+1, match_p->format, match_p->ptr);
		  
		  if (n_items < 1)
		    g_message (" read_control : Error : %s \n", name);
		}
	    }
	}
    }

  ok = print_control ();
  
  return ok;

  
}



#ifdef _MD___
void
md_read_control (char *fname)
{
  char buffer[120];
  FILE *infile;
  GSList *slist;
  char *buf;
  char word[100];
  double rcutsq_3;
  
  infile = fopen (fname, "rt");

  if (infile == NULL)
    {
      g_print ("Fail to open file %s \n", fname);
      exit (1);
    }
  
  while (dnd_read_line (infile, buffer))
    {
      if (dnd_blank_string (buffer))
	continue;
      if (buffer[0] == '!' || buffer[0] == '#')
	continue;

      buf = dnd_next_word (buffer, word);
      
      if (str_eq_min (word, "BOX"))
	{
	  sscanf (buf, "%lf", &box);
	  box_2 = 0.5 * box;
	}
      else if (str_eq_min (word, "NPART"))
	sscanf (buf, "%d", &npart);
      else if (str_eq_min (word, "NSTEP"))
	sscanf (buf, "%d", &nstep);
      else if (str_eq_min (word, "TEMP"))
	sscanf (buf, "%lf", &temp);
      else if (str_eq_min (word, "TSTEP"))
	sscanf (buf, "%lf", &tstep);
      else if (str_eq_min (word, "NINIT"))
	sscanf (buf, "%d", &ninit);
    }

  r_cutsq = (0.4999 * box);
  r_cutsq *= r_cutsq;

  rcutsq_3 = 1./(r_cutsq*r_cutsq*r_cutsq);
  
  e_cut = 4.0 * rcutsq_3 * (rcutsq_3  - 1);
  
  tstep2 = tstep * tstep;
  
  fclose (infile);
  
  fr = g_new0 (vector3, npart);
  vel = g_new0 (vector3, npart);
  r = g_new0 (vector3, npart);

  g_print (" Number of Particles : %d \n", npart);
  g_print (" Boxlength           : %f \n", box);
  g_print (" density             : %f \n", (npart)/(box*box*box));
  g_print (" Cut-Off Radius      : %f \n", sqrt (r_cutsq));
  g_print (" Cut-Off Energy      : %f \n", e_cut);
  g_print (" Number of Steps     : %d \n", nstep);
  g_print (" Number of Init Steps: %d \n", ninit);
  g_print (" TimeStep            : %f \n", tstep);
  
}

void
md_read_data (char *fname)
{
  char buffer[120];
  FILE *infile;
  GSList *slist;
  char *buf;
  
  g_return_if_fail (fname != NULL);
  
  infile = fopen (fname, "rt");
  
  g_return_if_fail (infile != NULL);
  
  slist = NULL;
  
  while (dnd_read_line (infile, buffer))
    {
      if (dnd_blank_string (buffer))
	continue;
      if (buffer[0] == '!' || buffer[0] == '#')
	continue;
      
      buf = g_strdup (buffer);
      
      if (slist == NULL)
	{
	  slist = g_slist_append (slist, buf);
	  command_slist = slist;
	}
      else 
	{
	  slist = g_slist_append (slist, buf);
	  slist = slist->next;
	}
    }

  fclose (infile);
  
}

#endif
