#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <math.h>
#include "dnd-utils.h"
#include "rtf.h"
#include "param.h"
#include "defs.h"

#define VSWITCH 1

static GList *param_nonbond_list = NULL;

static void param_add_bond (char *buf);
static void param_add_angle (char *buf);
static void param_add_dihedral (char *buf);
static void param_add_improper (char *buf);
static void param_add_nonbond (char *buf);
static void param_read_hbond (char *buf);
static void param_add_hbond (char *buf);
static void read_nonbond_options (FILE *);

#define X_TYPE 1000

struct _param_bond
{
  RTF_AtomType *rtf_at1, *rtf_at2;
  double force_constant;   /* Force constant for bond */
  double distance;        /* Rest distance for bond */
};


struct _param_angle
{
  RTF_AtomType *rtf_at1, *rtf_at2, *rtf_at3;
  double force_constant;   /* Force constant */
  double angle;           /* Theta 0 */
  double k_ub;            /* Urey-Bradley force constant */
  double r_ub;            /*  Urey-Bradley distance */
};


struct _param_dihedral
{
  RTF_AtomType *rtf_at1, *rtf_at2, *rtf_at3, *rtf_at4;
  double force_constant;
  int   periodicity;
  double phi_max;
};


struct _param_improper
{
  RTF_AtomType *rtf_at1, *rtf_at2, *rtf_at3, *rtf_at4;
  double force_constant;
  int   periodicity;
  double phi_min;
};


struct _param_nonbond /* vdw_param */
{
  RTF_AtomType *rtf_at;
  double eps;
  double sigma;
  double eps14;
  double sigma14;
};


struct _param_hbond
{
  int hbexpn[4];   /* ?? */
  double dcuthb;
  gboolean bcuthb;
  double dctonhb;
  gboolean bctonhb;
  double dcutha;
  gboolean bcutha;
};



gboolean
read_param (char *fname)
{
  char buffer[200];
  char word[120]; 
  char *buf;
  FILE *parm_file;
  
  g_return_val_if_fail (fname != NULL, FALSE);
  
  parm_file = fopen (fname, "rt");

  g_return_val_if_fail (parm_file != NULL, FALSE);
        
  while (dnd_read_line (parm_file, buffer))
    {
      buf = dnd_next_word (buffer, word);
      
      if (!buf)
	continue;
      
      if (word[0] == '!' || word[0] == '*')
	continue;
      
      if (str_eq_min (word, "REMARK"))
	continue;
      
      if (str_eq_min(word, "BOND"))
	param_add_bond (buf);
      else if (str_eq_min (word, "ANGL"))
	param_add_angle (buf);
      else if (str_eq_min (word, "DIHE"))
	{
	  /* add_dihedral_param (); */
	}
      else if (str_eq_min (word, "IMPR"))
	param_add_improper (buf);
      else if (str_eq_min (word, "NBOND"))
	{
	  read_nonbond_options(parm_file);
	}
      else if (str_eq_min (word, "NONB"))
	param_add_nonbond (buf);
      else if (str_eq_min (word, "HBOND"))
	param_add_hbond (buf);
           
    }

  fclose (parm_file);

  return TRUE;
  
}


gdouble
get_bond_value (int at1, int at2)
{
  gint i;
  gdouble ret_val = 0.0;
  ParamBond *bond;
  GList *list;
  
  for (list = param_bond_list; list; list = list->next)
    {
      bond = list->data;
      
      if ((bond->rtf_at1->vdw_type == at1 &&
	   bond->rtf_at2->vdw_type == at2) ||
	  (bond->rtf_at1->vdw_type == at2 &&
	   bond->rtf_at2->vdw_type == at1))
	{
	  ret_val = bond->distance;
	  break;
	}
      
    }

  if (ret_val < 0.01)
    {
      g_message("error in Param::get_bond_value ");
    }

  return ret_val;
}



gdouble
get_angle_value (RTF_AtomType *rtf_at1,
		 RTF_AtomType *rtf_at2,
		 RTF_AtomType *rtf_at3)
{
  int n_ang = g_list_length (param_angle_list);
  gint i;
  gdouble ret_val = 0.0;
  ParamAngle *angle;
  
  for (i = 0; i < n_ang; i++)
    {
      angle = g_list_nth (param_angle_list, i)->data;
      
      if (angle->rtf_at1 == rtf_at1 &&
	  angle->rtf_at2 == rtf_at2 &&
	  angle->rtf_at3 == rtf_at3 )
	{
	  ret_val = angle->angle;
	  break;
	}
    }

  if (ret_val < 0.01)
    {
      g_message ( "error in Param::get_bond_value ");
    }
  

  return ret_val;
}


static void
param_add_bond (char *buf)
{
  char  at1[6], at2[6];
  gint  cmp;
  ParamBond *bond_p;
  double force_constant, distance;
  RTF_AtomType *rtf_at1, *rtf_at2;
  
  sscanf (buf, " %s %s %lf %lf", 
	  at1, at2, &force_constant, &distance);
  

  rtf_at1 = g_hash_table_lookup (atom_type_hash, at1);
  rtf_at2 = g_hash_table_lookup (atom_type_hash, at2);
  
  bond_p = g_new0 (ParamBond, 1);
  
  bond_p->force_constant = force_constant;
  bond_p->distance = distance;

  cmp = strcasecmp (at1, at2);

  if (cmp < 0)
    {
      bond_p->rtf_at1 = rtf_at1;
      bond_p->rtf_at2 = rtf_at2;
    }
  else 
    {
      bond_p->rtf_at1 = rtf_at2;
      bond_p->rtf_at2 = rtf_at1;
    }

  param_bond_list = g_list_prepend (param_bond_list, bond_p);
}


static void
param_add_angle (char *buf)
{
  char  at1[6], at2[6], at3[6];
  ParamAngle *angle_p;
  double force_constant, angle;
  gint   cmp;
  RTF_AtomType *rtf_at1, *rtf_at2, *rtf_at3;

  sscanf (buf, " %s %s %s %lf %lf",
	  at1, at2, at3, &force_constant, &angle);
  
  angle_p = g_new0 (ParamAngle, 1);

  angle_p->k_ub = 0.0;
  angle_p->r_ub = 0.0;
    
  rtf_at1 = g_hash_table_lookup (atom_type_hash, at1);
  rtf_at2 = g_hash_table_lookup (atom_type_hash, at2);
  rtf_at3 = g_hash_table_lookup (atom_type_hash, at3);
  
  cmp = strcasecmp (at1, at3);
  
  if (cmp < 0)
    {
      angle_p->rtf_at1 = rtf_at1;
      angle_p->rtf_at2 = rtf_at2;
      angle_p->rtf_at3 = rtf_at3;
    }
  else
    {
      angle_p->rtf_at1 = rtf_at3;
      angle_p->rtf_at2 = rtf_at2;
      angle_p->rtf_at3 = rtf_at1;
    }
  
  param_angle_list = g_list_prepend (param_angle_list, angle_p);

}

static void
param_add_dihedral (char *buf)
{
  char at1[6], at2[6], at3[6], at4[6];
  ParamDihedral *dihedral_p;
  gint periodicity;
  double force_constant, phi_max;
  
  sscanf (buf, " %s %s %s %s %lf %d %lf",
	  at1, at2, at3, at4,
	  &force_constant,
	  &periodicity,
	  &phi_max);

  dihedral_p = g_new0 (ParamDihedral, 1);
  
  dihedral_p->rtf_at1 = g_hash_table_lookup (atom_type_hash, at1);
  dihedral_p->rtf_at2 = g_hash_table_lookup (atom_type_hash, at2);
  dihedral_p->rtf_at3 = g_hash_table_lookup (atom_type_hash, at3);
  dihedral_p->rtf_at4 = g_hash_table_lookup (atom_type_hash, at4);

  dihedral_p->force_constant = force_constant;
  dihedral_p->periodicity = periodicity;
  dihedral_p->phi_max = phi_max;
    
  param_dihedral_list = g_list_append (param_dihedral_list, dihedral_p);

}

static void
param_add_improper (char *buf)
{
  char at1[6], at2[6], at3[6], at4[6];
  ParamImproper *improper_p;
  gint periodicity;
  double force_constant, phi_min;
  
  sscanf (buf, "%s %s %s %s %lf %d %lf",
	  at1, at2, at3, at4,
	  &force_constant,
	  &periodicity,
	  &phi_min);
  
  improper_p = g_new0 (ParamImproper, 1);
  
  improper_p->rtf_at1 = g_hash_table_lookup (atom_type_hash, at1);
  improper_p->rtf_at2 = g_hash_table_lookup (atom_type_hash, at2);
  improper_p->rtf_at3 = g_hash_table_lookup (atom_type_hash, at3);
  improper_p->rtf_at4 = g_hash_table_lookup (atom_type_hash, at4);
  
  improper_p->force_constant = force_constant;
  improper_p->periodicity = periodicity;
  improper_p->phi_min = phi_min;
    
  param_improper_list = g_list_prepend (param_improper_list, improper_p);

}


static void
param_add_nonbond (char *buf)
{
  ParamNonBond *nonb_p;
  char at1[6];
  double eps, eps14;
  double sigma, sigma14;
  RTF_AtomType *rtf_at;
  
  sscanf (buf, " %s %lf %lf %lf %lf",
	  at1, &eps, &sigma, &eps14, &sigma14);
  
  nonb_p = g_new0 (ParamNonBond, 1);
  
  nonb_p->rtf_at = g_hash_table_lookup (atom_type_hash, at1);
  nonb_p->eps = eps;
  nonb_p->sigma  = sigma;
  nonb_p->eps14  = eps14;
  nonb_p->sigma14 = sigma14;

  param_nonbond_list = g_list_prepend (param_nonbond_list, nonb_p);
  
}


void
setup_vdw_table ( )
{
  gint n_vdw = g_list_length (param_nonbond_list);
  GList *li, *lj;
  ParamNonBond *nonb_i, *nonb_j;
  gint i, j;
  double eps, rij;
  static gboolean need_init = TRUE;
  
  if (!need_init)
    return;
  
  need_init = FALSE;
  A = g_new0 (double *, n_vdw);
  B = g_new0 (double *, n_vdw);
  
  for (i = 0; i < n_vdw; i++)
    {
      A[i] = g_new0 (double, n_vdw);
      B[i] = g_new0 (double, n_vdw);
    }
  
  g_print ("_kcal/EUNIT = %f \n", _kcal/(EUNIT*AVOGAD));
  
  for (li = param_nonbond_list; li; li = li->next)
    {
      nonb_i = (ParamNonBond *) li->data;
      i = nonb_i->rtf_at->vdw_type;
      
      for (lj = param_nonbond_list; lj; lj = lj->next)
	{
	  nonb_j = (ParamNonBond *) lj->data;
	  j = nonb_j->rtf_at->vdw_type;
	  
	  eps = sqrt (nonb_i->eps * nonb_j->eps);
	  eps *= (_kcal/(EUNIT*AVOGAD)); /* Convert Kcal/mole into amu (A/ps)^2 */
	  
	  rij = 0.5 * (nonb_i->sigma + nonb_j->sigma);
	  rij = rij * rij * rij;
	  rij *= rij;
	  B[i][j] = eps * rij;
	  rij *= rij;
	  A[i][j] = eps * rij;
	}
    }
  
}


static void
read_nonbond_options (FILE *parm_file)
{
  gboolean ok_exit = FALSE;
  char buffer[120];
  char *buf;
  char word[20];
  
  while (dnd_read_line (parm_file, buffer))
    {
      buf = dnd_next_word (buffer, word);
      
      while (TRUE)
	{
	  g_strup (word);
	  
	  if (str_eq_min (word, "END"))
	    {
	      ok_exit = TRUE;
	      break;
	    }
	  
	  if (str_eq_min (word, "CUTNB"))
	    {
	      int i = dnd_index (buffer, "CUTNB")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      
	      sscanf (buffer+i+j, "%lf", &param_nbond.cut_off_nb);
	    }
	  else if (str_eq_min (word, "TOLER"))
	    {
	      int i = dnd_index (buffer, "TOLER")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.tolerance);
	    }
	  else if (str_eq_min (word, "ATOM"))
	    param_nbond.do_atom = TRUE;
	  else if (str_eq_min (word, "GROUP"))
	    param_nbond.do_atom = FALSE;
	  else if (str_eq_min (word, "WMIN"))
	    {
	      int i = dnd_index (buffer, "WMIN")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.warn_min);
	    }
	  else if (str_eq_min (word, "INHIBIT"))
	    {
	      int i = dnd_index (buffer, "INHIBIT")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.inhibit);
	    }
	  else if (str_eq_min (word, "EPS"))
	    {
	      int i = dnd_index (buffer, "EPS")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.elec_eps);
	    }
	  else if (str_eq_min (word, "E14FAC"))
	    {
	      int i = dnd_index (buffer, "E14FAC")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.elec_e14fac);
	    }
	  else if (str_eq_min (word, "CDIELEC"))
	    {
	    }
	  else if (str_eq_min (word, "SHIFT"))
	    {
	    }
	  
	  else if (str_eq_min (word, "VSWIT"))
	    param_nbond.vdw_option = VSWITCH;
	  else if (str_eq_min (word, "CTONNB"))
	    {
	      int i = dnd_index (buffer, "CTONNB")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.switch_cut_on_nb);
	    }
	  else if (str_eq_min (word, "CTOFNB"))
	    {
	      int i = dnd_index (buffer, "CTOFNB")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%lf",
		      &param_nbond.switch_cut_off_nb);
	    }
	  else if (str_eq_min (word, "NBXMOD"))
	    {
	      int i = dnd_index (buffer, "NBXMOD")+1;
	      int j = dnd_index (buffer+i, "=")+1;
	      sscanf (buffer+i+j, "%d",
		      &param_nbond.exclusion_option);
	    }
	  else
	    {
	      g_message ("NBOND : MISSING NONBOND OPTION (%s) ", word);
	    }
	  
	  buf = dnd_next_word (buf, word);
	  
	  if (buf == NULL)
	    break;
	}
      
      if (ok_exit)
	break;
    }
}

static void	  
param_add_hbond (char *buf)
{
  
  
}
