#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "dnd-utils.h"
#include "rtf.h"

static char buffer[120];
static char *buf;

static void rtf_add_mass ();
static void rtf_add_residue (RTF_Residue *res, gboolean do_patch);
static void rtf_add_atom_residue (RTF_Residue *res, gint rtf_mode,
				  gint num_group);
static void rtf_add_bond_residue (RTF_Residue *res, gint rtf_mode);
static void rtf_add_angle_residue (RTF_Residue *res, gint rtf_mode);
static void rtf_add_dihedral_residue (RTF_Residue *res, gint rtf_mode);
static void rtf_add_donor_residue (RTF_Residue *res, gint rtf_mode);
static void rtf_add_ic_residue (RTF_Residue *res, gint rtf_mode);

static guint database_hash_func (gconstpointer key);
static gint  database_compare_func (gconstpointer a, gconstpointer b);


gboolean
read_rtf (char *fname)
{
  char word[120];
  FILE *rtf_file;
  RTF_Residue *res;
  gboolean debug = TRUE;
  gboolean do_patch = FALSE;
  gint     rtf_mode;
  gint     num_group;
  
  g_return_val_if_fail (fname != NULL, FALSE);
  
  rtf_file = fopen (fname, "rt");
  
  if (rtf_file == NULL)
    {
      g_message ("read_rtf : Fail to open file %s ", fname);
      return FALSE;
    }
  
  do_patch = FALSE;
  rtf_mode = RTF_NONE;
  
  while (dnd_read_line (rtf_file, buffer))
    {
      buf = dnd_next_word (buffer, word);
            
      if (!buf)
	continue;
      	
      if (word[0] == '!' || word[0] == '*')
	continue;
      
      if (str_eq_min (word, "REMARK"))
	continue;

      if (do_patch)
	{
	  if (str_eq_min (word, "DELE"))
	    {
	      rtf_mode = RTF_DELETE;
	      buf = dnd_next_word (buf, word);
	    }
	  else if (str_eq_min (word, "MODI"))
	    {
	      rtf_mode = RTF_MODIFY;
	      buf = dnd_next_word (buf, word);
	    }
	  else if (str_eq_min (word, "ADD"))
	    {
	      rtf_mode = RTF_ADD;
	      buf = dnd_next_word (buf, word);
	    }
	  else
	    rtf_mode = RTF_NONE;
	}
      else
	rtf_mode = RTF_NONE;
      
      if (str_eq_min (word, "MASS"))
	{
	  rtf_add_mass ();
	}
      else if (str_eq_min (word, "DECL"))
	{
	}
      else if (str_eq_min (word, "RESI"))
	{
	  do_patch = FALSE;
	  res = g_new0 (RTF_Residue, 1);
	  rtf_add_residue (res, do_patch);
	  num_group = 0;
	}
      else if (str_eq_min (word, "PRES"))
	{
	  do_patch = TRUE;
	  res = g_new0 (RTF_Residue, 1);
	  rtf_add_residue (res, do_patch);
	  num_group = 0;
	}
      else if (str_eq_min (word, "ATOM"))
	{
	  rtf_add_atom_residue (res, rtf_mode, num_group);
	}
      else if (str_eq_min (word, "BOND") || str_eq_min (word, "DOUB"))
	{
	  rtf_add_bond_residue (res, rtf_mode);
	}
      else if (str_eq_min (word, "ANGL") || str_eq_min (word, "THET"))
	{
	  rtf_add_angle_residue (res, rtf_mode);
	}
      else if (str_eq_min (word, "DIHE") || str_eq_min (word, "PHI"))
	{
	  rtf_add_dihedral_residue (res, rtf_mode);
	}
      else if (str_eq_min (word, "DONO"))
	{
	  rtf_add_donor_residue (res, rtf_mode);
	}
      else if (str_eq_min (word, "IC") || str_eq_min (word, "BUIL") ||
	       str_eq_min (word, "BILD"))
	{
	  rtf_add_ic_residue (res, rtf_mode);
	}
      else if (str_eq_min (word, "GROU"))
	num_group++;
    }

  /* Test */
  if (debug)
    {
      gint nres = g_hash_table_size (residue_hash);
      /*      gint npres = g_hash_table_size (presidue_hash); */
      gint npres = 0;
      gint natom = g_hash_table_size (atom_type_hash);
      
      g_print ("nres = %d ... npres = %d ... natom = %d \n",
	       nres, npres, natom);
    }
    
  fclose (rtf_file);

  return TRUE;
  
}

	  
static void
rtf_add_mass ()
{
  RTF_AtomType *atomp = g_new0 (RTF_AtomType, 1);
  char at_type[5];
  double mass;
  static int vdw_type = 0;
  
  if (!atom_type_hash)
    atom_type_hash = g_hash_table_new (database_hash_func,
				       database_compare_func);
  
  sscanf (buf, "%s %lf ", at_type, &mass);
  
  atomp->name = g_strdup (at_type);
  
  atomp->mass = mass;

  atomp->vdw_type = vdw_type;
  vdw_type++;
  
  g_hash_table_insert (atom_type_hash, atomp->name, atomp);
}


static void
rtf_add_residue (RTF_Residue *res, gboolean do_patch)
{
  char name[10];
 
 sscanf (buf, " %s ", name);
 res->name = g_strdup (name);

  if (!do_patch)
    {
      if (!residue_hash)
	residue_hash = g_hash_table_new (database_hash_func,
				     database_compare_func);

      g_hash_table_insert (residue_hash, res->name, res);
    }
  else
    {
      if (!presidue_hash)
	presidue_hash = g_hash_table_new (database_hash_func,
					  database_compare_func);
      g_hash_table_insert (presidue_hash, res->name, res);
    }
  
}


static void
rtf_add_atom_residue (RTF_Residue *res, gint rtf_mode, gint num_group)
{
  RTF_Atom *atomp = g_new0 (RTF_Atom, 1);
  char word[20];
  
  g_return_if_fail (res != NULL);
  
  buf = dnd_next_word (buf, word);
  
  atomp->name = g_strdup (word);
  
  atomp->mode = rtf_mode;
  atomp->num_group = num_group;
  
  if (rtf_mode == RTF_DELETE)
    {
      atomp->charge = 0.0;
      atomp->atom_type = NULL;
    }
  else
    {
      char type[10];
      int  i;
      
      i = dnd_index (buf, "TYPE")+1;
      
      if (i != 0)
	{
	  int j;
	  j = dnd_index (buf+i, "=")+1;
	  dnd_next_word (buf+i+j, type);
	  atomp->atom_type = g_hash_table_lookup (atom_type_hash, type);

	  if (atomp->atom_type == NULL)
	    {
	      g_error ("There is no Atom Type \"%s\" in Residue Topology File\n",
		       type);
	      exit (1);
	    }
	  
      	}
      
      i = dnd_index (buf, "CHAR")+1;
      
      if (i != 0)
	{
	  int j;
	  j = dnd_index (buf+i, "=")+1;
	  sscanf (buf+i+j, "%lf", &atomp->charge);
	}
      else
	{
	  g_error ("Error in RTF : %s\n", buffer);
	}
    }
  
  res->atom_list = g_list_append (res->atom_list, atomp);
  
}

static void
rtf_add_bond_residue (RTF_Residue *res, gint rtf_mode)
{
  RTF_Bond *bondp;
  char at_name1[10], at_name2[10];
  
  sscanf (buf, "%s %s", at_name1, at_name2);
        
  bondp = g_new0 (RTF_Bond, 1);
      
  if (strcmp (at_name1, at_name2) <= 0)
    {
      bondp->at_name[0] = g_strdup (at_name1);
      bondp->at_name[1] = g_strdup (at_name2);
    }
  else
    {
      bondp->at_name[1] = g_strdup (at_name1);
      bondp->at_name[0] = g_strdup (at_name2);
    }

  bondp->mode = rtf_mode;
    
  res->bond_list = g_list_append (res->bond_list, bondp);
  
}


static void
rtf_add_angle_residue (RTF_Residue *res, gint rtf_mode)
{
  RTF_Angle *anglep;
  char at_name1[10], at_name2[10], at_name3[10];
  
  sscanf (buf, "%s %s %s ", at_name1, at_name2, at_name3);
  
  anglep = g_new0 (RTF_Angle, 1);
  
  if (strcmp (at_name1, at_name3) <= 0)
    {
      anglep->at_name[0] = g_strdup (at_name1);
      anglep->at_name[1] = g_strdup (at_name2);
      anglep->at_name[2] = g_strdup (at_name3);
    }
  else
    {
      anglep->at_name[2] = g_strdup (at_name1);
      anglep->at_name[1] = g_strdup (at_name2);
      anglep->at_name[0] = g_strdup (at_name3);
    }

  anglep->mode = rtf_mode;
    
  res->angle_list = g_list_append (res->angle_list, anglep);
  
}

static void
rtf_add_dihedral_residue (RTF_Residue *res, gint rtf_mode)
{
  

  
}


static void
rtf_add_donor_residue (RTF_Residue *res, gint rtf_mode)
{
  

  
}



static void
rtf_add_ic_residue (RTF_Residue *res, gint rtf_mode)
{
  RTF_ic *icp = g_new0 (RTF_ic, 1);
  char at_name[4][20];
  
  icp->mode = rtf_mode;
  
  sscanf (buf, "%s %s %s %s %lf %lf %lf %lf %lf",
	  at_name[0], at_name[1], at_name[2], at_name[3],
	  &icp->bond[0],
	  &icp->angle[0],
	  &icp->phi,
	  &icp->angle[1],
	  &icp->bond[2]);


  icp->at_name[0] = g_strdup (at_name[0]);
  icp->at_name[1] = g_strdup (at_name[1]);
  icp->at_name[3] = g_strdup (at_name[3]);
    
  if (at_name[2][0] == '*')
    {
      icp->do_chiral = TRUE;
      icp->at_name[2] = g_strdup (at_name[2]+1);
    }
  else
    {
      icp->do_chiral = FALSE;
      icp->at_name[2] = g_strdup (at_name[2]);
    }

  res->ic_list = g_list_append (res->ic_list, icp);
}


static guint
database_hash_func (gconstpointer key)
{
  gchar *string;
  guint result;
  int c;

  string = (gchar *) key;
  result = 0;
  while (1)
    {
      c = *string;
      string++;
      if (c == 0)
	break;
      result += (result << 3) + c;
    }

  return result;
}

static gint
database_compare_func (gconstpointer a,
		       gconstpointer b)
{
  return (strcmp ((char *) a, (char *) b) == 0);
}

gint
get_vdw_type (char *atom_type)
{
  RTF_AtomType *rtf_at = g_hash_table_lookup (atom_type_hash, atom_type);
  
  return rtf_at->vdw_type;
  
}
