function [fctable, blocked] = FFCA(solver, o_network, varargin)
% FFCA version 1.01
% Perform Flux Coupling Analysis with the Feasibility-based approach.
%
% USAGE:
% 1. fctable = FFCA(solver, network)
%
%       Compute the flux coupling for every pair of reactions in the network.
%
%       PARAMETERS:
%       solver: LP solver to us - possible values: 'linprog', 'clp', 'lindo', 'glpk'
%       network: metabolic network structure with fileds
%           - stoichiometricMatrix
%           - reversibilityVector
%           - Reaction,
%           - Metabolites
%
%       OUTPUT:
%       fctable: in all cases it contains the coupling results.
%       Interpretation for element (i, j):
%           1 - fully coupled
%           2 - partially coupled
%           3 - reaction i is directionally coupled to j
%           4 - reaction j is directionally coupled to i
%           5 - uncoupled
%
% 2. fctable = FFCA(solver, network, r1)
%
%       Compute the flux coupling of r1 with every other reaction.
%
% 3. fctable = FFCA(solver, network, r1, r2)
%       
%       Compute the flux coupling between r1 and r2

t1 = cputime;

[network, blocked] = correctReversibilities(solver, o_network);

[met rea] = size(o_network.stoichiometricMatrix);

disp(' ');
disp(sprintf('Number of metabolites: %d', met));
disp(sprintf('Number of reactions: %d', rea));
disp(sprintf('Number of blocked reactions: %d', nnz(blocked)));


[Irev,Prev,Frev]=reaction_classificationFB(network);

disp(' ');
disp(sprintf('Number of Irev: %d', nnz(Irev)));
disp(sprintf('Number of Prev: %d', nnz(Prev)));
disp(sprintf('Number of Frev: %d', nnz(Frev)));

disp(' ');
disp(sprintf('Preprocessing runtime: %f', cputime-t1));

disp(' ');
disp(sprintf('#LP in preprocessing: %d', rea+nnz(o_network.reversibilityVector)));
disp(sprintf('#LP in main: %d', nnz(Irev)*(nnz(Irev)-1)+2*nnz(Prev)*nnz(Irev)+nnz(Prev)*(nnz(Prev)-1)/2+nnz(Frev)*(nnz(Frev)-1)/2 ));

if isempty (varargin) 
    s=size(network.stoichiometricMatrix);
    fctable=zeros(s(2),s(2));
    
    % compute the coupling types of all reactions-pairs, but only for the
    % upper triangular matrix to minimize effort

    for i=1:s(2)
        %percent_completed=round((i/s(2))*100);
        %display(percent_completed)
        for j=(i+1):s(2)
            fctable(i,j)=pairwiseCouplingFFCA(solver,network, i, j, Irev, Prev, Frev);
            if (fctable(i, j)==8)
                fctable(i,j)=pairwiseCouplingBwoS(solver,network, i, j);
            end
        end
    end

    % set lower triangular matrix (same values as in the upper one)
    for m=1:s(2)
        for n=(m+1):s(2)
            if (fctable(m,n)~=3 && fctable(m,n)~=4)
            fctable(n,m)=fctable(m,n);
            elseif (fctable(m,n)==3)
                fctable(n,m)=4;
            else
                fctable(n,m)=3;
            end
        end
    end
    
    % set the values on the diagonal to 1 (reactions are always fully coupled to themselves)
    for l = 1:s(2)
        fctable(l,l)=1;
    end

else
    if numel (varargin) == 1
        if find(blocked == varargin{1})
            disp(' ');
            disp('The specified reactions is blocked.');
            disp('No coupling computed');
            return
        else
            r1 = varargin{1} - nnz(blocked<varargin{1});
            s=size(network.stoichiometricMatrix);
            fctable=zeros(1,s(2));
            for i=1:s(2)
                    fctable(1, i)=pairwiseCouplingFFCA(solver,network, r1, i, Irev, Prev, Frev);
                    if (fctable(1, i)==8)
                        fctable(1,i)=pairwiseCouplingBwoS(solver,network, r1, i);
                    end
            end
        end
    else
        % compute the coupling type of reactions ri and rj
        if find(blocked == varargin{1})|find(blocked == varargin{2})
            disp(' ');
            disp('One of the specified reactions is blocked.');
            disp('No coupling computed');
            return
        else     
            r1 = varargin{1} - nnz(blocked<varargin{1});
            r2 = varargin{2} - nnz(blocked<varargin{2});
            fctable = pairwiseCouplingBwoS(solver,network, r1, r2);
        end
    end
    
end

 t2 = cputime;
 
 disp(' ');
 disp('Network statistics:');
 disp(sprintf('# directionally coupled: %d', nnz(fctable==3)));
 disp(sprintf('# partially coupled: %d', nnz(fctable==2)));
 disp(sprintf('# fully coupled: %d', nnz(fctable==1)));
 disp(sprintf('# uncoupled coupled: %d', nnz(fctable==5)));
 disp(' ');
 disp(sprintf('Total runtime: %f', t2-t1));
 