# Perl module to draw chromatogram file
# Malay <curiouser@gene.ccmbindia.org>
# Copyright (c) 2002 Malay Kumar Basu
# You may distribute this module under the same terms as perl itself
# Thanks to David H. Klatte for all the hard work!

=head1 NAME

BioSVG::Chromatogram, the object to draw the chromatogram file

=head1 SYNOPSIS

  my $chromatogram = BioSVG::Chromatogram->create_from_file(
                                -file=>"myfile.abi",
                                -type=>"ABI");
  $chromatogram->set_height(250); # Height of the SVG image
  $chromatogram->set_hscale(5); # Horizontal scale 
  $chromatogram->set_start(150); #Show it from sequence 150
  $chromatogram->set_stop(250); # Show from sequence 250
  $chromatogram->show(); # Dumps the SVG on the screen
  my $svg = $chromatogram->to_svg(); # Return the whole SVG

=head1 DESCRIPTION

The chromatogram created by Applied Biosystem automated DNA sequencing machine
is a binary file. For details of the file structure see David Klattes wonderful
technical information. This module requires four arrays @A, @B, @C, @T for four
types of traces. Array @basecalls containing the basecalls. Array @sequence for
the sequence. Scalar $height for the image height and $h_scale for horizontal
scale of the image. $h_scale determines how sharp will be the peaks.

By default the font size is 11. Font face is courier. You can change it by calling
set_font() method of the "Pastel::Graphics" object which is the parent class
of BioSVG::Chromatogram. The same way the width of the line can be changed by
changing the stroke by calling set_stroke(). _ 

=head1 FEEDBACK

Malay<curiouser@gene.ccmbindia.org>

=cut

package BioSVG::Chromatogram;
@ISA = qw(Pastel::Graphics);
use Pastel;
use BioSVG::Parser::ABI;

use Carp;
use strict;

=head1 CONSTRUCTORS

=head2 create_from file()

  Function : Creates and returns BioSVG::Chromatogram object. 
  Usage    : my $chr->BioSVG::Chromatogram->create_from_file( 
                                                  -file=>"fullpathname",
						  -type=>"ABI",
                                                  -width=>$width,
                                                  -height=>$height,
                                                  -hscale=>$scale,
						  -panel=>"multi/single"
                                                   );
  Args     : -file=> full path name of the file
             -type=> "ABI" or "SCF". Only "ABI" file is suppoted now
             -hscale=> Horizontal scale. Default 2.
             -width=>  Width of the SVG image. Used only when "panel" is set 
                       to multi. Otherwise the image width is calculated as-
                       hscale * tracelength.
             -height=> Height of the SVG image. Default 200
             -panel=>  "multi" or "single". Deafault "single". The number of panel
                       in the SVG image. If "multi" the image is drawn in multiple
                       panels. The number of panel will depend on the horizontal 
                       scale and number of traces. If "single", then the whole
                       chromatogram is drawn in a single panel.
 Returns  : The reference of the chromatogram object created.  

=cut

sub create_from_file {
  my ($class, @args) = @_;
  $class = ref($class)||$class;
  my $self = $class->SUPER::new(@args);
  bless $self, $class;
  $self->_init(@args);
 
    #print "****", $self->{_mac_header}, "\n";
    return $self;
}

sub _init {
 
  my ($self, @args) = @_;
     $self->SUPER::_init(@args);
     my ($file,$type,$height, $hscale,$panel) = $self->_rearrange(["FILE","TYPE","HEIGHT","HSCALE","PANEL"],@args );
     
     if(!defined($file)){
       croak "Filename not supplied in BioSVG::Chromatogram::_init()\n";     }
     if(!defined($height)){
       $height = 200;
     }
     if(!defined($hscale)){
       $hscale = 3.0;
     }
     if(!defined($panel)){
       $panel = "single";
     }
     
  if(!defined($type)){
      croak "Type not defined in BioSVG::Chromatogram::new()!\n";
    }
  if($type =~ /ABI/i){   
     $self->{_trace} = BioSVG::Parser::ABI->new($file);
  }  
  if($type =~ /SCF/i){
     croak "SCF files are not supported yet!\n";
}   
$self->{_hscale} = $hscale;
     $self->{height} = $height;
     $self->{_num_panel} = $panel;

     my $trace_length = $self->{_trace}->get_trace_length();
     $self->{_trace_length} = $trace_length;
     # Set the width of the parent
    if($self->{_num_panel} =~ /single/ ){
     $self->set_width($hscale * $trace_length);
   }
     
     # Get all the necessary data from the ABI object
     $self->{_max_trace} = $self->{_trace}->get_max_trace();
     $self->{A} = [$self->{_trace}->get_trace("A")];
     $self->{G} = [$self->{_trace}->get_trace("G")];
     $self->{C} = [$self->{_trace}->get_trace("C")];
     $self->{T} = [$self->{_trace}->get_trace("T")];
     $self->{seq} = $self->{_trace}->get_sequence();
     $self->{basecalls} = [ $self->{_trace}->get_base_calls()];
     $self->{sample_name} = $self->{_trace}->get_sample_name();
     
     $self->_reverse_scale(); # Reverses the trace in the y coordinate and scale it
    $self->{color_a} = Pastel::Color->green()->darker();
    $self->{color_g} = Pastel::Color->black();
    $self->{color_black} = $self->{color_g};
    $self->{color_c}= Pastel::Color->blue();
    $self->{color_t} = Pastel::Color->red();
    $self->{color_n} = Pastel::Color->pink();
    $self->set_stroke(Pastel::BasicStroke->new(-width=>1));
    
    
     # We don't require ABI object anymore
     delete($self->{_trace});
     
     #$self->{g} = Pastel::Graphics->new(-width=>$trace_length * $hscale, -height=>$height);
     #$self->{g}->show();
}

#=head2 create_from_file()

# Usage   : my $chromatogram = BioSVG::Chromatogram->create_from_file(
#							 -file=>"fullpathname",
#							 -type=>"ABI",
#                                                         -width=>$width,
#							 -height=>$height,
#                                                         -hscale=>$scale,
#							 -panel=>"multi/single"
#                                                         );
# Function: Creates a chromatogram object from a file specified.
# Args    : -file=> full path name of the file
#           -type=> "ABI" or "SCF". Only "ABI" file is suppoted now
#           -width=>  Width of the SVG image. Used only when "panel" is set 
#                     to multi. Otherwise the image width is calculated as-
#                     hscale * tracelength.
#           -hscale=> Horizontal scale. Default 2.
#           -height=> Height of the SVG image. Default 200
#           -panel=> "multi" or "single". Deafault "single". The number of panel
#                     in the SVG image. If "multi" the image is drawn in multiple
#                     panels. The number of panel will depend on the horizontal 
#                     scale and number of traces. If "single", then the whole
#                     chromatogram is drawn in a single panel.
# Returns : The reference of the chromatogram object created.  

#=cut

#sub create_from_file {
#  my $self = shift;
#  my %args = @_;
#  if(defined($args{"-file"}){
#  my $class = $self->new(@_);
  
  
#} else {
#  croak "Filename not supplied in BioSVG::create_from_file()!\n";
#}
#}

=head1 METHODS

=head2 draw()

Usage   : $chr->draw();
Function: Draws the chromatogram in the buffer. There is no output. To get the
          chromatogram in SVG format, you need to call get_svg() method on the
          chromatogram object. You can also use show() to dump the output in
          STDOUT. By deafult it will draw the chromatogram


=cut

sub draw {
   my $self = shift;
   #$self->_make_x();
   if($self->{_num_panel}=~ /single/){
     $self->_draw_single_panel();
   } else {
     $self->_draw_multi();
  }
}

sub get_x {
  my $self = shift;
  #my $width = $self->get_width();
  my $trace_length = $self->{_trace_length};
 my @x; 
 for( my $i = 0; $i < $trace_length -1 ; $i++){
    $x[$i] = int ( $self->{_hscale} * $i );
 }
 return @x;
}

sub _draw_single_panel {
  my $self = shift;
  my @x = $self->get_x();
  $self->set_width( $self->{_hscale} * $self->{_trace_length});
  $self->set_paint( $self->{color_a});
  $self->draw_polyline(\@x, $self->{A});
  $self->set_paint($self->{color_g});
  $self->draw_polyline(\@x, $self->{G});
  $self->set_paint($self->{color_c});
  $self->draw_polyline(\@x, $self->{C});
   $self->set_paint($self->{color_t});
  $self->draw_polyline(\@x, $self->{T});

$self->set_font(Pastel::Font->new(-family=>"Courier", -style=>"BOLD",-size=>16));
  $self->set_paint($self->{color_t});
  
  $self->draw_string("Sample name : $self->{sample_name}", 15,15);
  $self->set_font(Pastel::Font->new(-family=>"Courier", -style=>"BOLD",-size=>12));
  
 $self->set_paint($self->{color_black});
$self->draw_string('Generated by BioSVG (c) 2002, Malay <curiouser@gene.ccmbindia.org>',10,$self->get_height() - 15);
   $self->set_font(Pastel::Font->new(-family=>"Courier",-style=>"BOLD",-size=>11)); 
 my @seq = split(//,$self->{seq});
  my @base_calls = @{$self->{basecalls}};
  my $base_call_y = 40;
  my $scale_y = $self->get_height() - 30;
  
  for ( my $i = 0; $i < @base_calls; $i++){
    if ( $i > @seq ) { last;}
    $base_calls[$i] = int ( $base_calls[$i] * $self->{_hscale});

    if( $seq[$i] =~ /A/i ){
      $self->set_paint($self->{color_a});
      $self->draw_string("A",$base_calls[$i]-3,$base_call_y);
    }
    if($seq[$i] =~ /G/i){
      $self->set_paint($self->{color_g});
      $self->draw_string("G",$base_calls[$i]-3,$base_call_y);
    }
    if($seq[$i] =~ /T/i){
      $self->set_paint($self->{color_t});
      $self->draw_string("T",$base_calls[$i]-3,$base_call_y);
    }
    if($seq[$i] =~ /C/i){
      $self->set_paint($self->{color_c});
      $self->draw_string("C",$base_calls[$i]-3,$base_call_y);
    }
    if($seq[$i] =~ /N/i){
      $self->set_paint($self->{color_n});
      $self->draw_string("N",$base_calls[$i]-3,$base_call_y);
    }
    if((($i%10) == 0) && ($i != 0)){
      #print "TIME FOR NUMBER\n";
      #print $i;
      $self->set_paint($self->{color_black});
      $self->draw_string($i,$base_calls[$i - 1]-3,$scale_y);
  }}
  
  
}

sub _reverse_scale {
  my $self = shift;
  my $scale = $self->_get_scale();
  my $height = $self->get_height();
  my @a = @{$self->{A}};
  my @g = @{$self->{G}};
  my @c = @{$self->{C}};
  my @t = @{$self->{T}};
  my $bottom = 40;

  for ( my $i = 0; $i < @a ; $i++){
     $a[$i] = int( $height - $bottom - ( $a[$i] * $scale));
       $g[$i] = int( $height - $bottom - ( $g[$i] * $scale));
       $c[$i] = int( $height - $bottom - ( $c[$i] * $scale));
 $t[$i] = int( $height - $bottom - ( $t[$i] * $scale));
  }

  $self->{A} = [ @a ];
  $self->{G} = [ @g ];
  $self->{C} = [ @c ];
  $self->{T} = [ @t ];

  
}

sub _get_scale {
   my $self = shift;
   my $top = 50;
   my $bottom = 40;
   my $max = $self->{_max_trace};
   my $height = $self->get_height();
   return ( ($height - ($top + $bottom)) / $max);
}
1;

   
