# GenericFeature module for Genquire I

# by David Block <dblock@gene.pbir.nrc.ca>

# Copyright David Block and NRC-PBI

# POD Documentation - main docs before the code

=head1 NAME

GenericFeature - GenericFeature Object for Genquire I

=head1 SYNOPSIS

This is an abstract base class, so it should not be called
directly.

=head1 DESCRIPTION

In order to make a collection of objects sit on a database, this parent class
was created.  It takes care of getting and setting values for any attributes.
It also implements Bio::SeqFeatureI, given child
classes that have the right attributes.  Creating new objects that are children
of this class inserts rows into the database, and changing the objects changes the
values in the database.
Its children in the Genquire I implementation include Feature, Gene,
Feature::Annotation, and Feature::mod.

=head1 FEEDBACK

Like it or lump it, report to dblock@gene.pbi.nrc.ca.  Feel free to add
to the docs available at
http://bioinfo.pbi.nrc.ca/dblock/wiki

=head2 Reporting Bugs

Email the author with any bug reports.

=head1 AUTHOR - David Block

email dblock@gene.pbi.nrc.ca

=cut

package GQ::Server::GenericFeature;
$VERSION = 1.00;
use strict;
use vars qw( @ISA $AUTOLOAD );  #Keep 'use strict' happy
use Carp;
use DBI;
use Bio::SeqFeatureI;
use Bio::Tk::GO_Annotation;
use GQ::Server::Object;
use GQ::Root;
use Bio::Root::Root;
@ISA=qw(GQ::Server::Object Bio::Root::Root Bio::SeqFeatureI);

{
    #Encapsulated class data

                                 #    DEFAULT         ACCESSIBILITY
    GQ::Root->create(
		id          =>  ["",		   'read'],  
		lockid      =>  [undef,        'read'],
		par_pos     =>  [0,            'read/write'],
		tagvalues   =>  [undef,        'read/write'],
		GO          =>  [[],           'read/write'],
		entire_seq  =>  [undef,        'read/write'],
		contig_id   =>  [undef,        'read'],
		source_tag  =>  ['???',        'read/write'],
		primary_tag =>  ['???',        'read/write'],
		contig_start=>  [undef,        'read/write'],
		contig_stop =>  [undef,        'read/write'],
		score       =>  [0,            'read'],
		strand      =>  ['.',          'read'],
		frame       =>  ['.',          'read/write'],
		length      =>  [0,            'read/write'],
		name        =>  ['',           'read/write'],
		type        =>  ['',           'read'],
		last_modified =>[undef,        'read/write'],
		modified_by   =>['',           'read/write'],
		access        =>['rw',         'read'],
		    );


}


sub _db_keys {
    my ($self)=@_;
    my @list;
    foreach ($self->_standard_keys,$self->SUPER::_standard_keys) {
	#print "key $_\n";
	push @list,$_ unless /^context$|^type$|^id$|^lockid$|entire_seq|^tagvalues$|^adaptor$|^GO$|_pos$|modified/;
    }

    push @list;
    return ((sort @list), "id");  # needs to be sorted to be sure that it always pulls off the keys in the same order
}

sub _db_values {
    my ($self)=@_;
    my @list;
    foreach ($self->_db_keys) {
	#print "key $_ value ",$self->{$_},"\n";
	push @list,$self->{$_} unless /^id$/;
    }
    return @list;
}

sub delete_from_db {
    my ($self)=@_;
    return if $self->access eq 'ro';

    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);

    $self->adaptor->delete_from_db($self);
    $self->{entire_seq}=undef;
    return 1;
}

sub add_to_isa {
    my ($self, $type) = @_;
	# we need to concern ourselves only with the
	# BioPerl GeneStructureI family of features
	# GeneStructureI, TranscriptI, ExonI, Promoter, UTR, Poly_A_site
    if ($type =~ /genestructure/i) {
		$type .= 'I';
    }
	if ($type =~ /transcript/i) {
		$type .= 'I';
    }
    if ($type =~ /exon/i) {
		$type .= 'I';
    }

	my @isa;
	if ($self->{isa_list}){@isa = @{$self->{isa_list}}};
	# if we already have a valid BioPerl feature type, then
	# remove it from the list (it should be the last on the list)
	if ($isa[$#isa] =~ /Bio::SeqFeature::Gene/){unshift @{$self->{isa_list}}};
	
	# now push the new feature type on
    push @{$self->{isa_list}}, $type;
}

sub isa {         #overloaded from UNIVERSAL::isa
    my ($self, $type)= @_;
    foreach (@{$self->{isa_list}}) {
	return 1 if $type eq $_;
    }
    return $self->SUPER::isa($type);
}

######Have to implement everything in SeqFeatureI, or we'll get AbstractDeath!!!

sub absstart {
    my $self=shift;
    my $absstart=$self->seq_pos + $self->contig_start;
    return $absstart;
}

sub absstop {
    my $self=shift;
    my $absstop=$self->seq_pos + $self->contig_stop;
    return $absstop;
}

sub seq_pos {
    my ($self)=@_;
	return $self->{seq_pos} if exists $self->{seq_pos};
    my $contigobj=$self->context->contig_by_id($self->contig_id);
	return $self->{seq_pos}=$contigobj->abs_start - 1;
}

sub par_pos {
    my ($self,$pos)=@_;
    if (defined $pos) {
	$self->{par_pos}=$pos;
    }
    return $self->{par_pos};
}

sub start {
	#print "start: ", ($_[0]->absstart - $_[0]->par_pos + 1),"\n";
	return $_[0]->absstart - $_[0]->par_pos + 1
}

sub end {
	#print "end: ", ($_[0]->absstop  - $_[0]->par_pos + 1),"\n";
	return $_[0]->absstop  - $_[0]->par_pos + 1
}

sub gff_string {    #overloaded from SeqFeatureI
    my ($self)=@_;
    my $string=$self->SUPER::gff_string;
    my ($seqname,$source,$pri,$start,$end,@everything_else)=split "\t",$string;
    my $newstring=join "\t",$seqname,$source,$pri,$self->contig_start,$self->contig_stop,@everything_else;
    return $newstring;
}

sub gff_strand {
    my ($self,$value)=@_;
    if ($value) {
	if ($self->_update_db('strand',$value)) {
	    $self->{strand}=$value;
	} else {
	    die "Database update unsuccessful: $DBI::errstr, $!";
	}
    }
    return $self->{strand};
}

sub strand {
    my ($self,$value)=@_;
	if (defined $value) {
	$value =~ s/-1/-/;
	$value =~ s/1/+/;
	$value =~ s/0/./;
	if ($self->_update_db('strand',$value)) {
	    $self->{strand}=$value;
	} else {
	    die "Database update unsuccessful: $DBI::errstr, $!";
	}
    }
    my $strand=$self->{strand};
    $strand =~ tr/+./10/;
    $strand =~ s/-/-1/;
	return $strand;
}

sub sub_SeqFeature {
    my ($self)= @_;
    if ($self && $self->can("features")) {
	return $self->features;
    }
}

sub entire_seq {
    my ($self)=@_;
    return $self->{entire_seq};
}

sub attach_seq {
    my ($self,$seq)=@_;
    if ($seq->isa('Bio::PrimarySeqI')) {
	$self->{entire_seq}=$seq;
    } else {
	$self->throw('attach_seq must be given a Bio::PrimarySeqI object')
    }
}

########### TAG / VALUE IMPLEMENTATION ############

sub get_tag_values {
    my ($self,$data)=@_;
    return if $self->{tagged};
    my $taghash=$self->adaptor->get_tag_values($self,$data);
    $self->{tagvalues}=$taghash;
    $self->{tagged}=1;
}

sub insert_tag_values {
    my ($self)=@_;

    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);
    $self->adaptor->insert_tag_values($self);
}

sub has_tag {
    my ($self,$tag)=@_;
    my $flag;
    foreach ($self->_standard_keys) {
	$flag++ if $_ eq $tag;
    }
    return $flag if $flag;
    return if $tag =~ /gene_name/;         #Genquire-specific hack!  Find a better way someday!!!
    if (not defined $self->tagvalues) {
	$self->get_tag_values;
    }
    foreach (keys %{$self->tagvalues}) {
	$flag++ if $_ eq $tag;
    }
    return $flag;
}

sub each_tag_value {
    my ($self,$tag)=@_;
    if (exists $self->{$tag}) {
	return $self->{$tag};
    } else {
	if (not defined $self->tagvalues) {
	$self->get_tag_values;
	}
	return unless ${$self->tagvalues}{$tag};
	return @{${$self->tagvalues}{$tag}};
    }
}

sub add_tag_value {
    my ($self,$tag,$value)=@_;

    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);

    if ($self->_exists($tag)) {
	my $str='$self->'.$tag.'($value)';
	return eval $str;
    } else {
	if (not defined $self->tagvalues) {
	$self->get_tag_values;
	}
	$self->{tagged}=0;
	if ($self->adaptor->add_tag_value($self,$tag,$value) eq '1') {
	    push @{${$self->tagvalues}{$tag}},$value;
	    return $value;
	} else {
	    return 'Failure';
	}
    }
}

sub remove_tag {
    my ($self,$tag)=@_;

    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);

    if (not defined $self->tagvalues) {
	$self->get_tag_values;
    }
    $self->{tagged}=0;
    if ($self->adaptor->remove_tag($self,$tag)) {
	delete ${$self->tagvalues}{$tag};
	return $tag;
    } else {
	return 'Failure';
    }
}

sub all_tags {
    my ($self)=shift;
    if (not defined $self->tagvalues) {
	    $self->get_tag_values;
    }
    return keys %{$self->tagvalues};
}

#### GO Storage routines - uses Bio::Tk::GO_Annotation module ######

sub GO {
	#don't want to update db- that is done with GO_store!
	# this is simple storage and retrieval in memory
    my ($self,$new)=@_;
    if ($new) {
	push (@{$self->{GO}}, $new);
    }
    return $self->{GO};  # returns a listref of GO_Annotation objects
}


#########Getting GO Annotation objects from the database###########

sub GO_find {
    my ($self)=@_;
    return if $self->{gone};  # this actualy means the opposite of what it seems... it means "already there", not "gone" :-)
    $self->_wipe_GO;  # empties the $self->GO listref
    
    foreach($self->each_tag_value('GO_annotation')) { #a list of GO ids
		my $newGO = GO_Annotation->new(id      => $_,
						   term    => $self->_GO_get_term($_),
						   evidence=> $self->_GO_get_evidence($_),
						   );
		$self->GO($newGO);
    }
    $self->{gone}=1;
}

sub _wipe_GO {
    my ($self)=@_;
    #return if $self->{tagvalues}{GO_annotation};
    $self->{GO} = [];
	@{$self->{tagvalues}{GO_annotation}} = ();
	
    my $list=$self->adaptor->wipe_GO($self);
    foreach my $value(@$list) {
		push @{$self->{tagvalues}{GO_annotation}},$value;
    }
}

sub _GO_get_term {
    my ($self,$id)=@_;
    return $self->adaptor->GO_get_term($id);
}

sub _GO_get_evidence {
    my ($self,$id)=@_;
    return $self->adaptor->GO_get_evidence($self,$id);
}

########Storing created GO Annotations##########

sub GO_store {
    my ($self,$list)=@_;

    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);

    my %already_there;
    foreach($self->each_tag_value('GO_annotation')) {
	$already_there{$_}++;
    }
    foreach my $ann (@$list) {
	unless ($already_there{$ann->id}) {
	    $self->add_tag_value('GO_annotation',$ann->id);
	    $self->adaptor->GO_add_term($ann);
	}
	$self->GO($ann);
	foreach my $code (keys %{$ann->evidence}) {
	    foreach my $ev (@{${$ann->evidence}{$code}}) {
		$self->adaptor->GO_insert_evidence($self->id,$code,$ev,$ann->id) or return;
	    }
	}
    }
    $self->{gone}=0;
    return 1;
}

######## REMOVING GO ANNOTATIONS ################

sub GO_delete {
    my ($self,$GO_id)=@_;

    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);

    $self->adaptor->GO_delete($self,$GO_id) or die "Unable to delete from database: $DBI::errstr";
    $self->{gone}=0;
    $self->GO_find;
}

######## FLAG Implementation ###################################
#
# uses the Flag and Flags tables
#
# Flags holds the text value of the flag ("Dave's protein")
# Flag contains the feature_id, (as in TagValue),
#   and the Flags.id in flag, and the contig
# There will be Flag Query Methods in the Context module
#
#################################################################

sub add_flag {
    my ($self,$flag)=@_;
    my $flagid=$self->context->_check_flag($flag);
    $self->adaptor->insert_flag($self,$flagid);
}

sub remove_flag {
    my ($self,$flag)=@_;
    my $flagid=$self->context->_check_flag($flag);
    $self->adaptor->remove_flag($self,$flagid);
}

######## END OF FLAG Implementation #######################

#Implement other get_... and set_... methods

# (create as necessary)

sub last_modified {
    my ($self)=@_;
    my $lastmod=$self->adaptor->last_modified($self);
    $self->{last_modified}=$lastmod;
}


sub modified_by {
    my ($self,$user)=@_;
    if ($user) {
	$self->{modified_by}=$user;
	$self->adaptor->put_mod_by($self,$user);
    }
    return $self->{modified_by};
}

sub _update_db {
    my ($self,$attr,$newval)=@_;
    return 1 unless $self->adaptor;
    return 1 if $self->access eq 'ro';  #make it look successful
    die "Must Acquire Lock On This Region To Proceed.\n"
      if $self->context->checklock(lockid       => $self->lockid,
				   contig_id    => $self->contig_id,
				   contig_start => $self->contig_start,
				   contig_stop  => $self->contig_stop);

    if ($self->adaptor->update_discard($self)) {
	if ($self->adaptor->update_db($self,$attr,$newval)) {
	    $self->modified_by($self->context->user);
	    $self->last_modified;
	    return 1;
	} else {
	    die "Database update unsuccessful - old data still in discard table\n".
	        "$DBI::errstr;  $!";
	}
    }
}


sub find_duplicate_tags {
    my ($self)=@_;
    return $self->adaptor->find_duplicate_tags($self->id);
}

sub seqname {
    my ($self)=@_;
    my $contigobj=$self->context->contig_by_id($self->contig_id);
    return $contigobj->name;
}


sub getBlastHits {
    my ($self)=@_;
    
    unless ($self->{blasted}) {
	$self->{BlastHits}=$self->adaptor->getBlastHits($self->id);
	$self->{blasted}=1;
    }
    return $self->{BlastHits};
}

sub type {
    my ($self,$type)=@_;
    if (defined $type && $self->access eq 'rw') {

	die "Must Acquire Lock On This Region To Proceed.\n"
	  if $self->context->checklock(lockid       => $self->lockid,
				       contig_id    => $self->contig_id,
				       contig_start => $self->contig_start,
				       contig_stop  => $self->contig_stop);

	$self->adaptor->change_type($self,$type);
	$self->{type}=$type;
    }
	eval {"require Bio::SeqFeature::Gene::$type;"};
	unless ($@) {
		$self->add_to_isa("Bio::SeqFeature::Gene::$type");
	}
    return $self->{type};
}

sub parent {
    my ($self)=@_;
    my $parent_id=$self->adaptor->find_parent_id($self);
	return 0 unless $self->entire_seq;
    return $self->entire_seq->feature($parent_id);
}

#  added by Mark as a test
############################
sub increment_right {
    my ($self,$incr)=@_;
    $self->contig_stop($self->contig_stop + $incr);
    $self->length($self->end - $self->start + 1);
}

sub decrement_right {
    my ($self,$incr)=@_;
    $self->contig_stop($self->contig_stop - $incr);
    $self->length($self->end - $self->start + 1);
}

sub increment_left {
    my ($self,$incr)=@_;
    $self->contig_start($self->contig_start + $incr);
    $self->length($self->end - $self->start + 1);
}

sub decrement_left {
    my ($self,$incr)=@_;
    $self->contig_start($self->contig_start - $incr);
    $self->length($self->end - $self->start + 1);
}

###################################



1;


