package GQ::Server::DB::Adaptor::ContainerAdaptor;
use strict;

use vars qw( $AUTOLOAD @ISA );  #Keep 'use strict' happy

use Carp;
use DBI;
use Storable qw(dclone);
use GQ::Root;
use GQ::Server::DB::Adaptor::GenericFeatureAdaptor;
@ISA = qw (GQ::Server::DB::Adaptor::GenericFeatureAdaptor);


{
#Encapsulated class data
#replace with your column names, and sensible defaults

    GQ::Root->create();
}

### write in any custom subroutines here

=head2 update_discard

 Title   : update_discard
 Usage   : if ($self->adaptor->update_discard($self)) {
 Function: move a copy of the current contents of the persistent version of a feature
           into a discard location so that the feature's history is preserved
 Returns : 1 on success, die with an error message on failure
 Args    : the feature that is about to be updated


=cut

sub update_discard {
    my ($self,$feat)=@_;
    my $table=$self->table;
    my $discard_table=$self->discard_table;
    my $user=$feat->context->user;
    my $cols=join ',',$feat->_db_keys;
    $cols .= ',discarded_whom';
    my $placeholders='?,' x ($feat->_db_keys + 1);
    chop $placeholders;
    eval {
	my $sth=$self->dbh->prepare("INSERT INTO $discard_table ".
				    "($cols) VALUES ($placeholders)")
	  or die "Can't prepare: $DBI::errstr";
	$sth->execute((map {$feat->{$_}} $feat->_db_keys), $user)
	  or die "Can't execute discard: $DBI::errstr";
    };
    if ($@) {
	die "Move to discard unsuccessful - no updating performed\n$@";
    } else {
	return 1;
    }
}

=head2 diff

 Title   : diff
 Usage   : my $difference=$self->adaptor->diff($self,$feature);
 Function: find the distance between the abs_start values of the contigs
           of two features from different contigs
 Returns : a number
 Args    : the two features whose contig abs_start difference you are looking for
 Note    : Could be replaced by
              $self->context->contig_by_id($self->contig_id)->abs_start -
              $feat->context->contig_by_id($feat->contig_id)->abs_start
           to avoid a further database call...

=cut

sub diff {
    my ($self,$feat,$feature)=@_;
    my $sth=$self->dbh->prepare('SELECT feat.abs_start-self.abs_start FROM Tiling_Path self, Tiling_Path feat '.
				'WHERE self.contig_id=? AND feat.contig_id=?')
      or die "Database error: $DBI::errstr";
    $sth->execute($feat->contig_id,$feature->contig_id) or die "Database error: $DBI::errstr";
    my ($diff)=$sth->fetchrow_array;
    return $diff;
}

=head2 add_element

 Title   : add_element
 Usage   : $self->add_element($fea,$new);
 Function: associate an element with a container feature
 Example : a gene has Transcript A.  Annotator finds a new Transcript,
           Transcript B, that is associated with that gene.  B is added to the
           gene in memory, and the adaptor makes it permanent:
           $gene->adaptor->add_element($gene,$TranscriptB);
 Returns : nothing
 Args    : the parent feature, and the element feature


=cut

sub add_element {
    my ($self,$fea,$new)=@_;
    my $sth=$self->dbh->prepare("REPLACE INTO Container (parent,element)".
			        "VALUES (?,?)");
    $sth->execute($fea->id,$new->id);
    $self->context->commit;
}

=head2 remove_element

 Title   : remove_element
 Usage   : $self->remove_element($fea,$old);
 Function: disassociate an element from a container feature
 Example : The above Transcript B was an error, and must be removed
           from the gene:
           $gene->adaptor->remove_element($gene,$TranscriptB);
 Returns : nothing
 Args    : the parent feature, and the element feature


=cut

sub remove_element {
    my ($self,$fea,$old)=@_;
    my $sth=$self->dbh->prepare("DELETE FROM Container ".
			        "WHERE parent=? AND element=?");
    $sth->execute($fea->id,$old->id);
    $self->context->commit;
}

=head2 add_feature

 Title   : add_feature
 Usage   : $self->adaptor->add_feature($self,$new);
 Function: associate a feature with a parent
           This routine simply calls add_element
 Returns : nothing
 Args    : the parent and new element objects


=cut

sub add_feature {
    my ($self,$fea,$new)=@_;
    $self->add_element($fea,$new);
}

=head2 remove_feature

 Title   : remove_feature
 Usage   : $self->adaptor->remove_feature($self,$old);
 Function: disassociate a feature from a parent
           This routine simply calls remove_element
 Returns : nothing
 Args    : the parent and new element objects


=cut

sub remove_feature {
    my ($self,$fea,$old)=@_;
    $self->remove_element($fea,$old);
}


1;
