package GQ::Client::ShowSequenceContext;
use Tk;
use strict;
use Bio::SeqI;
use Bio::PrimarySeq;
require GQ::Server::Feature::mod;
use Carp;
use Tk::widgets qw(Balloon);
use Tk::ROText;
use vars qw(@ISA $AUTOLOAD);
Tk::Widget->Construct('ShowSequenceContext');

@ISA = qw(Tk::MainWindow);

BEGIN {
    {				# encapsulated class data
	my %_attr_data = (
			txtSeqDisp   		=>  [undef,             'read/write'],
			txtTransDisp		=>  [undef,             'read/write'],
			SelectedSeqRegion	=>  [undef,             'read/write'],
			SelectedIDs 		=>  [[],				'read/write'],
			SelectedExons 		=>  [[],            	'read/write'],
			SelectedColors		=>  [[], 				'read/write'],
			span_start 			=>  [undef,				'read/write'],
			span_end			=>  [undef,				'read/write'],
			disp_strand			=>  [undef,					'read/write'], # BioPerl strand nomelclature, NOT GFF!!
			acceptor_site		=>  ["[AC]AGGT", 		'read/write'],
			acceptor_site_length => [5, 				'read/write'],
			donor_site			=>  ["[CT]AG[AGC][AT]", 'read/write'],
			donor_site_length 	=>  [5, 				'read/write'],
			SourceTag			=> 	[undef, 			'read/write'],
			);
	
	
	sub _accessible  {
	    my ($self, $attr, $mode) = @_;
	    return $_attr_data{$attr}[1] =~ /$mode/
	}

	# Classwide default value for a specified object attribute
	sub _default_for {
	    my ($self, $attr) = @_;
	    return $_attr_data{$attr}[0]
	}

	# List of names of all specified object attributes
	# A hash so names are not repeated
	sub _standard_keys { keys %_attr_data }

	use subs (keys %_attr_data)
    }
} #end of BEGIN block

sub AUTOLOAD {
    no strict "refs";
    my ($self, $newval) = @_;

    $AUTOLOAD =~ /.*::(\w+)/;

    my $attr=$1;
    if ($self->_accessible($attr,'write')) {

    	*{$AUTOLOAD} = sub {
    	    if (defined $_[1]) { $_[0]->{$attr} = $_[1] }
    	    return $_[0]->{$attr};
    	};    ### end of created subroutine

    ###  this is called first time only
    	if (defined $newval) {
    	    $self->{$attr} = $newval
    	}
    	return $self->{$attr};

    } elsif ($self->_accessible($attr,'read')) {

    	*{$AUTOLOAD} = sub {
    	    return $_[0]->{$attr} }; ### end of created subroutine
    	return $self->{$attr}
	}
    # Must have been a mistake then...
    croak "No such method: $AUTOLOAD";
}

sub new {
    my ($caller, $QueryScreen, $MapObj, $all_features, $s_start, $s_end, %args)=@_;
    my $caller_is_obj = ref($caller);
    my $class = $caller_is_obj || $caller;
    my $self = $QueryScreen->Toplevel(-title => "Sequence Context");
    bless $self, $class;
    foreach my $attrname ( $self->_standard_keys ) {
    	if ($caller_is_obj) {
    	    $self->{$attrname} = $caller->{$attrname} }
    	elsif ($args{$attrname}){
    		$self->{$attrname} = $args{$attrname}  }
    	else {
    	    $self->{$attrname} = $self->_default_for($attrname) }
    }

	#print "strand ", $self->disp_strand, " selected\n";
	
	if ($self->disp_strand =~ /\-/){$self->disp_strand("-1")}
	elsif ($self->disp_strand =~ /\+/){$self->disp_strand("1")}

	if ($s_start eq "0"){$s_start = 1}
	if ($s_end eq "0"){$s_end = 1}
	
    # the next two lines are specifically for the selection of arbitrary sequence spans (ctrl-mouse-click-n-drag)
    if (defined $s_start){$self->span_start(int($s_start))}    # if the span was already given
    if (defined $s_end){$self->span_end(int($s_end))}          # then take it

	my $uframe = $self->Frame(-background => "darkblue")->pack(-side => 'top', -expand => 0, -fill => 'x');
	my $lframe = $self->Frame->pack(-side => 'bottom', -expand => 1, -fill => 'both');
	$uframe->Label(-foreground => 'white', -background => 'darkblue', -text => "Default source-tag for modifications/additions to this data:")->pack(-side => 'left');
	$self->SourceTag($uframe->Text(-width => 20, -height => 1, -wrap => 'none')->pack(-side => 'left'));
	$self->SourceTag->insert('end', ($MapObj->MapSeq->context->user));

	my $leftframe = $lframe->Frame->pack(-side => 'left', -expand => 1, -fill => 'both');
	my $rightframe = $lframe->Frame->pack(-side => 'right', -expand => 0, -fill => 'y');
	

	my $txtRuler = $leftframe->ROText(
		-background => 'white',
		-height => 2,
		-width => 90,
		-wrap => 'none',
		-foreground => 'black')->pack(
			-side => 'top',
			-expand => 0,
			-fill => 'x');
	
	$txtRuler->insert('end', "1        10        20        30        40        50        60        70        80        90        100       110       120       130       140       150       160       170       180       190       \n");
	$txtRuler->insert('end', "|        |         |         |         |         |         |         |         |         |         |         |         |         |         |         |         |         |         |         |         ");
    $self->txtSeqDisp(
		$leftframe->Scrolled('ROText',
							 -exportselection=>1,
							 -selectbackground => "yellow",
							 -scrollbars => "se",
							 -width => 90,
							 -height => 40,
							 -background => "#002222",
							 -foreground => "#cccccc")->pack(
								 -side => 'top',
								 -expand => 1,
								 -fill => 'both'));
    
    
    $self->txtTransDisp(
		$rightframe->Scrolled('ROText',
							  -exportselection=>1,
							  -selectbackground => "yellow",
							  -scrollbars => "se",
							  -width => 30,
							  -height => 40,
							  -background => "#eeffff")->pack(
								  -side => 'top', -expand => 1, -fill => 'y'));

    my $SeqHelp = $self->Balloon;
    $SeqHelp->attach(($self->txtSeqDisp), -balloonmsg =>
					 "click once to select an exon.".
					 "Key commands are:  \n".
					" > extends exon 1bp 3', \n".
					 "< retracts exon 1bp 3', \n".
					"SHIFT > retracts exon 1bp 5', \n".
					 "SHIFT < extends exon 1bp 5', \n".
					 "A displays consensus acceptor splice sites, \n".
					 "D displays consensus donor splice sites, \n".
					 "S displays start codons. \n\n".
					 "CTRL-SHIFT-A will create a new feature de novo \n".
					 "from selected text.");

    my $result = $self->GetSequenceAndContext($QueryScreen, $MapObj, $all_features);   # also calls the "FillContextDisplay routine after strandedness is determined
    unless ($result){$self->destroy; return 0};  # failed to initiate the object

    unless ($all_features){$self->TranslateSelected($MapObj)};
	$self->SourceTag->bind("<FocusIn>" => sub {$self->txtSeqDisp->tagDelete("selected")});
	
	$self->configure(-title => "Sequence Context from ".$self->span_start." to ".$self->span_end);
    return $self;
}

sub GetSequenceAndContext {

    # must first get the start/stop of each selected exon.  Selected exons have the tag "selected_widget"

    my ($self, $QueryScreen, $MapObj, $all_features)=@_;
    my $SeqObj = $MapObj->MapSeq;

	my (@SelectedSeqs,@SelectedColors,$dir);

    $self->{SelectedIDs} = [];
    $self->{SelectedExons} = [];
    $self->{SelectedColors} = [];


    #   if (Tk::Exists($frmSeqDisp)) {$frmSeqDisp->destroy}

    my %Features;
	
	# the following routine sets the display strand to 1 or -1
	# if there are no features, then it will not be set, but
	# in that case it should have been set already from QueryScreen (a CTRL-select of a region will set the strand)
    my @FIDs = @{$MapObj->getSelectedIDs}; # get the list of exons on the working map that are currently tagged as select
    foreach my $FID (@FIDs) {	# for each FIDnnn that is selected
		my $Feature=$MapObj->AllFeatures($FID); # retrieve the feature object
		unless ($all_features){next unless($Feature->Feature->type =~ /^exon$/i)}
    	$Features{"$FID"}=$Feature; # put it into a hash with the widget ID as the key and the Feature object as the value
    	return 0 if ($self->disp_strand && ($self->disp_strand ne $Feature->strand));  # if features on both strands are selected then abort mission   	
    	$self->disp_strand($Feature->strand); # nab the strand as we go so we can use it later... note that only the first selected exon is considered!
    }
	return 0 unless $self->disp_strand;  # this can happen if the feature is not an exon type and the person is asking for context of only exons
    
    
    my ($min_start, $max_end);
    
    $min_start = $SeqObj->length; # min and max span the entire sequence
    $max_end = 0;

    if ($self->disp_strand =~ /\-/) {		# the exons are sorted left->right or right->left depending on which strand they are on
		foreach (sort { $Features{$b}->end <=> $Features{$a}->end } keys %Features) { # sort the keys (FID's) by the end-values of the associated BioSeqFeatures
			
			if ($Features{$_}->end > $max_end) {
    		$max_end = $Features{$_}->end;
    	    }			# grab the highest and lowest values for the selected region
    	    if ($Features{$_}->start < $min_start) {
    		$min_start = $Features{$_}->start;
    	    }
    	    push @{$self->SelectedIDs}, ($_);
		}			#reverse sorted

    } else {			# THIS IS A DUPLICATE OF THE CODE ABOVE - see for refs
		foreach (sort { $Features{$a}->start <=> $Features{$b}->start } keys %Features) {
    	    if ($Features{$_}->end > $max_end) {
    		$max_end = $Features{$_}->end;
    	    }
    	    if ($Features{$_}->start < $min_start) {
    		$min_start = $Features{$_}->start;
    	    }
    	    push @{$self->SelectedIDs}, ($_);
		}			#forward sorted
    }

    if (!defined $self->span_start || !defined $self->span_end){
        $max_end = $max_end + 50;
        $min_start = $min_start -50;
		
        $min_start = ($min_start < 1)?1:$min_start;
        $max_end = ($max_end > $SeqObj->length)?$SeqObj->length:$max_end;
        $self->span_start($min_start); # set the display range
        $self->span_end($max_end);
	}
	
    $self->FillContextDisplay($QueryScreen, $MapObj, $self->disp_strand);
    return 1;
}

sub FillContextDisplay {
    my ($self, $QueryScreen, $MapObj, $dir) = @_;
    my $SeqObj = $MapObj->MapSeq;

    #print "getting sequence from " . $self->span_start . " to " . $self->span_end . "\n";
    my $SelectedSeqContext;
    if ($dir =~ /\-/){
		
		my $tempseq = Bio::PrimarySeq->new(-seq => ($SeqObj->getseq($self->span_start, $self->span_end)));  # this is the genomic DNA region that we are working with
    	$SelectedSeqContext = Bio::PrimarySeq->new(-seq => ($tempseq->revcom->seq));
    } else {
    	$SelectedSeqContext = Bio::PrimarySeq->new(-seq => ($SeqObj->getseq($self->span_start, $self->span_end)));  # this is the genomic DNA region that we are working with
    }
	$self->txtSeqDisp->insert('end', ($SelectedSeqContext->seq));

    my @FIDs = @{$self->SelectedIDs}; #@{$MapObj->getSelectedIDs};  	# get the *ordered* list of exons on the working map that are currently tagged as select
    #print "Feature ID's in order @FIDs\n\n";
    my $span = $self->span_end - $self->span_start + 1;             # this is needed for the coordinate translation for text representations on the minus strand

    foreach my $FID (@FIDs) {                                          # for each of these
    	my $Feature=$MapObj->AllFeatures($FID);           # get the associated BioSeqFeature object
    	my ($start, $stop);
    	if ($Feature->strand =~ /\-/) {
        	$start = "1." . ($span - ($Feature->end - $self->span_start)- 1);   # something running from 80-90 on a 100 unit map
        	$stop = "1." . ($span - ($Feature->start - $self->span_start));      # will now run from 10 to 20   ---> reverse strand
    	} else {
        	$start = "1." . ($Feature->start - $self->span_start);
        	$stop = "1." . (($Feature->end - $self->span_start) + 1);
    	}
    	my $color = ${$MapObj->current_colors}{$Feature->source_tag};
    	my $dbid = "DB_ID" . $Feature->each_tag_value('id');
    	my $source_tag = "_source_tag_" . $Feature->source_tag;
    	#print $Feature->start . " -- " . $Feature->end . " assigning tags $start  $stop  $color  $fid  $dbid $source_tag  \n";
    	$self->txtSeqDisp->tagAdd($source_tag, $start, $stop);     # add the source_tag tag
    	$self->txtSeqDisp->tagAdd($FID, $start, $stop);        # add the identity of the map widget
    	$self->txtSeqDisp->tagAdd($dbid, $start, $stop);       # add the index number of the database entry for this exon

    	$self->txtSeqDisp->tagConfigure($FID, -foreground => $color);
    	$self->txtSeqDisp->tagBind($FID, "<Button-1>",
				   sub {
					$self->txtSeqDisp->focus;	
					$self->txtSeqDisp->tagDelete("selected");
					my @range = $self->txtSeqDisp->tagRanges("$FID");
					my $nt = $self->txtSeqDisp->get(@range);      # get the nucleotide sequence in that range
					
					$self->txtSeqDisp->tagAdd('selected', @range);
					$self->txtSeqDisp->tagConfigure("selected", -background => "#EEEEEE");
					$MapObj->clearSelections;
					$MapObj->selectFeatures([$FID]);
					$MapObj->recolorWithTag("default", "draft", [$self->{selected}]);
					$MapObj->recolorWithTag("default", "finished", [$self->{selected}]);
					$MapObj->recolorWithTag("#FF0000", "draft", [$FID]);
					$MapObj->recolorWithTag("#FF0000", "finished", [$FID]);
					$self->{selected} = $FID;
					
				    });
    }


    my $accLength = $self->{acceptor_site_length};
    my $donLength = $self->{donor_site_length};

    my $index = "end";
    while (($index = $self->txtSeqDisp->search(-backwards, -regexp, $self->{acceptor_site}, $index, "1.0"))){	 # look for splice acceptor sites
    	$self->txtSeqDisp->tagAdd("AcceptorSite", "$index", "$index + $accLength chars");
    }

    my $index = "end";
    while (($index = $self->txtSeqDisp->search(-backwards, -regexp, $self->{donor_site}, $index, "1.0"))){	 # look for splice acceptor sites
    	$self->txtSeqDisp->tagAdd("DonorSite", "$index", "$index + $donLength chars");
    }

    my $index = "end";
    while (($index = $self->txtSeqDisp->search(-backwards, "ATGG", $index, "1.0"))){	 # look for consensus start
    	$self->txtSeqDisp->tagAdd("StartSite", $index, "$index + 3 chars");
    }

    # removed these because they slow down the sequence display too much!
    #my $index = "end";
    #while (($index = $self->txtSeqDisp->search(-backwards, "TGA", $index, "1.0"))){	 # look for stop codon
    #	$self->txtSeqDisp->tagAdd("StopSite", $index, "$index + 3 chars");
    #}
    #my $index = "end";
    #while (($index = $self->txtSeqDisp->search(-backwards, "TAA", $index, "1.0"))){	 # look for stop codon
    #	$self->txtSeqDisp->tagAdd("StopSite", $index, "$index + 3 chars");
    #}
    #my $index = "end";
    #while (($index = $self->txtSeqDisp->search(-backwards, "TAG", $index, "1.0"))){	 # look for stop codon
    # 	$self->txtSeqDisp->tagAdd("StopSite", $index, "$index + 3 chars");
    #}

    $self->bind("<KeyPress-a>", sub {$self->txtSeqDisp->tagConfigure("AcceptorSite", -background => "#338888"); return});
    $self->bind("<KeyRelease-a>", sub {$self->txtSeqDisp->tagConfigure("AcceptorSite", -background => "#002222"); return});
    $self->bind("<KeyPress-d>", sub {$self->txtSeqDisp->tagConfigure("DonorSite", -background => "#664499"); return;});
    $self->bind("<KeyRelease-d>", sub {$self->txtSeqDisp->tagConfigure("DonorSite", -background => "#002222"); return;});
    $self->bind("<KeyPress-s>", sub {$self->txtSeqDisp->tagConfigure("StartSite", -background => "#009900");
				     #$self->txtSeqDisp->tagConfigure("StopSite", -background => "#990000");
				     return});
    $self->bind("<KeyRelease-s>", sub {$self->txtSeqDisp->tagConfigure("StartSite", -background => "#002222");
				       $self->txtSeqDisp->tagConfigure("StopSite", -background => "#002222");
				       return});

    $self->bind("<KeyPress-C>", sub {$self->txtSeqDisp->tagDelete("selected")});
    $self->bind("<Key-period>", sub {$self->nudge($MapObj, "R_out", $dir)});
    $self->bind("<Key-comma>", sub {$self->nudge($MapObj, "R_in", $dir)});
    $self->bind("<Key-greater>", sub {$self->nudge($MapObj, "L_in", $dir)});
    $self->bind("<Key-less>", sub {$self->nudge($MapObj, "L_out", $dir)});
    $self->bind("<Control-Key-A>", sub {$self->addHandAnnotation($MapObj, $QueryScreen)});

    $self->update;
}

sub addHandAnnotation {
	my ($self, $MapObj, $QueryScreen) = @_;
	my $SeqDisp = $self->txtSeqDisp;
	my @ranges = $SeqDisp->tagRanges("sel");
	print "got here 1 @ranges";
	return if ($#ranges == -1);
	print "got here 2";
			   
	my $feature_type;
	my $AskFeature = $self->DialogBox(-title => 'Type of Feature:',
		   -default_button=> 'OK',
		   -buttons => [qw/OK Cancel/]);
	$AskFeature->add("LabEntry",
		-textvariable=>\$feature_type,
		-label=>'Which feature type are you adding (eg. Promotor):',
		)->pack;
	my $answer=$AskFeature->Show;
	return if $answer eq 'Cancel';
	$feature_type =~ s/\s+/\_/g;  # feature types are not allowed to have spaces
	return if (!$feature_type || $feature_type eq "_");


	my $comment;
	my $AskComments = $self->DialogBox(-title => "Comment about this $feature_type:",
		   -default_button=> 'OK',
		   -buttons => [qw/OK Cancel/]);
	$AskComments->add("LabEntry",
		-textvariable=>\$comment,
		-label=>'please add a free-text comment here',
		)->pack;
	my $answer=$AskComments->Show;
	$comment = "none" if $answer eq 'Cancel';
	$comment =~ s/"/\'/g;  # replace all instances of quotation marks
	
	my $source;
	my $AskComments = $self->DialogBox(-title => "Provide a source-tag for this $feature_type:",
		   -default_button=> 'OK',
		   -buttons => [qw/OK Cancel/]);
	$AskComments->add("LabEntry",
		-textvariable=>\$source,
		-label=>"Please provide a 'source' for this annotation",
		)->pack;
	my $defaultsource = $self->SourceTag->get('1.0', 'end');
	chomp $defaultsource;
	$source = $defaultsource;
	my $answer=$AskComments->Show;
	return if $answer eq 'Cancel';
	$source =~ s/"/\'/g;  # replace all instances of quotation marks
	unless ($source){$source = "hand_annotation"}
	while (@ranges){      # ranges of selected tags.  this will give the offsets relative to the self->span_end/self->span_start
	my $start = shift @ranges;
	$start =~ /\d\.(\d+)/;
	$start = $1;
	my $stop = shift @ranges;
	$stop =~ /\d\.(\d+)/;
	$stop = $1;

	my $abs_start;
	if ($self->disp_strand =~ /\-/){
		$abs_start = ($self->span_end) - $stop + 1;     # have to make it relative to the forward strand (in GFF style)
	} else {
		$abs_start = ($self->span_start) + $start;		# there is no +1 here because of the way Tk reports selection spans... inclusive.
	}
	my ($ctg_id, $ctg_start) = $MapObj->MapSeq->where_am_i($abs_start);
	my $parent_pos = $MapObj->MapSeq->bc_start + $MapObj->MapSeq->start - 1;
	my $length = $stop - $start;    # is this correct?

	#print "creating new feature with strand ", $self->disp_strand, "\n";
	# I don't know why Feature::mod expects strand to be +/- instead of 1/-1
	# but it does... so we have to convert back to this designation here
	# probably because feature::mod can be constructed from GFF strings...??
	my $strd;
	if ($self->disp_strand eq "-1"){$strd = "-"}
	else {$strd = "+"}
	my $Feature = GQ::Server::Feature::mod->new(
		context => $QueryScreen->context,
		lockid => $MapObj->MapSeq->lockid,
		contig_id=>$ctg_id,
		source_tag=>$source,
		primary_tag => $feature_type,
		contig_start=> $ctg_start,
		length=> $length,
		par_pos=>$parent_pos,
		strand => $strd,
		frame => ".",
		type => "Misc_Feature",
		);
	
	$Feature->add_tag_value("comment", $comment);

	my $NewWidgetID;
	$NewWidgetID = shift @{$MapObj->mapFeatures('draft',[$Feature])};
	my $DB_ID = $Feature->id;
	$start = "1.$start";
	$stop = "1.$stop";   # convert the coordinates to the text widget string coordinates
	$SeqDisp->tagAdd($NewWidgetID, $start, $stop);							
	$SeqDisp->tagAdd($DB_ID, $start, $stop);

	my $color = ${$MapObj->current_colors}{$source};
	$SeqDisp->tagConfigure($NewWidgetID, -foreground => $color);
	$SeqDisp->tagBind($NewWidgetID, "<Button-1>",
				sub {$SeqDisp->tagDelete("selected");
					my @range = $SeqDisp->tagRanges("$NewWidgetID");
					$SeqDisp->tagAdd('selected', $start, $stop);
					$SeqDisp->tagConfigure("selected", -background => "#EEEEEE");
				});
				
	$SeqDisp->tagDelete("sel");
	$SeqDisp->tagAdd('selected', $start, $stop);
	$SeqDisp->tagConfigure("selected", -background => "#EEEEEE");
	$MapObj->DraftCanvas->addtag('now_current', 'withtag' => $NewWidgetID);     # Newly created widget.
	$MapObj->_selectFeature($MapObj->DraftCanvas,$MapObj->DraftMap,'sngl');    # naughty naughty!  But it draws the selection box as well as adding selected tag...
	}  # end of foreach $start, $stop @ranges.
}
sub TranslateSelected {
    my ($self, $MapObj) = @_;
    my @FIDs = @{$self->SelectedIDs};  	# get the *ordered* list of exons on the working map that are currently tagged as select
    my $SeqObj = $MapObj->MapSeq;
    my @ORF = ();

    my %seqs;
    #$self->txtTransDisp->configure(-state => "normal");
    #print "\nFeatures to be translated (in order) @FIDs\n";
	my $seq;
    foreach my $FID (@FIDs) {                                          # for each of these
		my $Feature=$MapObj->AllFeatures($FID);           # get the associated BioSeqFeature object

		my $start = $Feature->start;                                # get start and stop
		my $stop = $Feature->end;
		#print "\n$FID start $start stop $stop\n";
		if ($self->disp_strand =~ /-/){
			$seq .= scalar reverse ($SeqObj->getseq($start,$stop));  # this depends on orientation, to put the new seq before or after the current
			$seqs{$FID} =scalar reverse ($SeqObj->getseq($start,$stop)) ;
		} else {
			$seq .=$SeqObj->getseq($start,$stop);  # extract the associated sequence and append it
			$seqs{$FID} = $SeqObj->getseq($start,$stop);
		}
		
    }
	
	if ($self->disp_strand =~ /-/){
		$seq =~ tr/ACTG/TGAC/
	}
	
	my $SelectedSeq=Bio::PrimarySeq->new( -seq => $seq); # make it a seq object
    $self->txtTransDisp->delete('1.0', 'end');                                   # clear the text window

    my $SelectedTrans = $SelectedSeq->translate;       						# returns a ::Seq (protein) object
    $self->txtTransDisp->insert('end', ($SelectedTrans->seq));                  # write the output to the text window

    my $index="end";
    while (($index = $self->txtTransDisp->search(-backwards, "*", $index, "1.0"))){     # find all stop codons and tag them for highlighting
    	$self->txtTransDisp->tagAdd("StopCodon", $index);                            # this is kind of a bizarre way of doing it, but it works
    }
    $self->txtTransDisp->tagConfigure("StopCodon", -background => "red");     # light up all stop codons in red
    #$self->txtTransDisp->configure(-state => "disabled");
    $self->update;
}


sub _getSelectedWidgetID {
    # the following mess of calls is simply to get the ID number of the currently selected exon
    # (Text-tag processing in Tk is frustratingly limited compared to canvases!)
    #
    my ($text) = @_;
    my @range = $text->tagRanges("selected");   # this returns the start and end indexes for any text with the "selected" tag
    return unless (@range);                      # if there is nothing, then bugger off
    my $index = shift @range;                    # get the start index from the range (all you need is any old index for the next step)
    my @tags = $text->tagNames($index);          # use this index to retrieve the tags associated with the piece of text at that index position
    my ($WidgetID, $DB_ID);                                 # this will include the ExonID (EIDnnn) number for the associated map widget
    foreach my $tag(@tags){
		#print "tag $tag\n";
    	if ($tag =~ /(FID\d+)/) {$WidgetID = $1}
    	if ($tag =~ /(DB_ID\d+)/) {$DB_ID = $1}
    }
	print "\n\n";
    if (!$WidgetID){print "\nGenquire reports: widget not found??\n"; return}
    return ($WidgetID, $DB_ID);
}

sub _cloneFeatureObject {

    my ($WidgetID, $MapObj, $self) = @_;
    my $SCFeature = $MapObj->AllFeatures($WidgetID);
    my $Feature = $SCFeature->Feature;  #DB - AllFeatures returns a SeqCanvasFeature, not a GQ::Server::Feature!
    my ($NewFeature,$NewWidgetID);
	my %clone = %$Feature;
	delete $clone{id};  # need to start with a "pure" new feature
	
    if ($Feature->access eq "rw"){
			$NewFeature = $Feature;    # if it is already a hand-annotation then just hand it back to the calling routine
			$NewWidgetID = $WidgetID;
    } else {
		my $source = $self->SourceTag->get('1.0', 'end');
		$source = (($source =~ /(.*)\n*/) && $1);  # get rid of any carriage returns.
		$source =~ s/\W/_/g;
		#$NewFeature = GQ::Server::Feature::mod->new(%clone);		# otherwise make a duplicate of it 
		$clone{'access'} = 'rw';  # make it read/write
		$NewFeature = GQ::Server::Feature->new(%clone);		# otherwise make a duplicate of it 
		# added this to ensure that my source tag is the one that gets mapped.
		$NewFeature->source_tag($source);

		$MapObj->DraftCanvas->dtag($WidgetID, 'selected');			# since the original object is no longer the one we are manipulating
		$MapObj->FinishedCanvas->dtag($WidgetID, 'selected');		# remove the 'selected' tag from that object and add it to the
		$MapObj->DraftCanvas->delete("sel_box_$WidgetID");
		$MapObj->FinishedCanvas->delete("sel_box_$WidgetID");
		
		$NewWidgetID = $MapObj->mapFeatures('draft', [$NewFeature]);	# this returns a reference to an array of one element
		$NewWidgetID = shift @{$NewWidgetID};							# get rid of this reference and get the actual value
		
		$MapObj->DraftCanvas->addtag('now_current', 'withtag' => $NewWidgetID);     # Newly created widget.
		$MapObj->_selectFeature($MapObj->DraftCanvas,$MapObj->DraftMap,'multi');    # naughty naughty!  But it draws the selection box as well as adding selected tag...
		# now we need to replace this entry in the list of selected widgets which is encapsulated in this ShowSequenceContext object
		my $i = 0;
		foreach my $widget(@{$self->SelectedIDs}) {
			if ($widget eq $WidgetID){splice @{$self->SelectedIDs}, $i, 1, $NewWidgetID}
			++$i;
		}
    }
    return ($NewFeature, $NewWidgetID);

}

sub nudge {

    my ($self, $MapObj, $NudgeType, $dir) = @_;
    my $text = $self->txtSeqDisp;
    my ($OldWidgetID, $OldDB_ID) = _getSelectedWidgetID($text);          # I don't know yet if DB_ID is going to be useful here...??
    return unless ($OldWidgetID);
    my ($Feature, $NewWidgetID) = _cloneFeatureObject($OldWidgetID, $MapObj, $self);	# if this object is already a hand-annotation this simply returns the object handle
    						                                        # if it is not currently a hand-annotation then it clones the feature into a new Feature object
    if ($Feature->access ne "rw"){warn "\ncan't modify a read-only database entry\nAccess:".($Feature->access)."\n"; return}  # last-minute double check in case something went awry

    my @oldrange = $text->tagRanges("selected");
    my $oldindex = $oldrange[0];                                        # keep track of one of the old positions to retrieve the tags from this position later
	$text->tagRemove($OldWidgetID, @oldrange);							# exchange the old widget ID for the new one
	my $source = $Feature->source_tag;
    if ($NudgeType eq "R_out"){
    	$text->tagAdd($NewWidgetID, $oldrange[0], ($oldrange[1] . "+ 1 chars"));  # assogn the new WidgetID over the new range
		if ($self->disp_strand =~/-/) {
			$Feature->decrement_left(1);		              # GFF features always start/stop in a left to right orientation
		} else {                                              # therefore on the - strand the "start" is at the 3' end of the gene
			$Feature->increment_right(1);                     # and to extend the feature "right" you have to subtract 1 (move it left)
		}
	} elsif ($NudgeType eq "L_out"){
    	$text->tagAdd($NewWidgetID, ($oldrange[0] . "- 1 chars"), $oldrange[1]);  # assogn the new WidgetID over the new range
	if ($self->disp_strand =~/-/) {
			$Feature->increment_right(1);
		} else {
			$Feature->decrement_left(1);
		}

	} elsif ($NudgeType eq "R_in"){
    	$text->tagAdd($NewWidgetID, $oldrange[0], ($oldrange[1] . "- 1 chars"));  # assogn the new WidgetID over the new range
		if ($self->disp_strand =~/-/) {
			$Feature->increment_left(1);
		} else {
			$Feature->decrement_right(1);
		}

	}elsif ($NudgeType eq "L_in"){
    	$text->tagAdd($NewWidgetID, ($oldrange[0] . "+ 1 chars"), $oldrange[1]);  # assogn the new WidgetID over the new range
		if ($self->disp_strand =~/-/) {
			$Feature->decrement_right(1);
		} else {
			$Feature->increment_left(1);
		}
	
	} else {$text->tagAdd($OldWidgetID, @oldrange); return}      # no type was specified so return it to how it was and return
	
	$text->tagDelete("selected");                               # delete the "selected" tag
	my @range = $text->tagRanges("$NewWidgetID");               # and then get the new Widget range
	$text->tagAdd('selected', @range);                          # and write the "selected" tag over that range
	$text->tagConfigure("selected", -background => "#EEEEEE");# then color the background
	                                        #  NOW... we have to copy over all the widget tags to the new range
	my @tags = $text->tagNames($oldindex);  # pick up all the tags from the old index position
    foreach my $tag(@tags){                 # go through them one at a time
    	next if ($tag eq "DonorSite");
    	next if ($tag eq "AcceptorSite");  # ignore donor and acceptor sites
    	next if ($tag eq "StartSite");		# ignore start sites
    	
    	if ($tag =~ /_source_tag_/) {       # "source_tag" is a special case as this changes as soon as an exon is edited
    		$text->tagRemove($tag, @oldrange);
    		$text->tagAdd("_source_tag_$source", @range);# change it to reflect the fact that it is now a hand_annotation
    		next
    	}	
    	$text->tagRemove($tag, @oldrange);									  # remove all of those old tags over the old range
    	$text->tagAdd($tag, @range);                                          # and re-apply them to the new range
    }
    my $color = ${$MapObj->current_colors}{$Feature->source_tag};
    $text->tagConfigure($NewWidgetID, -foreground => $color);
	$self->txtSeqDisp->tagBind($NewWidgetID, "<Button-1>",
								sub {
									$self->txtSeqDisp->focus;	
									$self->txtSeqDisp->tagDelete("selected");
									my @range = $self->txtSeqDisp->tagRanges("$NewWidgetID");
									$self->txtSeqDisp->tagAdd('selected', @range);
									$self->txtSeqDisp->tagConfigure("selected", -background => "#EEEEEE");
								});

    $self->TranslateSelected($MapObj);

}


sub createHandAnnotatedSegment {
    my ($self, $MapObj) = @_;

	my @selectedRange = $self->txtSeqDisp->tagRanges("sel");
	
	

}

1;



