# Feature module for Workbench II

# by David Block <dblock@gene.pbir.nrc.ca>

# Copyright David Block and NRC-PBI

# POD Documentation - main docs before the code

=head1 NAME

Feature - Feature Object for Workbench II

=head1 SYNOPSIS

=head2 Loading Features from the database

Loading features from the database from chromosome $chr_id
where the feature overlaps the region between $start and $stop:

use Feature;

<need new code here>


$hash_of_features is a reference to a hash of features.
$par_pos is the chromosomal position of the start of the sequence
region you are interested in (often the same as $start).  It is used to
define the relative location of the feature within that sequence.

The keys of $hash_of_features will be the ids of the features, and the
values will be the Feature objects themselves.

Alternatively, finding all features from the database that have
length greater than 1000:

use Feature;

$hash_of_large_features=Feature->load(  <fix this>
                                      length =>['>',1000]
                                     );

=head2 Creating a new Feature:

use Feature;

my $feature=Feature->new(context     =>$context,
                         seqname     =>$seqname,
                         contig_start=>$start,
                         contig_stop =>$stop,
                         source_tag  =>'Genscan'
                         primary_tag =>'Intr',
                  );

=head2 Deleting a Feature:

$feature->delete_from_db;

=head2 Finding out about a Feature:

$start=$feature->start;

returns the start of the gene relative to its parent sequence ($par_pos).

$end=$feature->end;

returns the end of the gene relative to its parent sequence ($par_pos).

$absstart=$feature->absstart;

returns the gene's start position on the chromosome.

$absstop=$feature->absstop;

returns the gene's end position on the chromosome.

$strand=$feature->strand;

returns the gene's strand (+1,-1,0)

$gff_strand=$feature->gff_strand;

returns the gene's strand (+,-,.)

$length=$feature->length;

returns the gene's length.

=head1 DESCRIPTION

Feature is a feature on a sequence. It follows the GFF format in
terms of how the data is stored.  Its start and stop values are relative to
Virtual Contigs, whose position is stored in the LookUp table.

It lives in the database in the Feature table.  It is a GenericFeature, which
means it implements Bio::SeqFeatureI as well as all the methods common to Feature,
Gene, Feature::Annotation, and Feature::mod.

=head1 FEEDBACK

Like it or lump it, report to dblock@gene.pbi.nrc.ca.  Feel free to add
to the docs available at
http://bioinfo.pbi.nrc.ca/dblock/wiki

=head2 Reporting Bugs

Email the author with any bug reports.

=head1 AUTHOR - David Block

email dblock@gene.pbi.nrc.ca

=cut

package GQ::Server::Feature;
$VERSION = 1.00;
use strict;

use vars qw( $AUTOLOAD @ISA);  #Keep 'use strict' happy

use Carp;
use DBI;

use GQ::Server::GenericFeature;
use GQ::Root;

@ISA = qw(GQ::Server::GenericFeature);

{
    #Encapsulated class data
    #replace with your column names, and sensible defaults

	                            #   DEFAULT      ACCESSIBILITY
    GQ::Root->create();
}


### write in any custom subroutines here

sub _initialize {
    my ($self) = @_;

    $self->adaptor($self->context->adaptor($self));

    if ($self->type) {
		my $type=$self->type;
		
		# SeqCanvas, and other modules, rely on ->isa to determine feature type
		# in some cases (i.e. our ->type call is only good for Genquire
		# and since we strive to be BioPerl compatible...  A similar routine to
		# the one below is found in GenericFeature::type
		eval {"require Bio::SeqFeature::Gene::$type;"};
		unless ($@) {
			$self->add_to_isa("Bio::SeqFeature::Gene::$type");
		}
    }
    
    
###  This section inserts the object's data into the database
###  only if the new method was called by the class, i.e.
###  not as a copy of a previous object,  OR
###  if the caller does not have an id at present
###  which it would if it were loaded from the data source (see sub load)
    if (not $self->id) {
		if (not ($self->contig_stop)) {
			if ($self->contig_start) {
			if ($self->length) {    #user supplied a length, not a stop
				$self->contig_stop($self->contig_start+$self->length-1);
			} else {                #user supplied neither length nor stop
				$self->contig_stop($self->contig_start);  #1 nt feature
			}
			} else {
			return;  #$self has neither start nor stop
			}
		}
		
		if (not ($self->length)) {
			$self->length($self->end - $self->start + 1);
		}

		die "Must Acquire Lock On This Region To Proceed.\n"
		if $self->context->checklock(lockid       => $self->lockid,
					   contig_id    => $self->contig_id,
					   contig_start => $self->contig_start,
					   contig_stop  => $self->contig_stop);

		if ($self->score < 1) {
			$self->score(int($self->score*1000)); # convert to out of 1000
		}
		
		$self->{id}=$self->adaptor->insert_new(keys  =>[$self->_db_keys],
							   values=>[$self->_db_values],
							  );

		$self->type($self->primary_tag) unless $self->type;
		$self->adaptor->change_type($self, $self->type);  # this is necessary because the feature "type" 
		# is overloaded to represent both the type name (in the feature hash) and type id (in the database).
		# this the call to $self->_db_keys above filters out the "type", since the database is expecting
		# an integer type id, not a type name...  so the feature is initialized without a type, which in the 
		# genuqire database defaults to type "1", whicih is unfortunately the type id of a Gene type feature... UGH!
		# calling the change_type method as part of teh initialization ensures that the type id is updated to 
		# represent the real feature type.

		$self->insert_tag_values if defined $self->tagvalues;
    }
}

1;







