package GQ::Client::QueryScreen;

use strict;
use GQ::Client::Utilities;
use GQ::Server::DB::createVirtContig;
use GQ::Client::txtOutput;
use GQ::Client::txtHandAnnotate;
use GQ::Client::PlugInHandler;
use Tk::widgets qw(BrowseEntry FileSelect);
use Tk::JPEG;
use Bio::Tk::SeqCanvas;
use Cwd;
use vars qw(@ISA $AUTOLOAD);
use GQ::Server::Sequence;
use GQ::Client::Chat;
use GQ::Client::ShowSequence;
use GQ::Client::ShowSequenceContext;
#use GQ::Client::Blastmap;
use GQ::Client::annotMenuBrowser;
use GQ::Client::GenomeMapper;
use Bio::Tk::GO_Browser;
use LWP::UserAgent;
use GQ::Client::ContigListEventTrapper;
use Admin::createBlastDB;
use GQ::Server::Feature;
use GQ::Server::Gene;
use GQ::Server::Transcript;
use GQ::Root;

Tk::Widget->Construct('QueryScreen');

@ISA= qw(Tk::MainWindow);

{				#Encapsulated class data

                                   #  DEFAULT          ACCESSIBILITY
    GQ::Root->create(
		rootlevel       => 1,
		top             => [undef,          'read/write'],
		context         => [undef,          'read/write'],
		TextOutput      => [undef,          'read/write'],
		chat            => [undef,          'read/write'],
		lblSysMess      => [undef,          'read/write'],
		tools_height    => [350,            'read/write'],  # width and height of the "QueryScreen" window itself
		tools_width     => [250,            'read/write'],
		QS_colour		=> ["#ffccee", 	'read/write'],
		
		Msg1			=> ["gimme gimme gimme", 'read/write'],
		Msg2			=> ["A sequence with features", 'read/write'],
		
		QueryC	      	=> [undef,          'read/write'], # the main canvas upon which to draw the query parameter elements
		FileMenu        => [undef,          'read/write'],
		OptionsMenu     => [undef, 	  		'read/write'],
		PluginsMenu     => [undef, 	  		'read/write'],
		ToolsMenu       => [undef,          'read/write'],
		DBMenu          => [undef,          'read/write'],
		sys_message     => ['',             'read/write'],
		btnBegin        => [undef,          'read/write'],
		MapWindows      => [[], 	    	'read/write'], # can have multiple map windows from various calls
		MainWindows     => [[],             'read/write'], #list of all top-level windows in application
		ToolsHash       => [{},		  		'read/write'], # the visible tools will be determined by the features in the displayed sequence
		SelectedFrame   => [undef, 	  		'read/write'],
		MouseOverID     => [undef, 	  		'read/write'], # holds the FID of the widget that the mouse is currently over (needed to recolor on a mouse-leave)
		WORKING_DIR     => ['.',            'read/write'], # holds the directory where the sequence files are stored
		PLUGINS_DIR     => ['.',            'read/write'], # holds the directory where the sequence files are stored
		SOURCE_PLACEHOLDERS  => [undef, 	'read/write'], # deprecated, but possibly revived in the future
		BROWSER			=> [undef, 			'read/write'], # name of the default browser
		MapEST          => ['unmapped',     'read/write'], # holds the state of the MapEST checkbutton, and is queried when a new map is drawn
		BLAST_URL       => ['',	            'read/write'],
		BLAST_CONFIG    => [undef,	        'read/write'],
		TEMP_DIR       	=> [".", 	    	'read/write'],
		colors         	=> [{}, 	    	'read/write'], # holds a local copy of the default color for a given source - used for masking and unmasking sources
		BioPerlFeatureTypes => [{
						# this entire entry should be overwritten by the data in Context
						# but just in case, there are default values that, fingers crossd, should prevent crashes.
						# *****   NOTA BENE   *************************
						Gene=>'Bio::SeqFeature::Gene::GeneStructure', # THIS MUST NOT BE MODIFIED!!!!
						# *********************************************
						
						Transcript=>'GQ::Server::Transcript',  # These and below may be modified
						Exon=>'GQ::Server::Feature',
						UTR=>'GQ::Server::Feature',
						'TRNA Gene'=>'GQ::Server::Gene::GeneStructure',
						'RNA Exon'=>'GQ::Server::Feature',
						Promoter=>'GQ::Server::Feature',
						Intron=>'GQ::Server::Feature',
						Poly_A_Site=>'GQ::Server::Feature',
						Misc_Feature=>'GQ::Server::Feature',
							}, 		'read/write'],
		GenomeMap     	=> [undef, 	    	'read/write'], # reference to the chromosome map widget
		    );

    my $unique_index = undef; # a counter just to have unique index numbers for various features over all maps opened by the query screen
    # these are used, for example, by the import GFF routine, to give an index to the features being mapped
    # or to index the Blast features being mapped.


    # unique index incrementer
    sub _next_unique_index {
	if ($unique_index) {
	    ++$unique_index;
	} else {
	    $unique_index = 0;
	}
	return $unique_index;
    }
    
    sub _getNonGeneFeatureTypes {
		my ($self) = @_;
		my %FeatureHash = %{$self->BioPerlFeatureTypes};
		my @non_genes;
		foreach my $key(keys %FeatureHash){
			push (@non_genes, $key) unless (($FeatureHash{$key} =~ /GeneStructure/) || ($FeatureHash{$key}=~ /Transcript/));
		}
		return @non_genes;
	}

}

sub _assignBindings {

    my ($MapWindow, $QueryScreen, $MapObj) = @_;
    
	my $menu = $MapObj->Menu;
	my $ReCast = $MapObj->ReCastMenu;
	
	$QueryScreen->_defineReCast($ReCast, $MapObj);  # in SeqCanvas there are callbacks for recasting that need to be over-ridden with genquire callbacks
	
	my $plugins = $QueryScreen->_retrievePlugIns();
	my $pluginmenu = $MapWindow->Menu(-type => 'normal',
							  -menuitems => $plugins,
									 );
	my $f = $menu->cascade(-label => '~Send To', -tearoff => 0, -menuitems => $plugins);
    
    
	$menu->command(-label=> 'Blast Sequence with hit-import',
				   -command => sub {
					   $QueryScreen->reBlastExon($MapObj)
					   }
				   );
	$menu->command(-label=>'Select Blasted Exons in Common',  -command => sub {$QueryScreen->selectCommonExons($MapObj) });
	$menu->command(-label=> 'Show Sequence',
				   -command => sub {
					   my $w=GQ::Client::ShowSequence->new($QueryScreen, $MapObj);
					   push @{$QueryScreen->MainWindows}, $w
					   }
				   );
	$menu->command(-label=>'Sequence Context (exons only)',
				   -command => sub {
					   my $w=GQ::Client::ShowSequenceContext->new($QueryScreen,$MapObj);
						push @{$QueryScreen->MainWindows}, $w
					   });
	$menu->command(-label=>'Sequence Context (all features)',
				   -command => sub {
					   my $w=GQ::Client::ShowSequenceContext->new($QueryScreen,$MapObj,"unlimited");
					   push @{$QueryScreen->MainWindows}, $w
					   });
	$menu->command(-label=>'Clear Selections',     -command => sub {$MapObj->clearSelections});
	$menu->command(-label=>'DELETE FROM DATABASE', -command => sub {$MapObj->deleteFeatures($MapObj->getSelectedIDs)});
	$menu->command(-label=>'Export Fasta & GFF',  -command => sub {$QueryScreen->exportGFF($MapObj)});
	$menu->command(-label=>'Assign Custom Colors',
				   -command => sub {my $w=$MapObj->assignCustomColors($MapWindow);
					push @{$QueryScreen->MainWindows}, $w });
	$menu->command(-label=>"Annotate - Free Text", -command => sub {$QueryScreen->annotateSelected($MapObj)});
	$menu->command(-label=>"Annotate - From Menu", -command => sub {$QueryScreen->annotateSelected_defined($MapObj)});
												
	my $canvas = $MapObj->FinishedCanvas;
	my $menu2 = $canvas->Menu(-type => 'normal');
	$menu2->command(-label=>'Select SubFeatures',-command=> sub {$QueryScreen->selectsubFeatures($MapObj,$MapObj->getSelectedFeatures) });
	$menu2->command(-label=> 'Blast Sequence with hit-import',
				   -command => sub {
					   $QueryScreen->reBlastExon($MapObj)
					   }
				   );
	$menu2->command(-label=> 'Show Sequence',
				   -command => sub {
					   my $w=GQ::Client::ShowSequence->new($QueryScreen, $MapObj);
					   push @{$QueryScreen->MainWindows}, $w
					   }
				   );
	$menu2->command(-label=>'Sequence Context (exons only)',
				   -command => sub {
					   my $w=GQ::Client::ShowSequenceContext->new($QueryScreen,$MapObj);
						push @{$QueryScreen->MainWindows}, $w
					   });
	$menu2->command(-label=>'Sequence Context (all features)',
				   -command => sub {
					   my $w=GQ::Client::ShowSequenceContext->new($QueryScreen,$MapObj,"unlimited");
					   push @{$QueryScreen->MainWindows}, $w
					   });
	$menu2->command(-label=>'Clear Selections',     -command => sub {$MapObj->clearSelections});
	$menu2->command(-label=>'DELETE FROM DATABASE', -command => sub {$MapObj->deleteFeatures($MapObj->getSelectedIDs)});
	$menu2->command(-label=>'Export Fasta & GFF',  -command => sub {$QueryScreen->exportGFF($MapObj)});
	$menu2->command(-label=>'Assign Custom Colors',
				   -command => sub {my $w=$MapObj->assignCustomColors($MapWindow);
					push @{$QueryScreen->MainWindows}, $w });
	$menu2->command(-label=>"Annotate - Free Text", -command => sub {$QueryScreen->annotateSelected($MapObj)});
	$menu2->command(-label=>"Annotate - From Menu", -command => sub {$QueryScreen->annotateSelected_defined($MapObj)});


    $canvas->Tk::bind ("<Button-3>" => sub {$menu2->Popup(-popover => 'cursor',
						        -popanchor => 'nw'); });

	$MapWindow->bind ("<Button-1>" =>      # left mouse button brings up any additional details about the item into the light-blue text window
		      sub {
    			  unless ($QueryScreen->TextOutput) {
					my $topText = MainWindow->new(-title => "Details of Selected Items");
					push @{$QueryScreen->MainWindows}, $topText;
					my $txtOutput = GQ::Client::txtOutput->new($topText);
					$QueryScreen->TextOutput($txtOutput->window);
					$QueryScreen->TextOutput->bind ("<Destroy>" => sub {eval{undef $QueryScreen->{TextOutput}}});
    			  }
    			  #$QueryScreen->findPartnerFeatures($MapObj);
    			  $QueryScreen->writeTags($MapObj);   # writeTags is the routine that formats and writes the details to the text window.
    			  my @results = $QueryScreen->getBlastHit($MapObj->getSelectedFeatures);  # extracts blast hits from the database
    			  foreach my $result(@results){$QueryScreen->printTextOut([$result])}                        # returns an list of array refs,
    		      $QueryScreen->lightupDups($MapObj);
		      });
	
	
    $MapWindow->bind ("<Double-Button-1>" => sub { $QueryScreen->Annotate($MapObj) });

    $MapWindow->bind ("<Control-ButtonRelease-1>" => sub {$QueryScreen->selectFreeSequence($MapObj);});

    #$MapWindow->Tk::bind ("<Button-3>" => sub {$MapWindow->Menu->Popup(-popover => 'cursor',
	#					        -popanchor => 'nw'); });

	#$MapWindow->bind("<<DROP>>" => sub {$QueryScreen->RaiseAnnotation($MapObj)});
	
    $MapWindow->bind ("<Key-C>" => sub {$MapObj->clearSelections; $QueryScreen->printTextOut([""], 'wipe');});  # clears all mouse-selected features

    $MapWindow->bind ("<Key-P>" => sub {$QueryScreen->Text_to_Printer;                # this routine DOES NOT WORK very well
    									$QueryScreen->Canvas_to_Printer});            # but we are too lazy to fix it for the moment

    #$MapWindow->bind ("<Key-4>" => sub {my $Sim4 = GQ::Client::WB_Sim4->new($QueryScreen);        # start the Sim4 analysis
	#									push @{$QueryScreen->MainWindows}, $Sim4});

    $MapWindow->bind ("<Key-c>" => sub {my $w=GQ::Client::ShowSequenceContext->new($QueryScreen, $MapObj, "unlimited");  # bring up selected exons in their sequence context
				        push @{$QueryScreen->MainWindows}, $w});

    $MapWindow->bind ("<Key-s>" => sub {my $w=GQ::Client::ShowSequence->new($QueryScreen, $MapObj);                       # bing up selected exons with no additional sequence info
				        push @{$QueryScreen->MainWindows}, $w});
    #$MapWindow->bind ("<Control-Key-a>" => sub {$QueryScreen->annotateSelected($MapObj)});   # obselete - now we insist on controlled vocabulary
    $MapWindow->bind ("<Control-Key-a>" => sub {$QueryScreen->annotateSelected_defined($MapObj)});


    $MapWindow->bind ("<Control-Key-D>" => sub {$MapObj->deleteFeatures($MapObj->getSelectedIDs)}); # delete selected features from database

    $MapWindow->bind ("<Control-Button-3>" => sub {$QueryScreen->reBlastExon($MapObj)} );       # if a local/remote Blast program has been specified in the wb.conf, then blast away!

    $MapWindow->bind ("<Shift-Button-3>" => sub {$QueryScreen->selectCommonExons($MapObj)} );   # query the BlastLookUp table for all exons which share a blast homology, and select them all
                                                                                                # only the exons ON THE SCREEN will be selected, though more may exist in the database!

    $MapWindow->bind ("<Motion>" => sub {
		my @mouseovertags = $MapObj->DraftCanvas->gettags("Mouse_over");
		if ((scalar @mouseovertags) == 0){                   # if the mouse is over nothing
			return unless ($QueryScreen->{MouseOverID});   # check if it formerly *was* over something
			my $current = $QueryScreen->MouseOverID;    # if so, then get the ID of the last mouseover widget (stored locally)
			return unless $current;
			$MapObj->recolorWithTag('default', 'draft', [$current]);   # and recolor it to its default color
			undef $QueryScreen->{MouseOverID};          # then inform the local copy that there is no current mouseover widget
			return;
		}
		foreach my $mouseovertag(@mouseovertags){       # if the mouse is over something then go through its tags
			if ($mouseovertag =~ /FID\d+/){            # look for its ID tag
				if ($mouseovertag ne $QueryScreen->MouseOverID){
					my $current = $QueryScreen->MouseOverID;                 # need to double-check here because
					if ($current){$MapObj->recolorWithTag('default', 'draft', [$current])}  # sometimes the mouse-leave event slips through...
				}
				my $feature = $MapObj->AllFeatures($mouseovertag);
				my $source = $feature->source_tag;
				# if this source type has been switched to invisible, then ignore this mouseover event
				return if (!(${$QueryScreen->ToolsHash}{$source}));  # if this isn't in the tools list then exit
				return if (${${$QueryScreen->ToolsHash}{$source}->cget('-variable')} eq "off");  # what a struggle!  Checkboxes in Tk are very badly designed...
				$QueryScreen->MouseOverID($mouseovertag);  # otherwise store the current mouseover ID locally for later re-coloring
				$QueryScreen->_checkFrame($mouseovertag, $MapObj);    # check the frame w.r.t. currently selected widgets and 'light up' if it is in frame
			}
		}
	});

}

sub _defineReCast {
	my ($self, $ReCast, $MapObj) = @_;
	$ReCast->delete(0, 1000);  # clear the existing list 'cuz its bulloks
	
	foreach my $type ($self->_getNonGeneFeatureTypes) {
		$ReCast->add(
			'command',
			-label => "$type",
            -command => sub {$self->reCastAs($MapObj, $type);},
        );
	}
}

sub reCastAs {
	my ($self, $MapObj, $type) = @_;
	my %FeatureHash = %{$MapObj->getSelectedFeatures};
	my $newfeature; my @del_list; my @add_list;
	foreach my $FID (keys %FeatureHash){
		my $feature = $FeatureHash{$FID};
		$feature->type($type);
	}		
}	


sub selectFreeSequence {
	# allows you to freely select a region of the canvas and bring up the
	# sequence associated with that region
	
	my ($self, $MapObj) = @_;
	my ($start, $end, $strand);
	
	if ($MapObj->{-orientation} eq "horizontal"){
		my $x1 = $MapObj->DraftCanvas->canvasx($MapObj->dragx1);
		my $x2 = $MapObj->DraftCanvas->canvasx($MapObj->dragx2);
		if ($x1 > $x2){($x1, $x2) = ($x2, $x1)};
		my $length = (($x2 - $x1)/($MapObj->DraftMap->{scale_factor}));   #/
		
		my ($left, $right) = $MapObj->DraftCanvas->xview;
		$start = ($x1/($MapObj->DraftMap->{scale_factor})) - ($left * ($MapObj->MapSeq->length)); #/
		$end = $start + $length;
		$start=0 if ($start<0)
	} else {
		my $x1 = $MapObj->DraftCanvas->canvasx($MapObj->dragy1);
		my $x2 = $MapObj->DraftCanvas->canvasx($MapObj->dragy2);
		if ($x1 > $x2){($x1, $x2) = ($x2, $x1)};
		my $length = (($x2 - $x1)/($MapObj->DraftMap->{scale_factor}));   #/
		
		my ($left, $right) = $MapObj->DraftCanvas->yview;
		$start = ($x1/($MapObj->DraftMap->{scale_factor})) - ($left * ($MapObj->MapSeq->length)); #/
		$end = $start + $length;
		$start=0 if ($start<0)
	}

	while (!$strand || !($strand =~/^[+-]$/)){
		my $AskFeature = $self->DialogBox(-title => 'Forward or Reverse Strand?',
		   -default_button=> 'OK',
		   -buttons => [qw/OK Cancel/]);
		$AskFeature->add("LabEntry",
			-textvariable=>\$strand,
			-label=>'Do you wish to view the forward (+) or reverse (-) strand?',
			)->pack;
		my $answer=$AskFeature->Show;
		return if $answer eq 'Cancel';
	}
	
	my $w=GQ::Client::ShowSequenceContext->new($self, $MapObj, "all", $start, $end, disp_strand => $strand);# or disp strand -
	return unless $w;
	push @{$self->MainWindows}, $w;
}

sub annotateSelected {
	# this allows you to add free tag/value pairs to single or
	# groups of selected features on the canvas.  Good for putting together
	# groups of related features which span multiple source_tag types.
	
	my ($self, $MapObj) = @_;
	my $Features = $MapObj->getSelectedFeatures;
	my @Features = (values %{$Features});
	return if ($#Features == -1);
	
	my $tag;
	my $AskFeature = $self->DialogBox(-title => 'Tag:',
			   -default_button=> 'OK',
			   -buttons => [qw/OK Cancel/]);
		$AskFeature->add("LabEntry",
			-textvariable=>\$tag,
			-label=>'What is the tag for this tag/value pair:',
			)->pack;
		my $answer=$AskFeature->Show;
	return if $answer eq 'Cancel';
	$tag =~ s/\s+/\_/g;  # tags are not allowed to have spaces
	return if (!$tag || $tag eq "_");


    my $value;
	my $AskComments = $self->DialogBox(-title => "Value for $tag:",
			   -default_button=> 'OK',
			   -buttons => [qw/OK Cancel/]);
		$AskComments->add("LabEntry",
			-textvariable=>\$value,
			-label=>'please add a free-text comment here',
			)->pack;
		$answer=$AskComments->Show;
	$value = "none" if $answer eq 'Cancel';
	$value =~ s/"/\'/g;  # replace all instances of quotation marks

	foreach my $Feature (@Features){
		$Feature->add_tag_value($tag, $value);
	}
}


sub annotateSelected_defined {
	# this allows you to add pre-defined tag/value pairs (from a text file) to single or
	# groups of selected features on the canvas.  Good for putting together
	# groups of related features which span multiple source_tag types.
	
	my ($self, $MapObj) = @_;
	my $Features = $MapObj->getSelectedFeatures;
	my @Features = (values %{$Features});
	return if ($#Features == -1);
	
	my $mw = $self->Toplevel(-background => "black");
    my $annotBrowser = GQ::Client::annotMenuBrowser->new($mw, $self, \@Features);
	eval {$mw->destroy};
}	




sub _checkFrame {
	# this is an on-the-fly check of the frame of a
	# mouse-over exon.  If it is in-frame with currently selected
	# exons then it recolors it as bright green until the mouse leaves the area
	
    my ($self, $FID, $MapObj) = @_;
    my $SCF = $MapObj->AllFeatures($FID);  # get the currently moused-over Feature object
    my $Feature = $SCF->Feature;
    my $SeqObj = $MapObj->MapSeq;
    my $dir;
    my @SelectedSeqs;
    my %exons;
    my @exonIDs = @{$MapObj->getSelectedIDs};  	# get the list of exons on the working map that are currently tagged as select
    return if ($#exonIDs == -1);                # leave if there are none selected
    foreach (@exonIDs) {
    	my $exonSCF=$MapObj->AllFeatures($_);  # get the Feature object of that exon
    	my $exon = $exonSCF->Feature;
    	return unless ($exon && $exon->has_tag('id'));	  # seq feature generic throws an ugly error if you ask for a tag that doesn't exist
    	$exons{$exon->each_tag_value('id')}=$exon;    # add it to a hash of exons
    	$dir=$dir || $exon->strand;                   # and grab the strand information (only the last one is meaningful! So if you are
    												  # being silly and select across strands then you will get a garbage result)
     }
    $exons{$Feature->each_tag_value('id')} = $Feature;   # add the current mouse-over exon to this list

    if ($dir =~ /-/) {                                   # now sort the list by start or end coordinate so it can be properly translated
    	foreach (sort { $b->end <=> $a->end } values %exons) {  # if on the - strand then it is the end-coordinate that is sorted
    	    my $seq=Bio::PrimarySeq->new( -seq => ($SeqObj->getseq($_->start,$_->end))); # get the sequence of the next exon in start-order
    	    push @SelectedSeqs, ($seq->revcom->seq);    		     # push that sequence onto an array for later manipulation
    	}        #reverse sorted
    }
    else {
    	foreach (sort { $a->start <=> $b->start } values %exons) {   # on the + strand the start coordinate is sorted
    	    my $seq=Bio::PrimarySeq->new( -seq => ($SeqObj->getseq($_->start,$_->end)));
    	    push @SelectedSeqs, ($seq->seq);     # push that sequence onto an array for later manipulation
    	}       #forward sorted
    }



    foreach (@SelectedSeqs) {             # take the ordered series of sequences
	s/\s//;
    }
    my $SelectedSeq = Bio::PrimarySeq->new(-seq => (join '',@SelectedSeqs), -moltype => 'dna');  # join them
    my $SelectedTrans = $SelectedSeq->translate()->seq;                                          # translate them
    if (!($SelectedTrans =~ /\*\w/)) {$MapObj->recolorWithTag('#aaffcc', 'draft', ["Mouse_over"])}  # check if they contain a stop (*) codon
                                                                                                    # and if not, then brighten the current exon
}

sub _drawDependentTools {
	# this subroutine fills the drop-down Options menu allowing
	# the user to switch on/off the display of any particular
	# source tag.  The list is filled based on the sources which are currently being displayed.
	
    my ($self, $MapObj) = @_;

    my $q = $self->OptionsMenu;
    my $ret;  # holds the return (success/fail) of the creation of the object
    my %box;  # holds the entry widgets, keyed by the attribute they define
    if ($MapObj) {        # there has to actually BE a map visible in order to do this
    	$q->separator();
    	foreach my $source_tag ($MapObj->FinishedSourceLabels, $MapObj->Sources) { # for each of the sources on both canvases
    		my $state = "on";                                                            # default is "on" (visible)
    		next if (${$self->ToolsHash}{$source_tag});         # ignore if it already exists in the ToolsHash
    		${$self->ToolsHash}{$source_tag} = $q->checkbutton( # if not, then add a checkbutton with this label to the ToolsHash
					-label => $source_tag,
					-selectcolor => '#00FF00',
					-variable => \$state,
					-onvalue => 'on',
					-offvalue => 'off',
					-command => sub {	foreach my $MapRef(@{$self->MapWindows}){  # when the checkbutton is toggled, send the state (on/off)
											my $Map = ${$MapRef}[0];               # along with every visible $Map object to the _toggleVisibility routine
											if ($Map){$self->_toggleVisibility($Map, $source_tag, $state)}  # which will change its color to make it visible/not
										}
									});
    		
    	}
    } else {return}
}


sub _toggleVisibility {

	my ($self, $MapObj, $source_tag, $state) = @_;

    if ($state eq 'on') {
    	$MapObj->recolorWithTag('default', 'draft', ["Source $source_tag"]);    # call the SeqCanvas object's recolorWithTag routine
    	$MapObj->recolorWithTag('default', 'finished', ["Source $source_tag"]); # asking for the default source color to be displayed
    } else {
    	$MapObj->recolorWithTag('#FFFFFF', 'draft', ["Source $source_tag"]);    # or asking for the canvas background color to be displayed
    	$MapObj->recolorWithTag('#EEEEFF', 'finished', ["Source $source_tag"]); # if we are trying to make them "invisible"
    }
	
}


sub _recolorKnownTags {
	# certain primary_tag's such as promotors and polyA (eg. coming from GenScan) can be
	# easily picked out and highlighted in some way that makes them distinct from the
	# exons which are found by the same gene-finder.  In this routine, we outline them with
	# a thick blue or black line to make this distinction.
	my ($MapObj) = @_;
	return unless $MapObj;
	$MapObj->DraftCanvas->itemconfigure("Type prom", -outline => 'blue', -width => '3');
	$MapObj->DraftCanvas->itemconfigure("Type plyA", -outline => 'black', -width => '3');
	$MapObj->FinishedCanvas->itemconfigure("Type prom", -outline => 'blue', -width => '3');
	$MapObj->FinishedCanvas->itemconfigure("Type plyA", -outline => 'black', -width => '3');
	
}

sub _localDoZoom {
	# this routine is called when a docked_annotation window
	# is brought up on a map which has already been zoomed-in
	# the text which has just been drawn now has to be zoomed
	# and scrolled to the same extent as the map is
	my ($Map) = @_;
    my ($desired_scale,$zoom_ratio,$min_zoom,$max_zoom) = ($Map->{zoom_level},  (2 - $Map->AnnotTextMap->{scale_factor})/100,  $Map->AnnotTextMap->{scale_factor},  2);    #/
    my $current_loc = $Map->current_loc;     # the location of the last clicked widget, for centering purposes
    my $pre_scale_factor = $Map->AnnotTextMap->{scale_factor};   # get current scale factor
    my $normalized_desired_scale = (($desired_scale/100)**2)*100*$zoom_ratio + $min_zoom;  # calculate what we want the zoom to be based on the other maps
    my $zoom_factor = $normalized_desired_scale/$pre_scale_factor;
	$Map->AnnotTextMap->Zoom($zoom_factor, $current_loc);	# now zoom in the annotated text to the same extent, with the center point at the current location	
}

sub _dockedAnnotations{
	# this draws a pink text-canvas alongside the draft and finished annotation canvases
	# in order that the annotatinos for any given feature can be seen right beside
	# that feature in an AceDB-ish kind of way.
	my ($self, $dock_state) = @_;
	if ($dock_state eq "dock")	{                   # are we drawing or erasing this window?  if drawing then...
		my @Maps;
		foreach my $MapRef(@{$self->MapWindows}){   # first get all visible maps
			push @Maps, ${$MapRef}[0];
		}
		foreach my $Map(@Maps){                     # for each visible map
			next if ($Map->{-orientation} eq "horizontal");  # check if it is horizontal or vertical - only vertical maps can have docked annotations
			my $ALF = $Map->AnnotTextFrame($Map->MapFrame->Frame); #->pack(-side => 'left'));   # create a new frame for the pink canvas
			$ALF->Canvas(-width =>400, -height => 100, -background => "#ffdddd")->pack(-side => 'top', -anchor => 'nw', -expand => 0);  # draw the pink canvas
			$Map->AnnotTextCanvas($ALF->Canvas( -background => "#ffdddd", -width => 400, height => ($self->screenheight-200))->pack(-side => 'left', -anchor => 'nw'));
			# make the canvas into an AnnotMap object
			$Map->AnnotTextMap($Map->AnnotTextCanvas->AnnotMap(0, 0, 400, ($self->screenheight-200), -axis_loc => 10, -orientation => 'vertical', -range => ($Map->{-range})));
	        # ensure that all three maps respond to the scroll-bar
	        $Map->ScrollBar->configure(-command => sub{ $Map->FinishedCanvas->yview(@_); $Map->DraftCanvas->yview(@_); $Map->AnnotTextCanvas->yview(@_)});
		
	        foreach my $SCF(values %{$Map->AllFeatures}){
				my $FID = $SCF->FID;
				if ($Map->is_finished_feature("$FID")){		# only write annotations for gene-level objects (i.e. on the blue canvas)
					my @coords;
					push @coords, [$SCF->start, $SCF->end];  # need to know the (vertical) extents of this feature to draw the text
					my $text;
					foreach my $tag($SCF->all_tags) {            # now get all tag/value pairs for this annotation
						next if ($tag eq "group_id");
						next if ($tag eq "GO_annotation");
						$text .= "$tag:\cI " . (join " \cI ", $SCF->each_tag_value($tag)) . "\n"; # note that it may have multiple values per tag...
					}
					unless ($SCF->Feature->GO && @{$SCF->Feature->GO}[0]){$SCF->Feature->GO_find};
					foreach my $GO (@{$SCF->Feature->GO}){
						my $id = $GO->GO_id;
						my $Term = $GO->term;
						chomp $id; chomp $Term;
						$text .= "\n\cI$id  $Term\n";  # write it as tag=value into the window and add a Tk-tag to the entire piece of tex
					}
					# now map it as if it were a feature itself!  using the -just_labels switch
					$Map->AnnotTextMap->MapObject(\@coords, '-ataxis' => 5, '-label' => $text, '-labelfont' => "Courier 10 normal", '-labelcolor' => '#000000', '-just_labels' => "yes", '-color' => '000000');
				} else {}
			}
			
			# the stuff below doesn't work because I have set the dimensions inside of SeqCanvas
			$Map->MapFrame->packForget;
			$Map->AnnotTextFrame->pack(-side => 'left', -anchor => 'nw', -expand => 0);
			$Map->MapFrame->pack(-side => 'left',  -anchor => 'nw', -expand => 0);
			
			my $a = $Map->MapFrame->toplevel->geometry;
			$a =~ /(\d+)x(\d+)\+-?(\d+)\+-?(\d+)/;      #get current screen position of top-level window eg. 500x300+20+-45
			$Map->MapFrame->toplevel->geometry("".($1+400)."x"."$2+$3+$4");  # set it so that the control bar is entirely visible at the top of the screen

			
			_localDoZoom($Map);  # now force the map to be the same zoom level as the other maps.
		}
	} else {     # we are trying to DESTROY the docked window
		my @Maps;
		foreach my $MapRef(@{$self->MapWindows}){   # get all maps
			
			push @Maps, ${$MapRef}[0];
		}
	    foreach my $Map(@Maps){
			next if ($Map->{-orientation} eq "horizontal");  # from the horizontal maps
		    next if (!$Map->AnnotTextFrame);                 # ensure that there really is a docked annotation
		    eval {$Map->AnnotTextCanvas->packForget; $Map->AnnotTextFrame->packForget; $Map->AnnotTextFrame->destroy};                   # and if there is, then blow it to bits
		    undef $Map->{AnnotTextCanvas};                   # reset the hash values to undef
		    undef $Map->{AnnotTextFrame};
		    # now only the two canvases need to respond to the scroll-bar
			$Map->ScrollBar->configure(-command => sub{ $Map->FinishedCanvas->yview(@_); $Map->DraftCanvas->yview(@_)});
	        my $a = $Map->MapFrame->toplevel->geometry;
			$a =~ /(\d+)x(\d+)\+-?(\d+)\+-?(\d+)/;      #get current screen position of top-level window eg. 500x300+20+-45
			$Map->MapFrame->toplevel->geometry("".($1-400)."x"."$2+$3+$4");  # set it so that the control bar is entirely visible at the top of the screen
		}
	}
	
}
		
	
sub new {
	# this routine is largely undocumented as it simply creates the main window
	# and initializes the various menus and such... nothing particularly interesting.
	
	# make splash screen *********************************************
	# ****************************************************************
    my $splash = MainWindow->new();
	my $xoff = int(($splash->screenwidth)/2) - 275;  #/
	my $yoff = int(($splash->screenheight)/2) - 118; #/
	
	$splash->geometry("550x237+$xoff+$yoff");
	
	my $sc = $splash->Canvas(-width => 550, -height => 237)->pack();
	my $splashimage = $sc->Photo(-format => 'jpeg', -file => "./Genquire_Splash.jpg");
	$sc->createImage(275, 118, -image => $splashimage);
	$splash->update;
	# ********************************************** end of splash screen
	# *******************************************************************



    my ($caller, $top, %args) = @_;
    my $caller_is_obj = ref($caller);
    my $class = $caller_is_obj || $caller;
    my $self = $top->Toplevel;

    bless $self, $class;

    $self->Busy;$self->update;   # set the cursor while all this is going on...
    $splash->raise();

    foreach my $attrname ( $self->_standard_keys ) {
    	if (exists $args{$attrname}) {
    	    $self->{$attrname} = $args{$attrname} }
    	elsif ($caller_is_obj) {
    	    $self->{$attrname} = $caller->{$attrname} }
    	else {
    	    $self->{$attrname} = _default_for($attrname) }
    }
    my ($orgname) = $self->context->organism->latin;

    $SIG{'INT'}=sub {$self->Quit};
    $SIG{'TERM'}=sub {$self->Quit};

	# here we need to update the list of valie FeatureTypes based on what is in the Context object.
	# this is all wel and good, except that anything that is destined to be a GQ::Server::Gene must
	# be intercepted and made into a Bio::SeqFeature::Gene::GeneStructure at the level of the GUI
	# or things don't work right...
	my %type_hash = $self->context->_get_types;
	if (scalar(keys %{$self->BioPerlFeatureTypes})){  # there has to be something there, or we are in BIIIIG trouble!
		foreach (keys %type_hash){
			if ($type_hash{$_} eq "GQ::Server::Gene"){$type_hash{$_} = "Bio::SeqFeature::Gene::GeneStructure"}
		}
		$self->BioPerlFeatureTypes(\%type_hash);
	}
	# okay, we're done!
	
	# lets goooooooo!!!

    $self->title("Genquire - $orgname");

    my $MenuFrame = $self->Frame(-relief => 'ridge', -borderwidth => '2')->pack(-side => 'top', -fill => 'x');

    $self->FileMenu($MenuFrame->Menubutton(-text => "File", -tearoff => '0',
					  -menuitems => [
							['command' => "Overlay XML File", -state => "disabled", -command => sub {$self->overlayXML}],
							undef,
							['command' => "Create/Import New Sequence", -command => sub {$self->importSeq}],
							undef,
							['command' => "Export Fasta & GFF", -command => sub {$self->exportGFF}],
							['command' => "Export XML", -state => "disabled", -command => sub {$self->exportXML}],
							undef,
							['command' => "Export Annotated cDNAs", -command => sub {$self->exportcDNA}],
							undef,
							['command' => "Quit", -command => sub {$self->Quit}],

							])->pack(-side => "left"));

    $self->OptionsMenu($MenuFrame->Menubutton(-text => "Options", -tearoff => '0',
    								-menuitems => [])
    						)->pack(-side => "left");

    my $dockedannot = "undock";
    my $MapEST = "unmapped";
    $self->ToolsMenu($MenuFrame->Menubutton(-text => "Tools", -tearoff => '0',
    								-menuitems => [
    									['checkbutton' => "Docked Annotation Canvas",
    											-variable => \$dockedannot,
    											-onvalue => "dock",
    											-offvalue => "undock",
    											-command => sub {$self->_dockedAnnotations($dockedannot)}],
    									#['checkbutton' => "MapESTs",
    									#		-variable => \$MapEST,
    									#		-onvalue => "mapped",
    									#		-offvalue => "unmapped",
    									#		-command => sub {$self->MapEST($MapEST);$self->mapEST($MapEST)}],


    									#['command' => "Do Sim4 Alignment",-command => sub {my $Sim4 = GQ::Client::WB_Sim4->new($self);
										#				           push @{$self->MainWindows}, $Sim4}],


    								])
    						)->pack(-side => "left");

    my $plugins = $self->_retrievePlugIns();

    $self->PluginsMenu($MenuFrame->Menubutton(-text => "Plug-ins", -tearoff => '0',
    								-menuitems => $plugins,
    								)
    					)->pack(-side => "left");
    								

    #  $AllToolsFrame holds the query boxes and chromosome pictures
    $self->{AllToolsFrame} = $self->Frame()->pack(-side => 'top', -fill => "both", -expand => 1);

    #  $ToolsF holds the query boxes
    my $ToolsF = $self->{AllToolsFrame}->Frame(-relief => 'ridge', -borderwidth => '2')->pack(-side => 'left', -fill => "y");


	my $bg = $self->QS_colour;
	
    $self->QueryC($ToolsF->Scrolled('Canvas',
				'-background' => $bg,
				'-scrollbars' => 'se',
				'-width' => ($self->tools_width),
				'-height' => ($self->tools_height)
				)->pack(-side => 'left', -fill => 'y'));

	
	$self->lblSysMess($self->Label(-background => '#000000', -foreground =>"#ccccFF", -text => "System messages...")->pack(-side=>'bottom', -fill=>'x'));

	my $q = $self->QueryC;   #just to make later referencing easier - $q = "query boxes"

    my ($ret, %box);
	my @box=qw(contig start stop orientation);
    foreach my $box(@box){   # initialize a hash with empty values for all of the keys in the above list
    	$self->{$box} = "";     # these will later become keys of $self->{key}
    }

    my @contigs;

    # select Contig label
    my $lblChromosome = $q->Label(-background => $bg, -text=>'Select Contig');
    $ret = $q->createWindow(100,10, -window=>$lblChromosome, -anchor => "w");


    # drop down list of contigs; default list width is 1 as it is currently empty
    # this is tricky - to get other visible elements to respond to changes in the drop-down list, the value must be "tied"
    # to cause an "event" to be triggered.  See the ContigListEventTrapper.pm module to
    # to understand this event.  In the end, it just lights up the contig on the contig map.
    tie $self->{contig}, "GQ::Client::ContigListEventTrapper", $self;
    my $cboxcontig = $q->BrowseEntry(-width => 1, -listwidth => 1, -background => '#ffffff', -relief=>'sunken', -variable=>\$self->{contig});
    $self->{contiglist} = $cboxcontig; # encapsulate the widget as we need it for the "import sequence" routines


    my $length=10;
    foreach my $chr($self->context->get_contigs_by_name) {
	#$self->{contig}=$chr; #is this necessary?  This is assigned by cboxcontig...
	$cboxcontig->insert(1, $chr);
	if (length($chr) > $length){$length = length($chr)}  # catch the longest entry to set the width of the widget
	push @contigs, $chr;
    }
    $cboxcontig->configure(-width => ($length), -listwidth => (($length)*3));	# +8 to make room for scrollbar
    										    	# *8 because listwidth is measured in pixels not characters
    $ret = $q->createWindow(100,30, -window=>$cboxcontig, -anchor => "w");

    my $btnPrevious = $q->Button(-text => "<<", -command => sub {$self->_getNext_or_Previous("previous")});
    $self->{PrevButton} = $btnPrevious;
    my $btnNext = $q->Button(-text => ">>", -command => sub {$self->_getNext_or_Previous("next")});
    $self->{NextButton} = $btnNext;
    $ret = $q->createWindow(100,30, -window=>$btnPrevious, -anchor => "e");
    $ret = $q->createWindow(130+(($length)*8),30, -window=>$btnNext, -anchor => "w");


	# label "Start" nucleotide
    my $lblStart = $q->Label(-background => $bg, -text=>'Start');
    $ret = $q->createWindow(100,60, -window=>$lblStart, -anchor => "w");

    # text entry "Start" nucleotide position
    my $cboxstart = $q->Entry(-background => '#ffffff', -width => 8, -relief=>'sunken', -textvariable=>\$self->{start});
    $ret = $q->createWindow(100,80, -window=>$cboxstart, -anchor => "w");

    my $lblStop = $q->Label(-background => $bg, -text=>'Stop');
    $ret = $q->createWindow(100,110, -window=>$lblStop, -anchor => "w");
    my $cboxstop = $q->Entry(-background => '#ffffff',  -width => 8,-relief=>'sunken', -textvariable=>\$self->{stop});
    $ret = $q->createWindow(100,130, -window=>$cboxstop, -anchor => "w");
	$self->{orientation} = "horizontal";
    my $rbtnOrientationH = $q->Radiobutton(-background => $bg, -width => 15, -anchor => 'w', -text=>'Horizontal', -value => "horizontal", -variable => \$self->{orientation});
    my $rbtnOrientationV = $q->Radiobutton(-background => $bg, -width => 15, -anchor => 'w', -text=>'Vertical', -value => "vertical", -variable => \$self->{orientation});
    			
	$ret = $q->createWindow(100,160, -window=>$rbtnOrientationH, -anchor => "w");
    $ret = $q->createWindow(100,185, -window=>$rbtnOrientationV, -anchor => "w");

	

    ############################################################################
    ############################################################################
    ##########################    BEGIN HERE   #################################
    ############################################################################
    ############################################################################


    # this is a 'critical' element of  the display -> this is where it all begins...
    $self->btnBegin($q->Button(-text=>$self->Msg1,
					-command => sub {
						return if ($self->{contig} eq "" or ($self->{stop} < $self->{start}) or (!($self->{orientation} =~ /horizontal|vertical/)));  # make sure the values are valid

						$self->btnBegin->configure(-text=>$self->Msg2);   # everything is a Go-Go... GRRR baby!!  VERY GRRRR!!
						my $chrthere = 0;
						foreach (@contigs) {  # @contigs was filled above from the database query of what contigs are available...
    				       if ($box{contig}==$_) {  # make sure that what they requested is actually one of the contigs that is available
							   $chrthere++;             # make this "true" if it is
    				       }
						}
						return unless $chrthere;     # if this is still 0 then that contig didn't exist
						$self->openMainWindows;
						$self->btnBegin->configure(-text=>$self->Msg1);   # everything is a Go-Go... GRRR baby!!  VERY GRRRR!!
						
					}
					));

    $ret = $q->createWindow(100,220, -window=>($self->btnBegin), -anchor => "w");
    
    my $COMMIT = $q->Button(-text=>"Commit Changes",
					-command => sub {
						$self->context->COMMIT;
					}
					);

    $ret = $q->createWindow(100,300, -window=>($COMMIT), -anchor => "w");


    # configure the scrolling canvas of the QueryScreen so that all widgets are displayed...
    $q->configure(-scrollregion => [$q->bbox("all")]);

    # make the chromosome display
	$self->createChromosomeDisplays();	

	#    now that everything is drawn get rid of splash screen
    $splash->destroy;

    $self->Unbusy; $self->update;
    
    #MainLoop;
    # it isn't obvious to me why the line above was there...??
    # anyway, now I return the QueryScreen object instead
    return $self;
}

sub _retrievePlugIns {
	my ($self) = @_;
	my $PLUGINS_DIR = $self->PLUGINS_DIR;
	my $CUR_DIR = cwd;
	my @plugin_list;
	chdir $PLUGINS_DIR;
	open PLUG, "plugins.conf";
	while (my $pluginline = <PLUG>){
		chomp $pluginline;
		next if (!$pluginline);
		next if ($pluginline =~ /^\s*\#/);
		my ($name, $executable, @args) = split ",", $pluginline;
		my $switch;
		$switch = "off";
		push @plugin_list, ['command' => "$name",
							-command => sub {#print "\n\nstarting plugin $name, $executable, @args\n\n";
										$self->_openPlugIn($name, $executable, @args);
    										}
    						];
	}
	chdir $CUR_DIR;
	return \@plugin_list;
}

sub _openPlugIn {
	my ($self, $plugin, $command, @args) = @_;
	# print "creating handler with args $command + @args\n";
	my $handler = GQ::Client::PlugInHandler->new(PARENT => $self, program => $plugin, command => $command, parameters => \@args);
	$handler->run;
	
	
}


sub createChromosomeDisplays {

    my ($self, $splash) = @_;
    my $organism = $self->context->organism->common;
	my $GenomeMap = GQ::Client::GenomeMapper->new($self, -title => $organism, -background => "white");
	$GenomeMap->Busy; $GenomeMap->update;
	if ($splash){$splash->raise} # if there is a splash screen then raise it in the Z-order
	my @chrs = $self->context->get_chrs_by_id;
	
	foreach my $chr_id(@chrs){
		my $assembled=$self->context->get_ordered_assemblies_by_chr($chr_id);
        $GenomeMap->addChromosome($chr_id, $assembled);
	}
	
	my $ChrMap = $GenomeMap->ChromosomeMap;
	push @{$self->MainWindows}, $GenomeMap;	
		
	$GenomeMap->bind("<Motion>" =>
			sub {my $contig = $GenomeMap->getMouseoverContig;
				return if (!$contig);
				$self->lblSysMess->configure(-text => "contig: $contig");
				$self->update;
				$ChrMap->dtag("current_ctg", "current_ctg");				# remove the tag
				$ChrMap->addtag("current_ctg", withtag => $contig);		# add "current" tag to current
				 });

	$GenomeMap->bind("<Button-1>" =>
			sub{my $contig =$GenomeMap->getSelectedContig;
				return if (!$contig);
				$self->{contig} = $contig;
				$self->lblSysMess->configure(-text => "contig: $contig");
				});
				
	$GenomeMap->bind("<Double-Button-1>" =>
			sub{
				my $contig = $GenomeMap->getSelectedContig;
				return if (!$contig);
				$self->{contig} = $contig;	# record it as current
				#$self->{chrmap_contig} = $contig;  # this is necessary because a single click changes $self->{contig} value
				$self->{start} = 0;  # this will cause the entire contig to be loaded
				$self->openMainWindows;  # load the seqcanvas for this contig
				});
	
	$self->_highlight_flagged_contigs($GenomeMap);
	$self->GenomeMap($GenomeMap);
	$GenomeMap->Unbusy; $GenomeMap->update;
	
}
sub _highlight_flagged_contigs {

	my ($self, $GenomeMap) = @_;
	my $ContigListRef = $self->context->get_flagged_contigs($self->context->user);
	$GenomeMap->lightAndAddTag("red", "flagged", $ContigListRef);	

}

sub openMainWindows {
	my ($self) = @_;
	
	# if a light-blue details text-box doesn't already exist then create one
	unless ($self->TextOutput) {
    	my $topText = $self->Toplevel(-title => "Details of Selected Items");
    	push @{$self->MainWindows}, $topText;  # add to list of open TopLevel windows (for destruction purposes)
    	my $txtOutput = GQ::Client::txtOutput->new($topText);  # this text box is consistently referred to as $txtOutput throughout this program
    	$self->TextOutput($txtOutput->window);     # and it is stored as an encapsulated thingy
    	$self->TextOutput->bind ("<Destroy>" => sub {eval{undef $self->{TextOutput}}});  # which destroys itself when closed
	}

	# if a chat window does not exist, then create it
	unless ($self->chat){ # && Tk::Exists($self->chat)) {    # bring up the chat window in the same way as above
	    $self->chat(GQ::Client::Chat->new($self, $self->context));
	    push @{$self->MainWindows}, $self->chat;
	    $self->chat->iconify;
	}

    # create a new Map window                           # ********************************
	my ($MapObj,$MapWin) = $self->createSeqCanvas;  	#   CALL THE "BEGIN" SUBROUTINE - creates the map-canvases and returns their handles
														# ********************************
    if ($self->{start} eq ''){$self->{start} = 1}
    if ($self->{stop} eq ''){$self->{stop} = $self->context->get_contig_length($self->{contig})}
	push @{$self->MapWindows}, [$MapObj, $self->{start}, $self->{stop}, $self->{contig}, $MapWin ];   # store start/stop/contig along with the reference to their MapObj
	$MapObj->MapFrame->bind("<Destroy>" => sub { my $index = 0; my $found = 0;
					foreach my $MapRef(@{$self->MapWindows}){  # go through list of all Map objects
						my $Map = ${$MapRef}[0];               # get the Map object itself
						$found = $index if ($Map == $MapObj);  # record the index position of this if it is the one we are destroying
						++$index;                              # increment the index position
					}                                          # loop
					eval {splice (@{$self->MapWindows}, $found, 1)};  # now splice out that record from the list
					});
	
	$self->_highlight_flagged_features($MapObj);
	
}
sub _highlight_flagged_features {
	my ($self, $MapObj) = @_;
	my @FeatureList = @{$self->context->get_flagged_features($self->GenomeMap->CurrentFlag)};
	
	# this routine is quite inefficient as it tries to lightup all flagged features
	# regardless of whether or not they are visible on the currently displayed maps.
	# Unfortunately, there is no way (yet) to find the intersection of the set of flagged
	# features, and the set of displayed features...
	
	foreach my $FeatureIndex (@FeatureList) {
	    #$MapObj->selectWithTag(["DB_ID $FeatureIndex"]);
		# there is a problem here - DB_ID Featureindex refers to both the widget and
	    # its label (if present).  This impedes the use of "width" or "outline" as valid
	    # highlighting options since text and rectangles respond differently to these option calls
	    # It should be possible to extract the individual sub-features which correspond to the
	    # label&rectangle by doing a "find", followed by a query for the tag "bioTk_Map_Label"
	    # which is applied to labels by bioTkPerl (line 794 bioTk_Map).  Then the two things could
	    # be highlighted differently.
	    # For now I am too lazy...
		my @features = $MapObj->DraftCanvas->find('withtag' => "DB_ID $FeatureIndex");
		foreach my $feature(@features) { # $feature contains the Tk Canvas widget id... deep inside of Tk
			next if (grep {$_ =~ "bioTk_Map_Label"} $MapObj->DraftCanvas->gettags($feature));  # ignore the label
			$MapObj->DraftCanvas->itemconfigure($feature, -fill => 'red', -outline => 'red', -width => 3);
		}
		# draft and finished canvases must be treated separately due to Tk's numbering of features 
		# which is redundant over the two canvases
		@features = $MapObj->FinishedCanvas->find('withtag' => "DB_ID $FeatureIndex");
		foreach my $feature(@features) { # $feature contains the Tk Canvas widget id... deep inside of Tk
			next if (grep {$_ =~ "bioTk_Map_Label"} $MapObj->FinishedCanvas->gettags($feature));  # ignore the label
			$MapObj->FinishedCanvas->itemconfigure($feature, -fill => 'red', -outline => 'red', -width => 3);
		}
	}
}


sub _getNext_or_Previous {
    my ($self, $np) = @_;  #$np holds either "next" or "previous"

    my $contig=$self->context->get_next($self->{contig},$np);

    if ($contig){
	$self->{contig} = $contig;
	$self->{start} = 0;  # this will cause the entire contig to be loaded
	$self->lblSysMess->configure(-text => "");
	$self->update;
	#$self->openMainWindows;
    }
    else {
	$self->lblSysMess->configure(-text => "End of Tiling Path Reached!");
	$self->update;
    }
}

sub createSeqCanvas {
    my ($self) = @_;
    my $start = $self->{start};
    my $stop = $self->{stop};
    my $orientation = $self->{orientation};
    my $contig = $self->{contig};
    my $dir = $self->WORKING_DIR;
    my $ADDITIONAL_SOURCES = $self->SOURCE_PLACEHOLDERS;

    $self->lblSysMess->configure(-text => "Querying Database");
    $self->update;

    my $SeqObj;            # first create the Bio::Sequence object based on these criterion
    if ($start) {
        my $length=$stop-$start+1;
		my $lockid=$self->context->lock(add=>[$contig,$start,$stop]);
		$lockid or warn "Unable to acquire lock on this region\n";
        $SeqObj=GQ::Server::Sequence->new(context=>$self->context, lockid=>$lockid, contig=>$contig, start=>$start, length=>$length, dir=>$dir); # if there is a start, then get a subseq
    } else {
		my $lockid=$self->context->lock(contig=>$contig);
		$lockid or warn "Unable to acquire lock on $contig\n";
        $SeqObj=GQ::Server::Sequence->new(context=>$self->context, lockid=>$lockid, contig=>$contig, dir=>$dir);   # otherwise get the entire contig
    }
    my $axis_length = ($orientation eq "horizontal")?($self->screenwidth - 200) : ($self->screenheight - 200);  # the long axis is 200 pixels smaller than the screen

    my $titleCaption= "Map of $contig ",(($start)?" from $start to $stop":"");
    my $mapWindow = $self->Toplevel(-title => $titleCaption);
    push @{$self->MainWindows},$mapWindow;
    my $frame = $mapWindow->Frame('-background' => '#ffffff')->pack(-side => 'top', -fill => 'both', -expand => 1);

    my %args;
    $args{label} = "gene_name";
    $args{"-orientation"} = $orientation;
	$args{BioPerlFeatureTypes} = $self->BioPerlFeatureTypes;
    if ($ADDITIONAL_SOURCES){$args{AdditionalSources}= $ADDITIONAL_SOURCES}

    # ***********************************************************************    	
    # ********** And here's where the magic begins!!!!  (of course, most of the magic is really in the $SeqObj)	label => "gene_name",
    my $MapObj = Bio::Tk::SeqCanvas->new($axis_length, $frame, ($self->lblSysMess), $SeqObj, %args);  # $MapObj is a handle to the map objects that you have just created
    # ***********************************************************************
	# ***********************************************************************
	
	_assignBindings($mapWindow, $self, $MapObj);    # this line assigns all of the button-click events to various subroutines
    _recolorKnownTags($MapObj);                     # this recolors things like promotors and polyA signals
    _drawDependentTools ($self, $MapObj);           # this fills the drop-down Options menu with the current set of displayed Sources (to be toggled on/off)
    #if ($self->MapEST eq 'mapped'){$self->mapEST('mapped', [$MapObj, $self->{start}, $self->{stop}, $self->{contig}])}; # check the EST checkbox in the Tools menu
                                                                                                                            # if it is "on", then map the EST's for this map-region
    $self->lblSysMess->configure(-text => ""); $self->update;   # erase the system-messages label as we are now finished

    return ($MapObj,$mapWindow);                                # return both the handles to the map object, and the window that contains it


}
sub overlayXML{}  # not available yet
sub exportFasta {  # deprecated at the moment...

	my ($self, $MapSeq)=@_;
    my ($seq, $name, $length, $start);

    if ($MapSeq) {
    	$seq = $MapSeq->seq;
    	$name = $MapSeq->base_contig;
		$length = $MapSeq->length;
		$start = $MapSeq->start;
        $name = $name . "-nt$start\_ln$length";
    } else {
    	my $SubSeqDiag= $self->DialogBox(-title => 'Choose Sequence Name',
    					 -default_button=> 'Export',
    					 -buttons => [qw/Export Cancel/]
    					);
    	$SubSeqDiag->add("Label",
    			 text => 'Please choose the sequence to export from the list:'
    			)->pack;
    	my $be=$SubSeqDiag->add("BrowseEntry",
    				-variable=>\$name,
    			       )->pack;
    	foreach my $sequence_name($self->context->get_contigs_by_name) {
    	    $be->insert("end",$sequence_name);
    	}
    	my $continue=$SubSeqDiag->Show;
    	return if $continue eq 'Cancel';

	$seq=$self->context->get_contig_sequence($name);
    	$length = length($seq);
    }

    my $path;
    my $AskFile = $self->DialogBox(-title => 'Export to Folder:',
				   -default_button=> 'Export',
				   -buttons => [qw/Export Cancel/]);
    $AskFile->add("LabEntry",
		-textvariable=>\$path,
		-label=>'Enter the full path to your desired output folder:',
		)->pack;
    my $pathdisp=$AskFile->Show;
    return if $pathdisp eq 'Cancel';

    $path = _addSlash($path);

    open OUT, ">$path".$name .".fasta" or sub {my $error = $self->DialogBox(-title => "Invalid Path/File error",
									-default_button => "OK",
									-buttons => ["OK"]);
									my $result = $error->Show;
									return};
	$self->lblSysMess->configure(-text => "Beginning FASTA Export");
    $self->update;
    	
    $seq =~ s/([^\n]{70})/$1\n/g;
    print OUT ">$name from Genquire Start $start Length $length\n";
    print OUT $seq;
    print OUT "\n";
    close OUT;

    my $complete = $self->DialogBox(-title => "Export Complete", -default_button=>"EXPORT COMPLETE - Click to continue", -buttons=>["EXPORT COMPLETE - Click to continue"]);
    my $result = $complete->Show;
    $self->lblSysMess->configure(-text => "");
    $self->update;
}
sub exportcDNA {
	my ($self) = @_;
    my $path;
    my $AskFile = $self->DialogBox(-title => 'Export to Folder:',
				   -default_button=> 'Export',
				   -buttons => [qw/Export Cancel/]);
    $AskFile->add("LabEntry",
		  -textvariable=>\$path,
		  -label=>"Enter the full PATH to your desired output folder.\nGene names are used as filenames.\nExisting files will be clobbererd!",
		 )->pack;
    my $pathdisp=$AskFile->Show;
    return if $pathdisp eq 'Cancel';
	
    $path = GQ::Client::Utilities::_addSlash($path);

	open OUT, ">$path" . "test" or sub {my $error = $self->DialogBox(-title => "Invalid Path/File error",
													-default_button => "OK",
													-buttons => ["OK"]);
									my $result = $error->Show;
									return 0};
	close OUT;
	unlink "$path" . "test";
	$self->lblSysMess->configure(-text => "Beginning cDNA Export");
	$self->update;								
	foreach my $MapRef(@{$self->MapWindows}){
		my $MapObj = ${$MapRef}[0];
		my %GeneFeatures = %{$MapObj->getFeaturesWithTag(["ObjectType GQ::Server::Gene"])};
		foreach my $Gene(values %GeneFeatures) {
			$self->selectsubFeatures($MapObj, $Gene);
			my $ExonHash = $MapObj->getSelectedFeatures;
			my ($SortedFIDs, $SortedFeatures) = sort_by_start_features($MapObj, $ExonHash);
			my $cDNA = "";
			foreach my $Feature (@{$SortedFeatures}){
				next unless ($Feature->type =~ /exon/i);
				my $temp= $MapObj->MapSeq->getseq($Feature->start, $Feature->end);
				if ($Feature->gff_strand =~ /-/){$temp =~ tr/ATCG/TAGC/; $temp = scalar reverse $temp}
				$cDNA .= $temp;
			}
			next if (length($cDNA) == 0);
			my $tempseq=Bio::PrimarySeq->new( -seq => $cDNA);
			my $translation = $tempseq->translate->seq;
			my $GeneName = join "", $Gene->each_tag_value("name");
			open OUT, ">$path" . "$GeneName";
			print OUT ">$GeneName.seq\n";
			for (my $i = 0; $i < (length($cDNA)+50); $i = $i + 50){
				print OUT substr($cDNA, $i, 50) . "\n";
			}
			print OUT "\n";
			print OUT ">$GeneName.pep\n";
			for (my $i = 0; $i < (length($translation)+50); $i = $i + 50){
				print OUT substr($translation, $i, 50) . "\n";
			}
			close OUT;
			$self->lblSysMess->configure(-text => "Exported $GeneName");
			$self->update;
		}
		$MapObj->clearSelections;
	}
	my $complete = $self->DialogBox(-title => "Export Complete", -default_button=>"EXPORT COMPLETE - Click to continue", -buttons=>["EXPORT COMPLETE - Click to continue"]);
	my $result = $complete->Show;
    $self->lblSysMess->configure(-text => "");
	$self->update;
	
}
sub exportGFF {    # exports both GFF file for visible features, as well as a fasta sequence for all relevant contigs (full contig!)

    my ($self, $MapObj) = @_;	# MapWindow may or may not be defined depending on whether it was called from
    							# the QueryScreen drop down menu, or the mapwindow pop-up menu
    my (@Maps, @Features);
    if (!$MapObj){
    	foreach my $MapRef(@{$self->MapWindows}){   # if called from QueryScreen, then make file for ALL open map windows
			push @Maps, ${$MapRef}[0];
    	}
	} else {push @Maps, $MapObj;                    # otherwise, just do the one window that is foremost (from the pop-up menu)
	}
	
    my $filename = "/";
    while (!($filename) || ($filename =~ /\/$/)){    # keep trying until there is a filename with no trailing slash
    	my $AskFile = $self->DialogBox(-title => 'Export to File:',
				   -default_button=> 'Export',
				   -buttons => [qw/Export Cancel/]);
    	$AskFile->add("LabEntry",
			-textvariable=>\$filename,
			-label=>'Enter the full PATH & FILENAME to export to:',
			)->pack;
		my $namedisp=$AskFile->Show;
    	$filename =~ s/\s//g;
    	
    	return if $namedisp eq 'Cancel';
    }



    foreach my $Map(@Maps){
	    $self->lblSysMess->configure(-text => "Contig GFF Export... (slow)");
	    $self->update;
    	open OUT, ">>$filename.gff" or sub {	my $error = $self->DialogBox(-title => "Invalid Path/File error",
								     -default_button => "OK",
								     -buttons => ["OK"]);
					my $result = $error->Show;
					return 0};

    	my $fh=\*OUT;

		$Map->MapSeq->export_GFF( $fh );

		print OUT "\n\n";
    # ==========  END OF EXPORT GFF
    # =============================
    	close OUT;
    	open OUT, ">>$filename.fasta" or sub {	my $error = $self->DialogBox(-title => "Invalid Path/File error",
								     -default_button => "OK",
								     -buttons => ["OK"]);
					my $result = $error->Show;
					return 0};


    # =============================
    # =====  BEGIN EXPORT FASTA

    	my $seq; # holds the actual sequence string - comes from database

    	#if ($MapSeq) {
    	my $MapSeq = $Map->MapSeq;  # Sequence object
    	
    	foreach my $contig(@{$MapSeq->get_contig_names}){   # all of the contigs which participate in this sequence
    	
			$self->lblSysMess->configure(-text => "Contig FASTA Export...");
            $self->update;
                    	
            $seq = $self->context->get_contig_sequence($contig);         # returns the sequence as a long string

            $seq =~ s/([^\n]{70})/$1\n/g;                # put newline every seventy characters
            print OUT ">$contig\n";                      # print fasta header & sequence after the relevant GFF lines
            print OUT $seq;
            print OUT "\n\n";
    	} # end of foreach my contig
    }  # end of foreach my map

    my $complete = $self->DialogBox(-title => "Export Complete", -default_button=>"EXPORT COMPLETE - Click to continue", -buttons=>["EXPORT COMPLETE - Click to continue"]);
    my $result = $complete->Show;
    $self->lblSysMess->configure(-text => "");
    $self->update;

	close OUT;
    return 1;
}
sub exportXML{}

=head2 deprecated - now expect EST's to be mapped via plugin or exist as features

sub mapEST{

    my ($self, $action, $MapWindow) = @_;
    #  $action is to map or to unmap the EST data
    my @MapWindows;

    if ($MapWindow) { push @MapWindows, $MapWindow; }
    # if it has been passed a map window reference then use it instead of running through them all
    else {@MapWindows = @{$self->MapWindows};    }

    my %est_sources;  # this is actually going to be a list, not a hash... but it is just easier this way
    if ($action eq "mapped") {
    	foreach my $MapRef (@MapWindows) {

	    my ($MapObj, $screen_start, $screen_stop, $base_contig)= @{$MapRef};

	    $self->lblSysMess->configure(-text => "Querying EST database...");
	    $self->lblSysMess->update;

	
	    foreach my $result($self->context->get_EST_hits($base_contig,$screen_start,$screen_stop)) {
			my ($ID, $start, $end, $subj_strand, $strand, $source) = @$result;
			$est_sources{$source} = undef;  # we just want a list of unique sources, and this is a quick and dirty way to do it
			$strand =~ tr/fr/+-/;
			$subj_strand =~ tr/fr/+-/;

			$self->lblSysMess->configure(-text => "Mapping $ID");
			$self->lblSysMess->update;

			my $feature=Bio::SeqFeature::Generic->new(-seqname=>"${base_contig}_${screen_start}_to_$screen_stop",
							  -source =>$source,
							  -primary=>"alignment".$strand,
							  -strand =>$subj_strand,
							  -start  =>$start,
							  -end    =>$end,
							 );
			$feature->add_tag_value('id', "NOT_IN_DB_$ID");
			$MapObj->mapFeatures('draft', [$feature]);
	    }
    	}
    } else {
	foreach my $MapRef (@MapWindows) {
	    foreach my $source(keys %est_sources){
	    	$MapRef->[0]->DraftCanvas->delete("Source $source");
	    }
	}
    }
    $self->lblSysMess->configure(-text => "");
    $self->lblSysMess->update;


}

=cut


sub parse_gff_string {
    my ($self,$string)=@_;

    my ($seq, $source, $primary, $start, $end, $score, $strand, $frame, $attribs) = split(/\t+/, $string);

    ####### parse comments ###### Mark's code except for %taghash
    my %taghash;
    $attribs =~ s/\#(.*)$//;				  # remove comments field (format:  #blah blah blah...  at the end of the GFF line)

    my @key_vals = split /;/, $attribs; # attributes are semicolon-delimited

    foreach my $pair ( @key_vals ) {
	my ($key, $values) = split /^\s*([\w\d]+)\s/, $pair; # separate the key from the value based on the = sign
	my @values;

	while ($values =~ s/"(.*?)"//) { # free text is quoted, so match each free-text block
	    if ($1) {
		push @values, $1;
	    }
	    # and push it on to the list of values (tags may have more than one value...)
	}

	my @othervals = split /\s+/, $values; # and what is left over should be space-separated non-free-text values
	foreach my $othervalue (@othervals) {
	    if (length($othervalue) > 0) {
		push @values, $othervalue;
	    }
	    # get rid of any empty strings which might result from the split
	}

	$taghash{$key} = \@values;

    }
    return ($seq, $source, $primary, $start, $end, $score, $strand, $frame,%taghash);
}

sub load_gff_string {

    my ($self,$string)=@_;
    my ($seq, $source, $primary, $start, $end, $score, $strand, $frame,%taghash)=$self->parse_gff_string($string);

    my $contig_id=$self->context->get_contig_id($seq);

	my ($gene, $annot, $feature);
	if (exists $taghash{gene_name}) {
	    my $gene_name=$taghash{gene_name};
	    delete $taghash{gene_name};
	    $gene=GQ::Server::Gene->new(context     =>$self->context,
					   lockid      =>'admin',
					   contig_id   =>$contig_id,
					   source_tag  =>$source,
					   primary_tag =>$primary,
					   name   	   =>$gene_name->[0],
					   contig_start=>$start,
					   contig_stop =>$end,
					   strand      =>$strand,
					   frame       =>$frame,
					   length      =>$end-$start+1,
					   tagvalues   =>\%taghash,
					  );
	}
	#elsif (exists $taghash{group_id}) {
	#    my $group_id=$taghash{group_id};
	#    delete $taghash{group_id};
	#    $annot=GQ::Server::Feature::Annotation->new(
	#						   context     =>$self->context,
	#						   lockid      =>'admin',
	#						   contig_id   =>$contig_id,
	#						   source_tag  =>$source,
	#						   primary_tag =>$primary,
	#						   group_id    =>$group_id->[0],
	#						   contig_start=>$start,
	#						   contig_stop =>$end,
	#						   score       =>$score,
	#						   strand      =>$strand,
	#						   frame       =>$frame,
	#						   length      =>$end-$start+1,
	#						   tagvalues   =>\%taghash,
	#						  );
	#}
	else {
	    ####### create Feature #######
	    $feature=GQ::Server::Feature->new(context     =>$self->context,
						 lockid      =>'admin',
						 contig_id   =>$contig_id,
						 source_tag  =>$source,
						 primary_tag =>$primary,
						 name		=> "Imported_From_Plugin_$source",
						 contig_start=>$start,
						 contig_stop =>$end,
						 score       =>$score,
						 strand      =>$strand,
						 frame       =>$frame,
						 length      =>$end-$start+1,
						 tagvalues   =>\%taghash,
						);
	}

    if ($gene) {return $gene}
    elsif ($annot){return $annot}
    elsif ($feature){return $feature}
    else {return 0}
}


sub importSeq {
    my ($self) = @_;
    my ($name,$length);
    my $dialog = $self->Dialog(-text => "Are you defining a sub-sequence of an existing database sequence as a new sequence entity,\nor importing a new FASTA file into the db?",
			       -title => 'Define or Import Sequence',
			       -default_button=> 'Import',
			       -buttons => ["Define Subsequence", "Import New", "Cancel"]);
    my $subseq=$dialog->Show;
    return if $subseq eq "Cancel";
    if ($subseq eq 'Import New'){

	my $FE = $self->FileSelect(	-directory => $self->WORKING_DIR,
					-width => 35,
					-filelabel => "Select a FASTA-formatted file to import",
					-filelistlabel => "Files",
					-dirlistlabel => "Directories",
					-dirlabel => "Select");
	my $File = $FE->Show;
	open IN,$File or return;
	my ($header,$seq);
	while (<IN>) {
	    if (/^>/) {
		if ($seq && $header) {
		    _create($File,$seq,$name,$header,$self);
		    $seq = '';
		}
		$header=$_; $name = ($header =~ /^>(\w+)/ && $1);
		my $AskName = $self->DialogBox(-title => 'Sequence Name',
					       -default_button=> 'Save',
					       -buttons => [qw/Save Cancel/]);
		$AskName->add("Label",
			      -text=>"Fasta Header: $header",
			     )->pack;
		$AskName->add("LabEntry",
			      -textvariable=>\$name,
			      -label=>'Enter the unique designation for this sequence:',
			     )->pack;
		my $namedisp=$AskName->Show;
		return if $namedisp eq 'Cancel';

	    $self->{contiglist}->insert(1, $name);
		
	    } else {
		chomp;
		$seq .= $_;
	    }  # end of while <IN>
	}  # end of if ($subseq eq 'no')
	_create($File,$seq,$name,$header,$self);

    } else { #$subseq eq 'Yes' - this sequence is merely a special substring of a present sequence
	my $AskName = $self->DialogBox(-title => 'Designation of Sub-sequence',
				    -default_button=> 'Save',
					-buttons => [qw/Save Cancel/]);
	$AskName->add("LabEntry",
				-textvariable=>\$name,
				-label=>'Enter a unique name for the Subsequence you are about to create:',
				)->pack;
	my $namedisp=$AskName->Show;
	return if $namedisp eq 'Cancel';

	my ($start,$parent);
	my $SubSeqDiag= $self->DialogBox(-title => 'Subsequence Details',
				-default_button=> 'Save',
				-buttons => [qw/Save Cancel/]
				);
	$SubSeqDiag->add("Label",
				text => 'Please choose the parent sequence from the list:'
				)->pack;
	my $be=$SubSeqDiag->add("BrowseEntry",
				-variable=>\$parent,
				)->pack;
	my $seq=$self->context->get_contig_info_by_name;
	foreach (sort keys %$seq) {
	    $be->insert("end",$_);
	}

	$SubSeqDiag->add("LabEntry",
			-textvariable=>\$start,
			-label=>'On which nucleotide of the parent sequence does your sequence start?',
			)->pack;
	$SubSeqDiag->add("LabEntry",
			-textvariable=>\$length,
			-label=>"What is your subsequence's length? ",
			)->pack;
	my $continue=$SubSeqDiag->Show;
	return if $continue eq 'Cancel';
	my $orgid=$self->context->organism->id;

	create_subsequence($self,$name,$seq,$parent,$start,$length);

	$self->{contiglist}->insert(1, $name);
    }

    # we need to destroy and re-create the genome mapper object
    # this is horrible, and eventually GenomeMapper will be able to
    # simply re-draw and extend chromosomes, but
    # this is the only way for now.
    #
    # in addition, at some point "$self" will actually be the
    # GenomeMapper object itself, so this routine will be removed.
    #my @MainWindows = @{$self->{MainWindows}};
	#foreach my $Window(@MainWindows){
	#	if (ref($Window) =~ /GenomeMapper/){
	#		$Window->destroy;
	#	}
	#}
	$self->createChromosomeDisplays;


}
sub _create {
    my ($File,$seq,$name,$header,$self)=@_;
    my ($start,$parent,$chr_id);
    my $SubSeqDiag= $self->DialogBox(-title => 'Subsequence Details',
				    -default_button=> 'Save',
				    -buttons => [qw/Extend Standalone Cancel/]
				    );
    $SubSeqDiag->add("Label",
		    text => "Sequences can only be extended from 'left' to 'right' (i.e. 5' to 3').\n\nPlease choose the parent (5'-overlapping) sequence from the list and click [Extend]\n OR \nclick [Standalone] to indicate that this sequence does not overlap with other database sequences:\n______________________________________________________\n\nSequence List: (for Extension only)"
		    )->pack;
    my $be=$SubSeqDiag->add("BrowseEntry",
			    -variable=>\$parent,
				)->pack;
    my $seqs=$self->context->get_contig_info_by_name;
    foreach (sort keys %$seqs) {
	$be->insert("end",$_);
    }
    my ($dominate);
    $SubSeqDiag->add("LabEntry",
		    -textvariable=>\$chr_id,
		    -label=>'If you know which chromosome your sequence belongs to, please enter it here: (Extension or Standalone) ',
		    )->pack;
    $SubSeqDiag->add("LabEntry",
		    -textvariable=>\$start,
		    -label=>'On which nucleotide of the parent sequence does your sequence start? (for Extension only)',
		    )->pack;
    $SubSeqDiag->add("Checkbutton",
		    -variable=>\$dominate,
		    -text=>'This sequence takes priority over the parent sequence (for Extension only)',
		    )->pack;
    my $continue=$SubSeqDiag->Show;
    $continue eq 'Cancel' && return ||
      $continue eq 'Standalone' && create_sequence($seq,$name,$self->context,$chr_id) ||
	create_sequence($seq,$name,$self->context,$chr_id,$dominate,$seqs->{$parent}[0],$start);
    $seq = '';
}
sub Annotate {

        my ($self, $MapObj) = @_;
        my $FeatureHashRef = $MapObj->getSelectedFeatures;
        return unless %$FeatureHashRef;
        my $CurrentComments;
        my $Feature;
        my @keys = (keys %{$FeatureHashRef});
        return if ($#keys > 0);    # can only annotate one feature at a time with the big window

	my $key = shift @keys;
	$Feature = ${$FeatureHashRef}{$key}; # get the feature

	my $AnnotationWindow = $self->Toplevel(-title => "Hand Annotation Window");
	push @{$self->MainWindows}, $AnnotationWindow;
	my $txtHandAnnotate = GQ::Client::txtHandAnnotate->new($AnnotationWindow, $self, $Feature);
}

sub selectCommonExons {
    my($self, $MapObj)=@_;
    my $c = $MapObj->DraftCanvas;
    my ($FeatureID, $strand, $source_tag, $type, $map, $DB_ID) = $MapObj->getSelectedTags;
    if (!$DB_ID){$self->lblSysMess->configure(-text => "no exon was selected for comparison...");$self->update;return}

    my (@Selections);
    #push @Selections, "DB_ID $DB_ID";   # keep the original one too!! (this is now dealt with in the lookup table itself)
    foreach my $NewID($self->context->get_common_exons($DB_ID)) {
    	push @Selections, "DB_ID $NewID";
    }
    $MapObj->selectWithTag(\@Selections, 'draft');

    if ($#Selections == 0){$self->lblSysMess->configure(-text => "This entry either has no Blast data or shares no homology"); $self->update;}
}

sub selectsubFeatures {
    my ($self, $MapObj, $Feature)=@_;
    #    my $Feature = $MapObj->getSelectedFeatures;
    my $this;
    if (ref($Feature) eq 'HASH') {
    	foreach (values %$Feature) {
    	    $this=$_;
    	}
    } else {
    	$this=$Feature;
    }
    $MapObj->clearSelections;
    map {$MapObj->selectWithTag(["DB_ID ".$_->id], 'finished') } $this->sub_SeqFeature;
#     if ($this->can("features")) {
# 		foreach (values %{$this->features}) {
# 	    	my $id=$_->id;
# 	    	$MapObj->selectWithTag(["DB_ID $id"], 'finished');
# 		} }
}

sub delete_from_database {
    my ($self, $MapObj, $Feature)=@_;
    return unless %$Feature;
    my $not_allowed=0;
    my (%not_allowed,%Genelist);
    my $FID1=[keys %$Feature];
    my %Featurelist=%$Feature;

    my @Sortedlist;
    foreach my $typevar qw(Gene Transcript) {           #we are going to sort these features into bins
	foreach (keys %Featurelist) {
	    if ($Featurelist{$_}->type eq $typevar) {     #genes first, then transcripts
		push @Sortedlist,$Featurelist{$_};
		delete $Featurelist{$_};                  #deleting as we go
	    }
	}
    }
    push @Sortedlist, values %Featurelist;                #then everything else
    
    foreach (reverse @Sortedlist) {                       #now delete from the bottom up
	print join "\t",($_->type,$_->contig_start,$_->length),"\n";
    	if (my $gene=$_->parent) { #it's a feature of a gene, then
    	    $Genelist{$gene->id} = $gene;  #store for later
    	    $gene = $gene->remove_feature($_);
	    #remove feature returns the gene, and if the gene has no exons left, returns undef
    	} elsif ($_->delete_from_db) { #it's a gene, then - or is it?
	                                             #check for successful deletion -
        	                                     #(failure if trying to delete something from the feature table)
    	} else {                                     #unsuccessful deletion
    	    $not_allowed{$_->id}++;                             #mark for not unmapping
    	    $not_allowed++;                                 #count unsuccessful attempts
    	}
    }
    my @FIDlist=@$FID1;                     #FID numbers from the original selection
    if ($not_allowed) {
#	print "You tried unsuccessfully to delete $not_allowed feature".($not_allowed>1?"s":"").".\n";
#	print "The database may be corrupt.\n";
	for (my $i=0; $i<=$#FIDlist; $i++) {
	    if ($not_allowed{$FIDlist[$i]}) {
		$FIDlist[$i]='';                         #blank that entry so that it is not unmapped
	    }
	}
    }
    $MapObj->unmapFeatures(\@FIDlist);
    $MapObj->clearSelections;
    $MapObj->selectWithTag(["Source gene"]);
    my $genelist2=$MapObj->getSelectedIDs;
    #print "All Genes ".join "\t",@$genelist2,"\n";
    $MapObj->clearSelections;
    #print "Bad Genes ".join "\t",(keys %Genelist),"\n";
    my @unmap_list;
    foreach (keys %Genelist) {                           ###BACK TO GENES###
		push @unmap_list, "DB_ID $_";
    }
    $MapObj->selectWithTag(\@unmap_list);
    my $gidlist=$MapObj->getSelectedIDs;  #now we know which widgets the genes are
    #print "Unmapping ".join "\t",@$gidlist,"\n";
    $MapObj->unmapFeatures($gidlist);     #so we unmap them
    my @glist=values %Genelist;
    for (my $i=0; $i<=$#glist; $i++) {
	#print join "\t",@glist, "\n";
	unless ($glist[$i]) {
	    splice @glist, $i, 1;   #If there is no gene there, don't map it.
	}
    }
    if (@glist) {             # and if there is anything left to put back,
	$MapObj->mapFeatures('finished',\@glist);   #we put it back
    }
    $MapObj->clearSelections;
}

sub getBlastHit {
	# send it the feature hash ref of {FID}=$Feature;
    my ($self,$Features) = @_;
    my @results;

    foreach my $feature (values %$Features) {
	if ($feature->can('getBlastHits')) {
	    foreach my $entry(@{$feature->getBlastHits}) {
		my ($gi,$prob,$desc)=@$entry;
		my $output = "\cINCBI $gi \cI $prob \cI $desc \n\n";
		push @results, $output;
	    }
	}
    }
    return @results;
}

sub reBlastExon {
    my ($self, $MapObj)=@_;
    my $c = $MapObj->DraftCanvas;
    $self->lblSysMess->configure(-text => "Beginning BLAST Search"); $self->update;

    my $FeatureHash = $MapObj->getFeaturesWithTag($MapObj->getSelectedIDs);  # get the selected features
    foreach my $feature(values %{$FeatureHash}){
        my $start = $feature->start;         # get relevant feature information
        my $stop = $feature->end;
        my $DB_ID = $feature->id;
        my $strand = $feature->strand;
        my $source = $feature->source_tag;

        $self->lblSysMess->configure(-text => "Beginning BLAST Search"); $self->update;

        my $seq = $MapObj->MapSeq->subseq($start, $stop);  # get the sequence for blasting

		if ($strand eq "-1") {$seq =~ tr/ATCG/TAGC/; $seq = scalar reverse $seq;}    # reverse complement

		my $BlastResultObj = GQ::Client::Utilities::CGI_blast($self, $DB_ID, $seq, $source);    # BioPerl Blast Object
		if ($BlastResultObj){

				$self->lblSysMess->configure(-text => "BLAST complete - parsing into database..."); $self->update;

				$self->context->parse_Blast_to_db($DB_ID,$BlastResultObj);

				$self->lblSysMess->configure(-text => "Blast Done - moving to next"); $self->update;
		} 
		else {
				$self->lblSysMess->configure(-text => "BLAST FAILED - please check your Blast configuration");
				$self->update;
				return
		}
    }
    $self->lblSysMess->configure(-text => "Blast Parsing Complete - Returning control to User"); $self->update;
} #end of sub reBlastExons

sub printTextOut {
    my ($self, $textref, $mode, @tags) = @_;
    my @text = @{$textref};
    my $txtOutput = $self->TextOutput;
    if ($txtOutput) {
	if ($mode eq "wipe") { $txtOutput->delete("1.0", "end") }
	foreach my $line (@text){
	    $txtOutput->insert('end', "$line\n", \@tags);
	}
    }
}

sub Text_to_Printer {
	my ($self) = @_;
	my $textOutput = $self->TextOutput;
	if ($textOutput){my $text = $textOutput->get("1.0", "end");
		open OUT, ">text_to_print";
		print OUT $text;
		close OUT;
		system `more text_to_print | lpr`;
		unlink "text_to_print";
		return 1;
	} else { return 0}
}


sub Canvas_to_Printer {
	
	my ($self) = @_;
    my @Maps;
    foreach my $MapRef(@{$self->MapWindows}){
		push @Maps, ${$MapRef}[0];
    }

    foreach my $Map(@Maps) {
		my $ps = $Map->DraftCanvas->postscript( '-x' => 0,
                                       '-y' => 0,
                                       -width => $Map->DraftCanvas->Width,
                                      -height => $Map->DraftCanvas->Height);
		open (PS, "| lpr"); # customize with -Pname e.g.
		print PS $ps;
		close (PS);
	}
}

sub findPartnerFeatures {
	# what we need to do is go through each of the features to
	# determine if there are other features on the canvas
	# which are part of the same... "bit" (eg. an EST may span several
	# exons, and this EST will appear as multiple features)
	# these need to be added to the list of selected features.
    my ($self, $MapObj) = @_;
    # **********************************************************
    # ************  THIS ROUTINE DOES NOT FUNCTION PROPERLY YET!
    # **********************************************************

    my %FeaturesHash = %{$MapObj->getSelectedFeatures};
    my %newFeaturesHash;
    foreach my $featureID(keys %FeaturesHash){
    	my $feature = $FeaturesHash{$featureID};
    	if ($feature->can("id")){  # SeqFeature Similarity (from blast call) can not id.
	    if ($feature->id =~ /(NOT_IN_DB_.*)/){
    		my %PartnerFeatures = %{$MapObj->getFeaturesWithTag([$1])};
    		%newFeaturesHash = (%newFeaturesHash,%PartnerFeatures);  # merge the results
	    } else {
    		$newFeaturesHash{$featureID} = $feature;  # add just this one feature
	    }
    	}
    }
    # %newFeaturesHash should now contain the selected features as well as their partners
    # so now we need to re-select the canvas widgets to reflect this
    # $MapObj->clearSelections;  # erase current
    my @FIDs = (keys %newFeaturesHash);
    #print "@FIDs\n";
    $MapObj->selectFeatures(\@FIDs);
}
sub writeWidgetTags {
	my ($self, $MapObj) = @_;
	my ($FID, $strand, $source, $type, $canvas, $DB_ID) = $MapObj->getSelectedTags;
	return if (!$canvas);
	$self->printTextOut(["\n\t\tFeatureID $FID
	                     \n\t\tStrand $strand
	                     \n\t\tSource $source
	                     \n\t\tType   $type"], 'wipe');
	
	
}

sub writeTags {
    my ($self, $MapObj) = @_;
    my $FeaturesRef = $MapObj->getSelectedFeatures;

    my ($sortedIDs, $sortedFeatures) = sort_by_start_features($MapObj, $FeaturesRef);
	if (scalar @{$sortedIDs} == 0){ # this is a test to see if the selected widget is a database widget, or a post-mapped widget (like a Blast hit, or GFF overlay, or EST alignment)
    	$self->writeWidgetTags($MapObj);	# if it is, then simply write the basic widget tags and move on.
    	return;
    }

    my @FeatureIDs = @{$sortedIDs};   # at this point we know it is a complex Seqcanvas feature, so lets rip it apart and get the goodies!
    my @Features = @{$sortedFeatures};

    $self->printTextOut([], "wipe");

    my $count=0;
    my ($subfeatures_output,$additional_output,$score);
    foreach my $feature(@Features) {
		my $key = $FeatureIDs[$count++];
		$self->printTextOut([$key]);
		if ($feature->can("features")) {  # beginning of subfeatures loop
	    	$subfeatures_output .= "\nSub-features:\n";
	    	my @subfeatures = $feature->sub_SeqFeature;   	# these are the sub-features of a gene object
	    	foreach my $subfeature (@subfeatures){
				$subfeatures_output .= " \cI " . $subfeature->source_tag . " " . $subfeature->primary_tag . "   ".$subfeature->type."  score: " . $subfeature->score . "\n";
				$subfeatures_output .= " \cI \cI DatabaseID : " . $subfeature->id . "\n";
				$subfeatures_output .= " \cI \cI length     : " . $subfeature->length . "\n";
				foreach my $tag($subfeature->all_tags) {
		    		next if ($tag eq "group_id"); # we already know this... it is the gene name
		    		next if ($tag =~ "GO_annotation"); 
		    		$subfeatures_output .= "\cI\cI $tag " . join " \cI ", $subfeature->each_tag_value($tag);
				}
				unless ($subfeature->GO && @{$subfeature->GO}[0]){$subfeature->GO_find};
				foreach my $GO (@{$subfeature->GO}){
					my $id = $GO->GO_id;
					my $Term = $GO->term;
					my %evid = %{$GO->evidence};
					chomp $id; chomp $Term;
					$subfeatures_output .= "\n\cI\cI\cI $id  $Term\n";  # write it as tag=value into the window and add a Tk-tag to the entire piece of text
					foreach my $code(keys %evid){
					chomp $code;
					$subfeatures_output .= "\cI\cI\cI\cI Evidence Type: $code\n";
        				foreach my $ref(@{$evid{$code}}){
							chomp $ref;
							$subfeatures_output .= "\cI\cI\cI\cI\cI Reference: $ref\n\n";
						}
					}
				}
	    	}
		} # end of subfeatures
	
	    $additional_output = "";
		foreach my $tag( $feature->all_tags) {
		    next if ($tag =~ "GO_annotation"); 
	    	$additional_output .= "\n\cI$tag\cI " . join " \cI ", $feature->each_tag_value($tag);
		}
		
		my $color = ${$MapObj->current_colors}{$feature->source_tag};
    	if (!$color){$color = "#000000"}
		if ($feature->can('score')) {
	    	$score = ($feature->score) ? '('.$feature->score.')' : '';  # if there is no score tag then leave this out.
		}
    	
    	$self->printTextOut(["                    ".$feature->seqname]);
    	$self->printTextOut(["start: ".$feature->contig_start." |----------------------------------------->len.".$feature->length." nt"], "", $feature->source_tag);
    	$self->printTextOut(["                    ".
							 $feature->source_tag." ".$feature->primary_tag."   ".$feature->type."  score $score\n"]);
    	$self->TextOutput->tagConfigure($feature->source_tag, -foreground => $color);
    	if ($subfeatures_output){
	    $self->printTextOut([$subfeatures_output]);
	    $subfeatures_output='';
	}
    	if ($additional_output){
	    $self->printTextOut([$additional_output, "\n"]);
	    $additional_output='';
	}

    	unless ($feature->GO && @{$feature->GO}[0]){$feature->GO_find};
    	foreach my $GO (@{$feature->GO}) {
    		my $id = $GO->GO_id;
    		my $Term = $GO->term;
    		my %evid = %{$GO->evidence};
    		chomp $id; chomp $Term;
    		$self->printTextOut(["\cI$id  $Term"]);  # write it as tag=value into the window and add a Tk-tag to the entire piece of text
    		foreach my $code(keys %evid){
    			chomp $code;
    			$self->printTextOut(["\cI \cI Evidence Type: $code"]);
            	foreach my $ref(@{$evid{$code}}){
					chomp $ref;
					$self->printTextOut(["\cI \cI \cI Reference: $ref"]);
				}
				$self->printTextOut(["\n"]);
			}
    	}

    } #end of foreach my $features

}

sub lightupDups {

    my ($self, $MapObj) = @_;
    my $FeatureHashRef = $MapObj->getSelectedFeatures;
    my @features = values(%{$FeatureHashRef});
    my @GenomeMaps = @{$self->{MainWindows}};
	foreach my $GenomeMap(@GenomeMaps){
		if (ref($GenomeMap) =~ /GenomeMapper/){
			$GenomeMap->dimAndRemoveTag("dupl");  # erase last set
    		foreach my $feature(@features){
				next if ($feature->primary_tag ne "dupl");
				# here is where we find the duplicated item and light up the contig on the map
				foreach my $contig($feature->find_duplicate_tags) {
	    			#print "this contig is |$contig|\n";
	    			$GenomeMap->lightAndAddTag("darkgreen", "dupl", [$contig]);
				}
			}
    	}
   }

}
sub selectORFExons {    # this is incomplete and non-functional at the moment
						# the intention is to computationally find ALL valid ORF combinations of
						# selected exons...
	my ($self, $MapObj) = @_;
	$self->lblSysMess->configure(-text => "");$self->update;
	my $FeatureIDsRef = $MapObj->getSelectedIDs;	
	my @FeatureIDs = @{$FeatureIDsRef};
	my $FeaturesRef = $MapObj->getSelectedFeatures;     # get all selected stuff
	my %FeatureHash = %{$FeaturesRef};


}

sub plugin_getContigIDs {
	my ($self) = @_;
	my $Genome = $self->GenomeMap;
	my $ContigListHash = $Genome->contigList;
	my @Contigs = (keys %{$ContigListHash});
	return @Contigs;
}

sub plugin_getNonRedundantContigSequence {
	my ($self, $name) = @_;
	my $seq = getNonRedundantContigSequence($self, $name);  # Admin::createBlastDB
	return $seq;
}

sub getFullContigSequence {
	my ($self, $contigname) = @_;
	my $contigobj=$self->context->contig($contigname);
	my $thisseq = $contigobj->sequence;
	$thisseq =~ s/\W//g;
	return $thisseq;
}
	
sub plugin_getOrganismName {
	my ($self) = @_;
	return $self->context->organism->latin;
}

sub plugin_getSequenceVersion {
	my ($self) = @_;
	return $self->context->version;
}

	

sub Quit {
    my ($self)=@_;
    #foreach (@{$self->MainWindows}) {
	#$_->destroy if Tk::Exists($_);
    #}
    eval {$self->destroy};
}


1;
