package GQ::Server::DB::Adaptor::SequenceAdaptor;
use strict;
use vars qw(@ISA $AUTOLOAD);
use Carp;
use DBI;
use GQ::Root;
use GQ::Server::DB::Adaptor::ObjectAdaptor;
@ISA=qw(GQ::Server::DB::Adaptor::ObjectAdaptor);


{
#Encapsulated class data

    GQ::Root->create();

}

=head2 where_am_i

 Title   : where_am_i
 Usage   : my ($Contig_id,$contig_loc)=$self->adaptor->where_am_i($self,$Sequence_loc);
 Function: given a location relative to a given sequence object, return the contig id and location in contig
           coordinates
 Returns : a list of two elements: a contig_id and a contig_location
 Args    : the sequence object, and the location on that sequence object


=cut

sub where_am_i {
    my ($self,$feat,$loc)=@_;
    my $abs_loc = $loc + $feat->start + $feat->bc_start - 2;
    my ($version,$assembly)=($feat->context->version,$feat->assembly);
    my $sth=$self->dbh->prepare("SELECT Tiling_Path.contig_id,abs_start FROM Tiling_Path,ContigAssembly ".
				"WHERE $abs_loc BETWEEN abs_start+VC_start-1 and abs_start+VC_start-1+VC_length-1 ".
				"AND version=? and assembly=? and ContigAssembly.id=Tiling_Path.contig_id");
    $sth->execute($version,$assembly);
    my ($VC_id,$abs_start)=$sth->fetchrow_array;
    my $contig_loc = $abs_loc - $abs_start + 1;
    return ($VC_id,$contig_loc);
}

=head2 get_contig_names

 Title   : get_contig_names
 Usage   : my @list=$self->adaptor->get_contig_names($self);
 Function: get the list of contig names whose virtual contigs overlap this sequence
 Example : foreach my $contig(@{$MapSeq->get_contig_names}){   # all of the contigs which participate in this sequence
 Returns : a list of contig names
 Args    : the sequence object


=cut

sub get_contig_names {
    my ($self,$feat)=@_;
    my @list;
    my $start=$feat->start + $feat->bc_start - 1;
    my $stop =$start + $feat->length -1;
    my $sth=$self->dbh->prepare("SELECT name FROM Tiling_Path,ContigAssembly,Contig WHERE ".
				"Contig.id=ContigAssembly.contig_id and ContigAssembly.id=Tiling_Path.contig_id AND ".
				"abs_start+VC_start-1<? AND abs_start+VC_start+VC_length-2>? AND version=? AND assembly=?");
    $sth->execute($stop,$start,$feat->context->version,$feat->assembly);
    while (my ($name)=$sth->fetchrow_array) {
	push @list,$name;
    }
    return \@list;
}

=head2 getseq

 Title   : getseq
 Usage   : my $seq=$self->adaptor->getseq($self,$start,$end)
 Example : my $feat_seq=$MapSeq->adaptor->getseq($MapSeq,$feat->start,$feat->end)
 Returns : a dna sequence corresponding to either the whole sequence, or the desired substring
 Args    : the sequence object, and optionally the start and end values of the desired sequence
           with no start/end values, the sequence of the whole sequence is returned


=cut

sub getseq {
    my ($self,$feat,$start,$end)=@_;
    my ($startpos,$length);
    if ($start) {                      #get start,stop relative to loaded sequence
	$startpos=$start + $feat->start - 1;
	$length=$end-$start+1;
    } else {
	$startpos=$feat->start;
	$length=$feat->length;
    }
    if (my $seq=$feat->SeqHash($startpos,$length)) {
	return $seq;
    }
		    ##### find the reference start position for the sequence
            ##### and get the appropriate Virtual Contigs
    my $sth=$self->dbh->prepare
      (
       "SELECT Contig.name,GREATEST(B.abs_start+?-A.abs_start,A.VC_start),".
       "LEAST(A.VC_start+A.VC_length-1,B.abs_start+?+?-1-A.abs_start) ".
       "FROM Tiling_Path A, Tiling_Path B, ContigAssembly ConA, ".
       "ContigAssembly ConB, Contig ".
       "WHERE B.contig_id=? AND ConA.version=ConB.version AND ".
       "ConA.assembly=ConB.assembly AND ".
       "ConA.id=A.contig_id AND ConA.contig_id=Contig.id AND ConB.id=B.contig_id AND ".
       "A.abs_start+A.VC_start-1<=?+B.abs_start-1+?-1 AND ".
       "A.abs_start+A.VC_start-1+A.VC_length-1>=?+B.abs_start-1 ".
       "ORDER BY A.abs_start"
      ) or die "Problem preparing: $DBI::errstr";


    $sth->execute($startpos,$startpos,$length,$feat->base_contig_id,$startpos,$length,$startpos);
    my $seq;
    while (my ($contig,$local_start,$last_bp)=$sth->fetchrow_array) {
	my $sublength=$last_bp-$local_start+1;
	my $contigobj=$self->context->contig($contig);  #fetch the right contig object
	$seq .= substr($contigobj->sequence,$local_start-1,$sublength);   #now take out the appropriate substrings
    }                                                     # note -1 to get to 0-based perl substr
    $feat->SeqHash($startpos,$length,$seq);
    return $seq;
}

=head2 _features

 Title   : _features
 Usage   : $seq->adaptor->_features($seq);
 Function: load all the relevant features from the database into the sequence
 Returns : nothing - the sequence is fully loaded, however
 Args    : the sequence object


=cut

sub _features {
    my ($self,$fea)=@_;
    my $start=$fea->start + $fea->bc_start - 1;
    my $stop =$start + $fea->length - 1;
    $fea->{features}=$self->load($fea,
				 context=>$fea->context,
				 assembly=>$fea->assembly,
				 lockid=>$fea->lockid,
				 par_pos=>$start,
				 parent=>$fea,
				 start=>['<',$stop],
				 stop =>['>',$start],
				);
    $self->assemble($fea);
}

=head2 load

 Title   : load
 Usage   : $self->load(%args)
 Function: load the sequence features into memory
 Returns : a reference to the hash of features
 Args    : a hash containing
            context  => the context object
            assembly => the assembly id
            lockid   => the current lock id
            par_pos  => the start of the parent sequence, in assembly coordinates
            parent   => the sequence object
            %args which will be passed to _make_sql (see documentation for _make_sql)

=cut

sub load {
    my ($self,$feat,%args)=@_;
    my ($context,$assembly,$lockid,$par_pos,$parent)=@args{'context','assembly','lockid','par_pos','parent'};
    foreach (qw(context assembly lockid par_pos parent)) {  #args has to be passed to _make_sql
	delete $args{$_};                                   #without these parameters
    }
    my %list;
    $args{assembly}=[$context->version,$assembly];

    my $sql=$self->_make_sql(keys =>[GQ::Server::Feature->_db_keys],
			     %args);
    my $sth=$self->dbh->prepare($sql);
    $sth->execute or die "Problem executing dynamic SQL Query:\n$sql\n$DBI::errstr";
    while (my $data=$sth->fetchrow_hashref) {
	next unless $self->context->_type($data->{type});
	my $newfeat=$self->context->_type($data->{type})->new(%$data,
							      lockid    => $lockid,
							      context   => $context,
							      par_pos   => $par_pos,
							     );
	$newfeat->attach_seq($parent);
	if ($newfeat && ref($newfeat) eq $self->context->_type($data->{type})) {
	    $list{$data->{id}}=$newfeat;
	}
    }
    return \%list;
}


=head2 _make_sql

 Title   : _make_sql
 Usage   : my $sql_text = $self->_make_sql(keys =>[$feat->_db_keys],
					   %args);
 Function: create the appropriate sql to load the features from the table that are
           within the parent sequence boundaries.
 Returns : an sql string which can be prepared and executed
 Args    : a hash with the following structure:
             key=>[list of parameters]
           the key corresponds to start, stop, assembly or one of the column names of the source table.
           the list of parameters may include:
             only one numeric value - find features where key=value
             two numeric values - find features where key BETWEEN value1 and value2
             a text string - find features where key LIKE 'value'
             'or', followed by a listref of possible values - find features that match any of the values
             one of <, >, or =, followed by a numeric value - find features that have key [<>=] value2

=cut


sub _make_sql {
    my ($self,%args)=@_;
    my $table='Feature';
    my $keys=$args{keys};
    my ($flag1,$flag2)=(0,0);  #so the where clause doesn't include name=Tiling_Path.name twice
    my $where_clause='';
    foreach my $key (keys %args) {
	next if $key =~/^keys$/;
	$where_clause .= ($where_clause)? " AND $key ": "WHERE $key ";
	if ($key =~ /start|stop/) {
	    unless ($flag1) {
		$where_clause =~ s/$key/$table.contig_id=Tiling_Path.contig_id AND $key/;
		$where_clause =~ s/WHERE/WHERE contig_start BETWEEN VC_start and (VC_start+VC_length-1) AND/;
		$flag1++;
	    }
	} elsif ($key =~ /assembly/) {
	    my $version=$args{$key}[0];
	    my $assembly=$args{$key}[1];
	    $where_clause =~ s/$key/$table.contig_id=ContigAssembly.contig_id and assembly=$assembly and version=$version/;
	    $flag2++;
	    next;
	}
	my $first= (defined $args{$key}[0])? $args{$key}[0]: undef;
	my $second=(defined $args{$key}[1])? $args{$key}[1]: undef;
	if (defined $second && $first =~ /[<>=]/) {
	    $where_clause .= "$first $second";
	} elsif ($first eq 'or') {
	    my $clause;
	    $clause .= join ' OR ',map {"$key = '$_'"} @$second;
#	    foreach my $crit(@$second) {
#		$clause .= "$_ = '$crit' OR "
#	    }
#	    for (1..3) {chop $clause}
	    $where_clause =~ s/ $key/ ( $clause )/;
	} elsif ($first =~ /\D/) {
	    $where_clause .= "LIKE '$first'";
	} elsif ($second) {
	    $where_clause .= "BETWEEN $first AND $second";
	} else {
	    $where_clause .= "= $first";
	}
    }
    $where_clause =~ s/ start/ contig_start+abs_start-1/;
    $where_clause =~ s/ stop/ contig_stop+abs_start-1/;
    $where_clause =~ s/chr_id/Contig.chr_id/g;
    my $attrs=join ',',@$keys;
    $attrs =~ s/,/,$table\./g;
    $attrs = "$table.".$attrs;
    $table .=",Tiling_Path,ContigAssembly" if ($flag1 || $flag2);
    #FeatureType addition to SQL
    $attrs .= ',FeatureType.type';
    $table .=",FeatureType";      #to get feature type information
    $where_clause .= ' AND Feature.type = FeatureType.id ';
    return "SELECT $attrs FROM $table $where_clause";
}

=head2 assemble

 Title   : assemble
 Usage   : $self->assemble($seq);
 Function: assemble the feature hash into a structured set of genes and transcripts
 Returns : nothing
 Args    : the sequence object this is to apply to, which must have a features hash


=cut

sub assemble {
    my ($self,$seq)=@_;
    my %container;
    my $genetype=$self->context->_type('Gene');             #get the class name for genes
    my $transcripttype=$self->context->_type('Transcript');     #and transcripts

    $container{Gene}=[grep {ref($_) eq $genetype} values %{$seq->{features}}];     #find the genes
    $container{Transcript}=[grep {ref($_) eq $transcripttype} values %{$seq->{features}}]; #find the transcripts

    foreach my $type qw(Transcript Gene) {  #order is important here
	my @to_delete;
	my %hash=$self->find_elements(@{$container{$type}});  #go get the corresponding feature ids
	foreach my $parentid (keys %hash) {                      #the list of containers (genes/transcripts)
	    foreach my $feat (@{$hash{$parentid}}) {                #the list of features (transcripts/exons/etc)
		unless ($seq->{features}{$feat} && $seq->{features}{$feat}->isa('GQ::Server::GenericFeature')) {
		    delete $seq->{features}{$feat};  #defeat auto-vivification
		    next;                            #don't add to structure
		}
		if (ref($seq->{features}{$feat}) ne $self->context->_type('Transcript')) {
		    $seq->{features}{$parentid}{features}{$feat}=$seq->{features}{$feat};  #attach feature to parent
		} else {
		    $seq->{features}{$parentid}{transcripts}{$feat}=$seq->{features}{$feat};  #attach transcript to parent
		}
		push @to_delete, $feat;                                      #MARK HASH ELEMENT FOR DELETION
	    }
	    $seq->{features}{$parentid}->flesh_out_details
	      if $seq->{features}{$parentid};  #this updates start/stop/length info for graphics
	}
	map {delete $seq->{features}{$_}} @to_delete;                    #NOW REMOVE THEM
    }
}

=head2 find_elements

 Title   : find_elements
 Usage   : %hash = $self->find_elements(@list_of_gene_objects);
 Returns : a hash keyed by the ids of the members of the @list, whose values are a list of
           feature ids that are elements of the parent (i.e. transcripts of the gene)
 Args    : @list: a list of parent feature objects that need to have their elements assembled


=cut


sub find_elements {
    my ($self,@list)=@_;
    my %hash;
    my $textlist=join "','", map {$_->id} @list;
    $textlist="'".$textlist."'";
    my $sth=$self->dbh->prepare("SELECT parent,element FROM Container WHERE parent in ($textlist)");
    $sth->execute;
    while (my ($parent,$element)=$sth->fetchrow_array) {
	push @{$hash{$parent}},$element;
    }
    return %hash;
}

1;

