# $Id: GenericFeatureAdaptor.pm,v 1.5 2002/01/24 17:59:08 markw Exp $
#
# Genquire module for GQ::Server::DB::Adaptor::GenericFeatureAdaptor
#
# Cared for by David Block <dblock@gene.pbi.nrc.ca>
#
# Copyright David Block
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

GQ::Server::DB::Adaptor::GenericFeatureAdaptor - Basic adaptor object for GQ::Server::GenericFeature objects

=head1 SYNOPSIS

    my $feature=$feature_class->new(%args);
    $feature->adaptor($feature->context->adaptor($feature));
    $feature->adaptor->do_something_to($feature);

=head1 DESCRIPTION

Real GQ::Server::XXX objects need some basic functionality that is implemented at this level.
The model used for persistent storage of Genquire objects is based on the GFF format, which
if used to its full potential is almost infinitely extensible.

The basic, common information about each feature (Gene, Transcript, Exon, etc.) is stored in
the Feature table.  This includes contig_id, contig_start, contig_stop, strand, length, 'score',
source_tag, primary_tag, frame, and a unique id.  These are returned by the object's _db_keys
method.

The object stores further structured information in the tagvalues hash and the GO list.  Tagvalues
are conceived of as tag/vector pairs, that is, each tag can have one or more values.  Each value
is stored as a single row in the TagValue table, but in memory, the different values are pushed
into a hash of lists keyed on the tag.

GO annotations are read in from the database and stored in GO_annotation objects.

The adaptor layer was designed so that the objects do not need to know about their storage
medium.  They are created by their parent sequence object, which calls its adaptor to load
its enclosed features.  The features are loaded from persistent storage by the SequenceAdaptor
and organized into Genes, Transcripts, etc.  These genomic features all have their own adaptors
which inherit from this GenericFeatureAdaptor.

This implementation of the adaptor layer creates only one instance of each type of adaptor per
active context, then gives each object of that type access to the same adaptor.  The objects
usually send themselves as the second parameter to any adaptor call, i.e.
$self->adaptor->do_something_to($self);

Most of the functionality here has to do with implementing the storage and retrieval of tagvalues,
GO annotations, and flags (see DatabaseAPI.pod).

The adaptors that inherit from GenericFeatureAdaptor will tend to override very few of these
methods, instead adding other methods relevant to their increased functionality.


=head1 FEEDBACK

=head2 Reporting Bugs

Like it or lump it, let me know at dblock@gene.pbi.nrc.ca

=head1 AUTHOR - David Block

Email dblock@gene.pbi.nrc.ca

=head1 CONTRIBUTORS

Genquire is the result of a collaboration between Mark Wilkinson
(mwilkinson@gene.pbi.nrc.ca) and David Block (dblock@gene.pbi.nrc.ca).

=head1 APPENDIX

The rest of the documentation details each of the object methods.
Internal methods are usually preceded with a _ .

=cut


package GQ::Server::DB::Adaptor::GenericFeatureAdaptor;
use strict;
use vars qw( @ISA $AUTOLOAD );  #Keep 'use strict' happy
use Carp;
use DBI;
use Bio::Tk::GO_Annotation;
use GQ::Server::DB::Adaptor::ObjectAdaptor;
use GQ::Root;
@ISA=qw(GQ::Server::DB::Adaptor::ObjectAdaptor);


{
    GQ::Root->create();
}

=head2 insert_new

 Title   : insert_new
 Usage   : $self->{id}=$self->adaptor->insert_new(%hash);
 Function: insert the relevant values into the relevant columns in a single row of
           the database
 Example : $self->{id}=$self->adaptor->insert_new(keys  =>[$self->_db_keys],
			               		  values=>[$self->_db_values],
					         );
 Returns : The database-generated unique id for the new feature
 Args    : a hash with keys 'keys' and 'values', the values being anonymous lists of
           the appropriate keys and values for insertion into the persistent store


=cut

sub insert_new {
    my ($self,%hash)=@_;
    my $table=$self->table;
    return unless $table;
    my $keys=$hash{keys};
    my $cols=join ",",@$keys;
    my $placeholders='?,' x (@$keys - 1);
    my $sth=$self->dbh->prepare("INSERT INTO $table ($cols) VALUES ".
					 "($placeholders NULL)");
    $sth->execute(@{$hash{values}});
    my $id=$self->dbh->{mysql_insertid};
    $sth=$self->dbh->prepare("REPLACE INTO Edit (id, modified_by) VALUES (?,?)");
    $sth->execute($id,$self->context->user);
    $self->context->commit;
    return $id;
}

=head2 delete_from_db

 Title   : delete_from_db
 Usage   : $self->adaptor->delete_from_db($self);
 Function: remove the data from storage associated with this feature
 Returns : nothing
 Args    : the feature whose persistent store is to be eradicated


=cut

sub delete_from_db {
    my ($self,$feat)=@_;
    if ($feat->_update_db) {

	my $table=$self->table;
	my $sth=$self->dbh->prepare("DELETE FROM $table WHERE id=?");
	$sth->execute($feat->id);
	$sth=$self->dbh->prepare("DELETE FROM TagValue WHERE feature_id=?");
	$sth->execute($feat->id);
	$sth=$self->dbh->prepare("DELETE FROM Container WHERE parent=? or element=?");
	$sth->execute($feat->id,$feat->id);
	$self->context->commit;
    }
}

=head2 last_modified

 Title   : last_modified
 Usage   : $self->{last_modified}=$self->adaptor->last_modified($self);
 Function: This method updates the last_modified attribute from persistent storage.
           It is part of the history subsystem of Genquire, keeping track of the history
           of changes to each feature.
 Returns : The last modified date of the feature, according to the persistent store
 Args    : the feature to be examined


=cut

sub last_modified {
    my ($self,$feat)=@_;
    my $sth=$self->dbh->prepare("SELECT last_modified FROM Edit WHERE ".
				"id = ?");
    $sth->execute($feat->id);
    my ($lastmod)=$sth->fetchrow_array;
    return $lastmod;
}

=head2 put_mod_by

 Title   : put_mod_by
 Usage   : $self->adaptor->put_mod_by($self,$user);
 Function: put the username of the user who made a change in a feature
           into the persistent store, creating a record of the change
           with a db-generated timestamp
 Returns : nothing
 Args    : the feature that is being edited, and the username doing the edit


=cut

sub put_mod_by {
    my ($self,$feat,$user)=@_;
    my $sth=$self->dbh->prepare("REPLACE INTO Edit (modified_by,id) ".
				"VALUES (?,?)");
    $sth->execute($user,$feat->id) or die "Database problem: $DBI::errstr";
    $self->context->commit;
}


=head2 update_discard

 Title   : update_discard
 Usage   : if ($self->adaptor->update_discard($self)) {
 Function: move a copy of the current contents of the persistent version of a feature
           into a discard location so that the feature's history is preserved
 Returns : 1 on success, die with an error message on failure
 Args    : the feature that is about to be updated


=cut

sub update_discard {
    my ($self,$feat)=@_;
    my $table=$self->table;
    my $discard_table=$self->discard_table;
    my $user=$self->context->user;
    my $cols=join ',',$feat->_db_keys;
    $cols .= ',discarded_whom';
    # my $placeholders='?,' x ($feat->_db_keys + 1);  # this doesn't work - _db_keys does not return an integer in scalar context
    my @phnumber = $feat->_db_keys;
	my $placeholders='?,' x (scalar(@phnumber) + 1);
    chop $placeholders;
    eval {
	my $sth=$self->dbh->prepare("INSERT INTO $discard_table ".
				    "($cols) VALUES ($placeholders)")
	  or die "Can't prepare: $DBI::errstr";
	$sth->execute((map {$feat->{$_}} $feat->_db_keys), $user)
	  or die "Can't execute discard: $DBI::errstr";
    };
    if ($@) {
	die "Move to discard unsuccessful - no updating performed\n$@";
    } else {
	return 1;
    }
}

=head2 get_tag_values

 Title   : get_tag_values
 Usage   : $self->tagvalues($self->adaptor->get_tag_values($self));
 Function: return the tag/value pairs associated with the feature
 Returns : a hash of lists of values, keyed on tags, associated with the feature
 Args    : the feature to be examined, and optionally, the data already known about
           the feature in a hashref.


=cut

sub get_tag_values {
    my ($self,$feat,$data)=@_;
    my $taghash={};
    my $dbh=$self->dbh;
    my $id=$data->{id}||$feat->id;
    my $sth=$dbh->prepare("SELECT Tags.tag,value FROM Tags,TagValue WHERE ".
			  "feature_id='$id' and Tags.id=TagValue.tag ".
			  "AND parent IS NULL")
      or die "Couldn't prepare: $DBI::errstr";
    $sth->execute or die "Couldn't execute: $DBI::errstr";
    while (my ($tag,$value)=$sth->fetchrow_array) {
	push @{$taghash->{$tag}},$value;
    }
    return $taghash;
}

=head2 insert_tag_values

 Title   : insert_tag_values
 Usage   : $self->adaptor->insert_tag_values($self);
 Function: insert the current taghash of the feature into the persistent store
 Returns : nothing
 Args    : the feature whose values are to be inserted, with a valid tagvalues field


=cut

sub insert_tag_values {
    my ($self,$feat)=@_;

    my $id=$feat->id;
    my %taghash=%{$feat->tagvalues};
    my $sth1=$self->dbh->prepare("SELECT id from Tags WHERE tag=?");
    my $sth2=$self->dbh->prepare("INSERT INTO TagValue (feature_id,tag,value) ".
				  "VALUES (?,?,?)");
    foreach my $tag(keys %taghash) {
	$sth1->execute($tag);
	my ($tagid)=$sth1->fetchrow_array;
	unless ($tagid) {
	    my $sth3=$self->dbh->prepare("INSERT INTO Tags (tag) VALUES (?)");
	    $sth3->execute($tag);
	    $tagid=$self->dbh->{mysql_insertid};
	}
	foreach (@{$taghash{$tag}}) {
	    next unless $_;
	    $sth2->execute($id,$tagid,$_);
	}
    }
    $self->context->commit;
}

=head2 add_tag_value

 Title   : add_tag_value
 Usage   : $self->adaptor->add_tag_value($self,'colour','Brown');
 Function: store a value in persistent storage associated with a tag and a feature
 Returns : 1 on success, 'Tag not present' if the tag is illegal and can't be added
 Args    : The feature that is getting the new tag value
           The new tag
           The new value


=cut

sub add_tag_value {
    my ($self,$feat,$tag,$value)=@_;

    my $sth=$self->dbh->prepare("SELECT id FROM Tags WHERE tag=?");
    $sth->execute($tag);
    my ($tagid)=$sth->fetchrow_array;
    unless ($tagid) {
	my $sth2=$self->dbh->prepare("INSERT INTO Tags (tag) VALUES (?)");
	$sth2->execute($tag);
	$tagid=$self->dbh->{mysql_insertid};
    }
    return 'Tag not present' unless $tagid;
    $sth=$self->dbh->prepare("INSERT INTO TagValue (feature_id,tag,value) ".
			     "VALUES (?,?,?)");
    $sth->execute($feat->id,$tagid,$value) or return;
    $self->context->commit;
    return 1;
}

=head2 remove_tag

 Title   : remove_tag
 Usage   : $self->adaptor->remove_tag($self,'Colour');
 Function: remove all values associated with a tag and a feature
 Returns : 1 on success, undef on failure
 Args    : the feature to be edited, and the tag whose values are to be removed


=cut

sub remove_tag {
    my ($self,$feat,$tag)=@_;

    #get tag id
    my $sth=$self->dbh->prepare("SELECT id FROM Tags WHERE tag=?");
    $sth->execute($tag);
    my ($tagid)=$sth->fetchrow_array;

    #store values about to be deleted
    my $values;
    $sth=$self->dbh->prepare("SELECT value FROM TagValue WHERE ".
			     "feature_id=? AND tag=?");
	$sth->execute($feat, $tagid);
    while (my ($val)=$sth->fetchrow_array) {
	$values .= "$val |";
    }
    chop $values;  #remove last |
    $self->update_discard($feat);
    my $idx=$self->dbh->{mysql_insertid};
    my $discard=$self->discard_table;
    $sth=$self->dbh->prepare("UPDATE $discard set Tag=?, value=? WHERE idx=?");
    $sth->execute($tagid,$values,$idx) or die "Couldn't store old values: $DBI::errstr";

    #now finally delete said values
    $sth=$self->dbh->prepare("DELETE FROM TagValue WHERE ".
			     "feature_id=? AND tag=?");
    $sth->execute($feat->id,$tagid) or return;
    $self->context->commit;
    return 1;
}

=head2 wipe_GO

 Title   : wipe_GO
 Usage   : my $list=$self->adaptor->wipe_GO($self);
 Function: rebuild from scratch the list of GO annotations (without term or evidence information)
 Returns : a listref of GO ids that have been associated with a feature
 Args    : the feature under investigation


=cut

sub wipe_GO {
    my ($self,$feat)=@_;
    my $id=$feat->id;
    my $sth=$self->dbh->prepare("SELECT value FROM Tags,TagValue WHERE ".
				"feature_id='$id' AND Tags.id=TagValue.tag ".
				"AND Tags.tag='GO_annotation' AND parent IS NULL")
      or die "Couldn't prepare: $DBI::errstr";
    my @list;
    $sth->execute or die "Couldn't execute: $DBI::errstr";
    while (my ($value)=$sth->fetchrow_array) {
	push @list, $value;
    }
    return \@list;
}

=head2 GO_get_term

 Title   : GO_get_term
 Usage   : my $term=$self->adaptor->GO_get_term($id);
 Function: get the GO term associated with the given id
 Returns : the GO term
 Args    : the GO id
 Note    : This implementation uses a local cache of terms that may go out of date,
           however, deletion of the local table will only break Genquire.  This routine
           is not capable of going to the GO mother ship and returning with the current id -
           that is the responsibility of the GO api developed by Chris Mungall.
           Further integration of these functions may be desirable in the future.

=cut

sub GO_get_term {
    my ($self,$id)=@_;
    my $sth=$self->dbh->prepare("SELECT term FROM GO_local WHERE id = ?");
    $sth->execute($id);
    my ($term)=$sth->fetchrow_array;
    return $term;
}

=head2 GO_get_evidence

 Title   : GO_get_evidence
 Usage   : my $evidence_hash=$self->adaptor->GO_get_evidence($self,$id);
 Returns : return a hashref that contains a list of evidence values associated with
           each type of evidence, related to a particular feature
 Args    : the feature whose GO evidence is being examined, and the GO id whose evidence
           is in question


=cut

sub GO_get_evidence {
    my ($self,$feat,$id)=@_;
    my %evid;
    my $sth=$self->dbh->prepare("SELECT Tags.tag,value FROM Tags,TagValue WHERE parent = ? ".
				"AND feature_id = ? AND TagValue.tag=Tags.id");
    $sth->execute($id,$feat->id);
    while (my ($code,$evidence)=$sth->fetchrow_array) {
	push @{$evid{$code}},$evidence;
    }
    return \%evid;
}

=head2 GO_add_term

 Title   : GO_add_term
 Usage   : $self->adaptor->GO_add_term($ann);
 Function: add a term to the local cache
 Returns : 1 on success
 Args    : A GO_Annotation object from bioperl-gui, or a hashref that contains at least
           term and id fields


=cut

sub GO_add_term {
    my ($self,$ann)=@_;
    my $sth=$self->dbh->prepare("REPLACE INTO GO_local (id,term) VALUES (?,?)");
    my $term = $ann->term;
    chomp $term;
    $sth->execute($ann->id,$term) or return;
    $self->context->commit;
    return 1;
}

=head2 GO_insert_evidence

 Title   : GO_insert_evidence
 Usage   : $self->adaptor->GO_insert_evidence($self->id,$code,$ev,$ann->id) or return;
 Function: insert evidence into the persistent store
 Returns : 1 on success
 Args    : the feature id the evidence is attached to
           the GO evidence code
           the actual evidence
           the GO id the evidence relates to


=cut

sub GO_insert_evidence {
    my ($self,$fid,$code,$ev,$ann_id)=@_;
    my $sth=$self->dbh->prepare("SELECT id FROM Tags WHERE tag='$code'");
    $sth->execute;
    my ($code_id)=$sth->fetchrow_array;

	unless ($code_id){
		my $sth=$self->dbh->prepare("INSERT INTO Tags (tag) values ('$code')");
		$sth->execute;
		$sth=$self->dbh->prepare("SELECT id FROM Tags WHERE tag='$code'");
		$sth->execute;
		($code_id)=$sth->fetchrow_array;
	}

    $sth=$self->dbh->prepare("INSERT INTO TagValue (feature_id,tag,value,parent) ".
			     "VALUES (?,?,?,?)");
    $sth->execute($fid,$code_id,$ev,$ann_id) or return;
    $self->context->commit;
    return 1;
}

=head2 GO_delete

 Title   : GO_delete
 Usage   : $self->adaptor->GO_delete($self,$GO_id) or die "Unable to delete from database: $DBI::errstr";
 Function: delete a GO annotation from persistent storage
 Returns : 1 on success
 Args    : the feature that is about to lose some GO annotation,
           and the GO id that the feature is about to lose


=cut

sub GO_delete {
    my ($self,$feat,$GO_id)=@_;
    my $id=$feat->id;

    #first get the 'GO_annotation' tagid
    my $sth0=$self->dbh->prepare("SELECT id FROM Tags WHERE tag='GO_annotation'");

    #then get the evidences for that annotation
    my $sth1=$self->dbh->prepare("DELETE FROM TagValue WHERE feature_id=? AND parent=?");

    #finally get that GO annotation itself
    my $sth2=$self->dbh->prepare("DELETE FROM TagValue WHERE feature_id=? AND tag=? AND value=?");

    #okay, now execute all of that
    $sth0->execute;
    my ($gotag)=$sth0->fetchrow_array;

    #but now we want to store the annotation history in tag/value format
    my $values;
    my $sth=$self->dbh->prepare("SELECT tag,value FROM TagValue WHERE feature_id=? and parent=?");
    $sth->execute($id,$GO_id);
    while (my ($tag,$val)=$sth->fetchrow_array) {
	$values .="$tag:$val |";
    }
    chop $values;
    $self->update_discard($feat);
    my $idx=$self->dbh->{mysql_insertid};
    my $discard=$self->discard_table;
    $sth=$self->dbh->prepare("UPDATE $discard SET Tag=?, value=? WHERE idx=?");
    $sth->execute($GO_id,$values,$idx) or return;

    #now that what's about to get blown away is stored in Discard, we can remove it
    $sth1->execute($id,$GO_id) or die $DBI::errstr;
    $sth2->execute($id,$gotag,$GO_id) or die $DBI::errstr;
    $self->context->commit;
    return 1;
}

=head2 insert_flag

 Title   : insert_flag
 Usage   : $self->adaptor->insert_flag($self,$flagid);
 Function: flag a feature with the given flag id
 Returns : nothing, or dies with an error message on failure
 Args    : the feature to be flagged, and the flag id it is to be flagged with


=cut

sub insert_flag {
    my ($self,$feat,$flid)=@_;
    my $sth=$self->dbh->prepare("INSERT INTO Flag (feature_id, flag, contig_id) ".
			        "VALUES (?,?,?)");
    $sth->execute($feat->id,$flid,$feat->contig_id)
      or die "Couldn't Flag ".$feat->id.": $DBI::errstr";
    $self->context->commit;
}

=head2 remove_flag

 Title   : remove_flag
 Usage   : $self->adaptor->remove_flag($self,$flagid);
 Function: remove the given flag id from the list of flags attached to a feature
 Returns : nothing, or dies with an error message on failure
 Args    : the feature to be edited, and the flagid to be removed


=cut

sub remove_flag {
    my ($self,$feat,$flagid)=@_;
    my $sth=$self->dbh->prepare("DELETE FROM Flag WHERE ".
			        "feature_id=? AND flag=?");
    $sth->execute($feat->id,$flagid) 
      or die "Couldn't remove $flagid from feature ".$feat->id.": $DBI::errstr";
    $self->context->commit;
}

=head2 find_duplicate_tags

 Title   : find_duplicate_tags
 Usage   : $self->adaptor->find_duplicate_tags($self->id);
 Function: find all of the features in the database that this feature is tagged as having
           homology to
 Returns : a list of feature ids
 Args    : the feature id to be checked


=cut

sub find_duplicate_tags {
    my ($self,$id)=@_;
    my @list;
    my $sth_dup=$self->dbh->prepare("SELECT value FROM Tags,TagValue WHERE ".
				    "Tags.tag='Homology_to' AND feature_id=? AND ".
                                    "Tags.id=TagValue.tag");
    $sth_dup->execute($id);
    while (my ($contig)=$sth_dup->fetchrow_array) {
	push @list,$contig;
    }
    return @list;
}

=head2 getBlastHits

 Title   : getBlastHits
 Usage   : $self->{BlastHits}=$self->adaptor->getBlastHits($self->id);
 Function: get a list of blast hit information associated with the current feature
 Returns : a list of anonymous lists, made up of a gi, probability and description from
           a blast hit.  This implementation brings this up from a cache, which is created
           when a user explicitly blasts a feature.  The results are stored so further blasts
           are unnecessary.  This may be a problem if the target database changes frequently -
           this data ages quickly, and should have more metadata associated with it.
 Args    : the feature id whose blast hits are to be gotten


=cut

sub getBlastHits {
    my ($self,$id)=@_;
    my @results;
    my $sth_blast = $self->dbh->prepare("Select gi, probability, description ".
					"from BlastAcc where exon_id = ?");
    $sth_blast->execute($id);

    while (my ($gi,$prob,$desc)=  $sth_blast->fetchrow_array){
	push @results,[$gi,$prob,$desc];
    }
    return \@results;
}

=head2 change_type

 Title   : change_type
 Usage   : $self->adaptor->change_type($self,$type);
 Function: change the type of the feature, including in the data store
 Returns : nothing, or dies with an error on failure
           will add the type to the database if it doesn't exist, but 
           it will NOT be a container-type object.
 Args    : the feature to be changed, and the new type


=cut

sub change_type {
   my ($self,$fea,$type) = @_;
	my $sth=$self->dbh->prepare("SELECT id FROM FeatureType WHERE type=?");
	$sth->execute($type);
	my ($type_id)=$sth->fetchrow_array;
	unless ($type_id){
		$self->dbh->do("insert into FeatureType (type, container) values ('".$type."', 'N')");
		$type_id = $self->dbh->{mysql_insertid};
	}
	$sth=$self->dbh->prepare("UPDATE Feature SET type=? WHERE id=?");
	$sth->execute($type_id,$fea->id) or die "Couldn't change $fea to type $type: $DBI::errstr";
	$self->context->commit;
}

=head2 find_parent_id

 Title   : find_parent_id
 Usage   : my $parent_id=$self->adaptor->find_parent_id($self);
 Function: return the id of the element's parent gene
 Example : return the gene id of a transcript, or of an exon
 Returns : a Feature table id
 Args    : the element


=cut

sub find_parent_id {
   my ($self,$feat) = @_;
   my $sth=$self->dbh->prepare("SELECT Feature.id FROM Feature,Container,FeatureType ".
			       "WHERE Feature.type=FeatureType.id AND FeatureType.type='Gene' ".
			       "AND Container.parent=Feature.id and Container.element=?"
			      );
   $sth->execute($feat->id);
   my ($gene_id)=$sth->fetchrow_array;
   return $gene_id;
}


1;

