package GQ::Server::GBFlat::Adaptor::SequenceAdaptor;
use strict;
use vars qw(@ISA $AUTOLOAD);
use Carp;
use DBI;
use GQ::Root;
use GQ::Server::GBFlat::Adaptor::ObjectAdaptor;
@ISA=qw(GQ::Server::GBFlat::Adaptor::ObjectAdaptor);


{
#Encapsulated class data and subroutines

    GQ::Root->create();
	
	my $unique_id = 0;  # need to have a class-wide unique ID for each feature object
	sub nextID{
		my ($self) = @_;
		++$unique_id;
		return $unique_id;
	}

}

=head2 where_am_i

 Title   : where_am_i
 Usage   : my ($Contig_id,$contig_loc)=$self->adaptor->where_am_i($self,$Sequence_loc);
 Function: given a location relative to a given sequence object, return the contig id and location in contig
           coordinates
 Returns : a list of two elements: a contig_id and a contig_location
 Args    : the contig ID, and the location on that contig object


=cut

sub where_am_i {
    my ($self,$feat,$loc)=@_;
    my $abs_loc = $loc + $feat->start;
    return ($feat->context->DEFAULT_ID,$abs_loc);
}

=head2 get_contig_names

 Title   : get_contig_names
 Usage   : my @list=$self->adaptor->get_contig_names($self);
 Function: get the list of contig names whose virtual contigs overlap this sequence
 Example : foreach my $contig(@{$MapSeq->get_contig_names}){   # all of the contigs which participate in this sequence
 Returns : a list of contig names
 Args    : the sequence object


=cut

sub get_contig_names {
    my ($self,$seq)=@_;

    my $contig = $seq->context->contig_by_id($seq->context->DEFAULT_ID);
	return [$contig->name];
}

=head2 getseq

 Title   : getseq
 Usage   : my $seq=$self->adaptor->getseq($self,$start,$end)
 Example : my $feat_seq=$MapSeq->adaptor->getseq($MapSeq,$feat->start,$feat->end)
 Returns : a dna sequence corresponding to either the whole sequence, or the desired substring
 Args    : the sequence object, and optionally the start and end values of the desired sequence
           with no start/end values, the sequence of the whole sequence is returned


=cut

sub getseq {
    my ($self,$feat,$start,$end)=@_;
    my ($startpos,$length);
    if ($start) {                      #get start,stop relative to loaded sequence
	$startpos=$start + $feat->start - 1;
	$length=$end-$start+1;
    } else {
	$startpos=$feat->start;
	$length=$feat->length;
    }
    if (my $seq=$feat->SeqHash($startpos,$length)) {
	return $seq;
    }

	my $seq = $feat->context->GBobj->subseq($startpos, ($startpos+$length)-1);
    $feat->SeqHash($startpos,$length,$seq);
    return $seq;
}

=head2 _features

 Title   : _features
 Usage   : $seq->adaptor->_features($seq);
 Function: load all the relevant features from the database into the sequence
 Returns : nothing - the sequence is fully loaded, however
 Args    : the sequence object


=cut

sub _features {
    my ($self,$Seq)=@_;
	#print "Seq_Start ",$Seq->start, " BC_start ", $Seq->bc_start," length ",$Seq->length,"\n";
    my $start=$Seq->start + $Seq->bc_start - 1;
    my $stop =$start + $Seq->length - 1;
    $Seq->{features}=$self->load($Seq,
				 context=>$Seq->context,
				 lockid=>$Seq->lockid,
				 par_pos=>$start,  # assembly cordinates (probably absolute chromosome coords)
				 parent=>$Seq, # the Seq object to which the features belong
				 max=>$stop,  
				 min =>$start,
				);
	# Genbank features from BioPerl parse can not be assembled... yet!
	# possibly future versions of BioPerl will have "container" feature types
    # $self->assemble($Seq);
}

=head2 load

 Title   : load
 Usage   : $self->load(%args)
 Function: load the sequence features into memory
 Returns : a reference to the hash of features
 Args    : a hash containing
            id       => unique feature ID
            contig_id=> ID of the contig object this feature is on
            contig_start
            contig_stop
            length => integer representing the length
            strand 	 => GFF format strand designation
            primary_tag
            frame
            score
            seqname
            source_tag
            context  => the context object
            lockid   => the current lock id
            par_pos  => the start of the parent sequence, in assembly coordinates

=cut

sub load {
    my ($self,$Seq,%args)=@_;
    my ($context,$assembly,$lockid,$par_pos,$parent)=@args{'context','assembly','lockid','par_pos','parent'};
    my %list;
	my $max = $args{max};
	my $min = $args{min};
    
    foreach my $feature($Seq->context->GBobj->all_SeqFeatures){
		next unless $self->context->_type($feature->primary_tag);  # should always have a default of Server::Feature
		next unless ($feature->start < $max);
		next unless ($feature->end > $min);
		next if ($feature->primary_tag eq "source");
	
		# strand must be in GFF format for Genquire to understand it.		
		my $strand = ($feature->strand =~ /\-/)?"-":"+";
		
		# create the feature
		my $newfeat=$self->context->_type($feature->primary_tag)->new(
			# *****  CRITICAL!! ****
			id => $self->nextID,
			# **********************
			
			contig_start => $feature->start,
			contig_stop  => $feature->end,
			length => ($feature->end - $feature->start + 1),
			# note that we don't set start/stop we set contig_start/stop
			
			strand => $strand,
			primary_tag => $feature->primary_tag,
			frame => $feature->frame,
			score=>$feature->score,
			seqname => $feature->seqname,
			source_tag => $feature->source_tag,
			
			contig_id => $self->context->DEFAULT_ID,
			lockid    => $lockid,
			context   => $context,
			par_pos   => $par_pos,  # always 1
			);
		#print "new feature id ", $newfeat->{id}, " start ", $newfeat->start, "(orig ",$feature->start,") strand ",$newfeat->strand;
		$newfeat->attach_seq($parent);
		if ($newfeat && ref($newfeat) eq $self->context->_type($feature->primary_tag)) {
			#print "  ADDED!!\n";
			$list{$newfeat->{id}}=$newfeat;
		}
		
		# need some hook out from the Genquire features to id them back to teh Genbank features....
		$feature->{GenquireID} = $newfeat->{id};
    }
    return \%list;
}

1;

