//WinBLAST is a graphical front-end for NCBI BLAST.
//Copyright (C) 2003 Alan Li, alanli@bioinformatics.org
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

using System;
using System.Diagnostics;
using System.IO;

namespace WinBLAST
{
	/// <summary>
	/// Summary description for WinBLASTWorker.
	/// </summary>
	public class WinBLASTWorker
	{
		private static string COMMAND = "blastall";

		private WinBLAST m_parent;

		private string m_programName;
		private Decimal m_expectedValue;
		private string m_dbName;
		private int m_alignmentView;
		private Decimal m_numOfAlignments;
		private bool m_isFileQuery;
		private string m_query;
		private string m_result;
		private string m_error;

		public WinBLASTWorker( WinBLAST blastForm )
		{
			m_parent = blastForm;
			m_programName = "";
			m_expectedValue = 0;
			m_dbName = "";
			m_alignmentView = 0;
			m_numOfAlignments = 0;
			m_isFileQuery = true;
			m_query = "";
			m_result = "";
			m_error = "";
		}

		public bool runBlast()
		{
			// various arguments to blastall
			string arguments = "-p " + m_programName + " ";
			arguments += "-e " + m_expectedValue + " ";
			arguments += "-d " + m_parent.getDatabase().findDBLocation( m_dbName ) + " ";
			arguments += "-m " + m_alignmentView + " ";
			arguments += "-b " + m_numOfAlignments + " ";

			// user selected to use a file as an input.
			if( m_isFileQuery )
				arguments += "-i " + m_query + " ";

			// we have to spawn another process
			// set the process startup info
			ProcessStartInfo startInfo = new ProcessStartInfo( COMMAND, arguments );
			startInfo.UseShellExecute = false;
			startInfo.RedirectStandardInput = true;
			startInfo.RedirectStandardOutput = true;
			startInfo.RedirectStandardError = true;
			startInfo.CreateNoWindow = true;

			// instantiate and start the process
			Process blastProc = new Process();
			blastProc.StartInfo = startInfo;
			blastProc.Start();

			StreamWriter writer = blastProc.StandardInput;
			StreamReader reader = blastProc.StandardOutput;
			StreamReader error = blastProc.StandardError;

			// using a sequence as the input.
			if( ! m_isFileQuery )
			{
				// get rid of all newlines, since that could truncate the input
				m_query = m_query.Replace( "\r\n", "" );
				m_query = m_query.Replace( "\n", "" );
				writer.Write( m_query );
				writer.Flush();
			}

			// we're done with the stdin
			writer.Close();

			// NOTE: output stream must be read before the error stream.
			m_result = reader.ReadToEnd();
			m_error = error.ReadToEnd();

			// we're done with the stdout and stderr.
			reader.Close();
			error.Close();

			return ( m_error.Length <= 0 );
		}

		public void setProgramName( string programName )
		{
			m_programName = programName;
		}

		public void setExpectedValue( Decimal expectedValue )
		{
			m_expectedValue = expectedValue;
		}

		public void setDBName( string dbName )
		{
			m_dbName = dbName;
		}

		public void setAlignmentView( int alignmentView )
		{
			m_alignmentView = alignmentView;
		}

		public void setNumOfAlignments( Decimal numOfAlignments )
		{
			m_numOfAlignments = numOfAlignments;
		}

		public void setIsFileQuery( bool isFileQuery )
		{
			m_isFileQuery = isFileQuery;
		}

		public void setQuery( string query )
		{
			m_query = query;
		}

		public string getResult()
		{
			return m_result;
		}

		public string getError()
		{
			return m_error;
		}
	}
}
