#line 2 "copyright.nw"
/*
    ALiBio: Algorithms Library for Bioinformatics
    Copyright (C) 2002 	Gianluca Della Vedova, Riccardo Dondi, Luca Fossati, 
    			Lorenzo Mariani, Patrizia Pagliarulo.
    			 	
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

http://bioinformatics.org/ALiBio

Lab. Bioinformatica
DISCo, Univ. Milano-Bicocca
via Bicocca degli Arcimboldi 8
20126 Milano (Italy)
*/

#line 282 "data_structures/trees.nw"
#ifndef ALIBIO_TREES_INCLUDED
#define ALIBIO_TREES_INCLUDED

    // include files
#include <string>
#include <vector>
#include <iostream>
#include <stdlib.h>
#include <assert.h>
#include <boost/rational.hpp>

namespace alibio {


#line 356 "data_structures/trees.nw"
template <class I, class J>
class tree;


#line 966 "data_structures/trees.nw"
template <class J> 
class edge
{

private:

#line 991 "data_structures/trees.nw"
    
    std::string name;
    boost::rational<J> weight;


#line 974 "data_structures/trees.nw"
public:


#line 1042 "data_structures/trees.nw"
          // constructors

    edge ()
        { name=""; set_weight(0,1); }

    edge (std::string n)
        { assert(n.length()>=0); set_name(n); set_weight(0,1); }

    ~edge () {} 

          // member functions

    std::string get_name () const
        { return name; }

    void set_name (std::string n)
        { assert(n.length()>=0); name=n; }

    boost::rational<J> get_weight () const
        { return weight; }

    void set_weight (J , J ); 

    const edge<J> &operator=(const edge<J> &);

    void print_edge() const;



#line 978 "data_structures/trees.nw"
};


#line 1884 "data_structures/trees.nw"
template <class J>
const edge<J> &
edge<J>::operator=(const edge<J> &rhs) {
    if (this!=&rhs){
        set_name(rhs.get_name());
        set_weight(rhs.get_weight().numerator(), rhs.get_weight().denominator());
    }
    return *this;       
}


#line 1913 "data_structures/trees.nw"
template <class J>
void edge<J>::set_weight (J num, J den) {
    assert(den!=0);
    if (den==1) weight=num;
    else weight.assign(num, den);
}


#line 1936 "data_structures/trees.nw"
template <class J>
void edge<J>::print_edge() const {
    std::cout << "\nname: " << get_name();
    std::cout << "\nweight: " << get_weight().numerator();
    if (get_weight().denominator()!=1)
        std::cout << "/" << get_weight().denominator();
    std::cout << "\n";
}




#line 732 "data_structures/trees.nw"
template <class I, class J> 
class vertex {

friend class tree<I, J>;

private:


#line 756 "data_structures/trees.nw"
typedef vertex<I, J> * v_addr;
 

#line 779 "data_structures/trees.nw"
    std::string name;
    std::vector<std::string> property;
    boost::rational<I> weight;
    vertex<I, J> *father;
    edge<J> upper_edge;
    std::vector<v_addr> descendents;


#line 742 "data_structures/trees.nw"
public:


#line 879 "data_structures/trees.nw"
          // constructors

    vertex () 
        { name=""; set_weight(0,1); father=NULL; property.resize(0); 
          descendents.resize(0); }

    vertex ( std::string n , vertex<I, J> *f, edge<J> &ue)
        { set_name(n); set_weight(0,1); father=f; upper_edge=ue; 
          property.resize(0); descendents.resize(0); }

    vertex (const vertex<I, J> & );
  
    ~vertex () 
    { for (int i=0; i<no_of_sons(); i++) delete descendents[i]; }

          // member functions

    std::string get_name () const
        { return name; }
   
    void set_name (std::string n)
        { assert(n.length()>=0); name=n; }
   
    v_addr get_father () const
        { return father; }

    edge<J> get_uedge () const
        { return upper_edge; }

    bool set_uedge (J num, J den)
        { 
      if (father!=NULL) { 
          upper_edge.set_weight(num, den); 
          return true;
      } else return false; 
    }
   
    void print_node () const;
   
    void new_son (std::string, std::string );

    v_addr son_no (unsigned int i) const
        { assert( (i>=0) && (i<no_of_sons()) ); return descendents[i]; }
   
    unsigned int no_of_sons () const
        { return descendents.size(); }
  
    std::string prop_no (unsigned int i) const
        { assert( (i>=0) && (i<no_of_props()) ); return property[i]; }

    unsigned int no_of_props () const
        { return property.size(); }
            
    void add_prop (std::string p)
        { assert(p.length()>=0); property.push_back(p); }
   
    boost::rational<I> get_weight () const
        { return weight; }
   
    void set_weight (I , I ); 

    const vertex<I, J> & operator= (const vertex<I, J> & );

#line 745 "data_structures/trees.nw"
};


#line 1683 "data_structures/trees.nw"
template <class I, class J>
vertex<I, J>::vertex (const vertex<I, J> &source ) {    
    assert( ((source.get_father()==NULL)&&(source.get_uedge().get_name()=="")) || ((source.get_father()!=NULL)&&(source.get_uedge().get_name()!="")) );
    father=source.get_father();

    operator=(source);

}


#line 1735 "data_structures/trees.nw"
template <class I, class J>
const vertex<I, J> &
vertex<I, J>::operator= (const vertex<I, J> &source ) {
    if (this!=&source) {
        upper_edge=source.get_uedge();
        set_name(source.get_name());
        set_weight(source.get_weight().numerator(),source.get_weight().denominator());
    
        descendents.resize(0);
    
        for (int i=0; i<source.no_of_props(); i++)
            add_prop( source.prop_no(i) );
    }

    return source;
}


#line 1778 "data_structures/trees.nw"
template <class I, class J>
void vertex<I, J>::new_son (std::string n, std::string edge_n) {
    edge<J> e;
    
    if (edge_n!="") e.set_name(edge_n);
    else e.set_name("e");
    
    v_addr obj=new vertex<I, J>(n, this, e);
    descendents.push_back(obj);
}


#line 1806 "data_structures/trees.nw"
template <class I, class J>
void vertex<I, J>::set_weight (I num, I den) {
    assert(den!=0);
    if (den==1) weight=num;
    else weight.assign(num, den);
}


#line 1841 "data_structures/trees.nw"
template <class I, class J>
void vertex<I, J>::print_node () const {
    std::cout << "\n\n\nNode " << get_name();
    std::cout << "\nfather's name: ";
    if (father==NULL) 
        std::cout << "-";
    else std::cout << (*father).get_name();
    std::cout << "\nconnecting edge: \n";
    get_uedge().print_edge();
    std::cout << "\nweight: " << get_weight().numerator();
    if (get_weight().denominator()!=1) 
        std::cout << "/" << get_weight().denominator();
    std::cout << "\nproperty list: ";
    for (int i=0; i<no_of_props(); i++) 
        std::cout << "\n" << prop_no(i);
    std::cout << ".\n";
}




#line 486 "data_structures/trees.nw"
template <class I, class J>
class tree_iterator {

    friend class tree<I, J>;
    typedef tree_iterator<I, J> iterator;

private:


#line 531 "data_structures/trees.nw"
    std::vector<unsigned int> map;
    vertex<I, J> *current;
    vertex<I, J> *father;


#line 496 "data_structures/trees.nw"
public:


#line 659 "data_structures/trees.nw"
          // constructors

    tree_iterator (vertex<I, J> * root) 
        { assert(root!=NULL); current=root; father=NULL; map.resize(0); } 

    tree_iterator (vertex<I, J> * node, std::vector<unsigned int> pos); 
    
    tree_iterator (const iterator & src)
        { assert(src.is_valid()); operator=(src); }
    
          // member functions

    bool is_valid () const;

    bool has_next (const iterator & end) const
        { return ( ( (*this).is_valid() )&&( operator!=(end) ) ); }

    bool has_prev (const iterator & begin) const
        { return ( ( (*this).is_valid() )&&( operator!=(begin) ) ); }
  
    iterator & operator++ ();
    
    iterator operator++ (int )
        { iterator aux(*this); operator++(); return aux; }

    iterator & operator-- ();
    
    iterator operator-- (int )
        { iterator aux(*this); operator--(); return aux; }
    
    bool operator== (const iterator & src) const
        { return (current==src.current); }

    bool operator!= (const iterator & src) const
        { return !( operator==(src) ); }

    bool operator< (const iterator & src) const;
    
    bool operator> (const iterator & src) const
        { return ( src.operator<(*this) ); }

    bool operator<= (const iterator & src) const
        { return ( ( operator<(src) )||( operator==(src) ) ); }

    bool operator>= (const iterator & src) const
        { return ( ( operator>(src) )||( operator==(src) ) ); }

    const iterator & operator= (const iterator & src);
    
    vertex<I, J> & operator* ()  // do not make the ret val a const ref!
        { return *current; } 

    void reset ();

    void make_consistent ();


#line 500 "data_structures/trees.nw"
};


#line 1356 "data_structures/trees.nw"
template <class I, class J>
tree_iterator<I, J>::tree_iterator (vertex<I, J> * node, std::vector<unsigned int> pos) 
{ 
    assert(node!=NULL);
    assert(pos.size()>=0);        
    
    if (pos.size()>0) {
        assert ((*node).get_father()!=NULL);
        father=(*node).get_father();
    } else father=NULL;

    current=node; 
    for (int i=0; i<pos.size(); ++i)
        map.push_back(pos[i]);
}


#line 1393 "data_structures/trees.nw"
template <class I, class J>
bool tree_iterator<I, J>::is_valid() const { 
    if (map.size()<0) return false;
    if (current==NULL) return false;
    if ( (map.size()>0)&&(father==NULL) ) return false;
    if ( (map.size()>0)&&((*current).get_father()!=father) ) return false;
    return true;  
}


#line 1439 "data_structures/trees.nw"
template <class I, class J>
tree_iterator<I, J> & 
tree_iterator<I, J>::operator++ () {
    if ((*current).no_of_sons()>0) {
        map.push_back(0);
        father=current;
    }
    else {
        ++map[map.size()-1]; 
        while (map[map.size()-1]==(*father).no_of_sons()) {
            map.resize(map.size()-1);
            ++map[map.size()-1];
            father=(*father).get_father();
        }
    }
    current=(*father).son_no(map[map.size()-1]);

    return (*this);
}


#line 1497 "data_structures/trees.nw"
template <class I, class J>
tree_iterator<I, J> & 
tree_iterator<I, J>::operator-- () {
    if (map[map.size()-1]==0) {
        map.resize(map.size()-1);
        current=father;
    }
    else {
        --map[map.size()-1];
        current=(*father).son_no(map[map.size()-1]);
        while ( (*current).no_of_sons()>0 ) {
            map.push_back( (*current).no_of_sons()-1 );
            current=(*current).son_no( map[map.size()-1] );
        }
    }
    father=(*current).get_father();

    return (*this);
}


#line 1543 "data_structures/trees.nw"
template <class I, class J>
bool tree_iterator<I, J>::operator< (const iterator & src) const { 
    if ( operator==(src) ) return false;

    iterator aux(*this);
    aux.reset();

    while ( (operator!=(aux))&&(aux.operator!=(src)) ) {
        aux.operator++();
    }
    
    return (operator==(aux));
}


#line 1577 "data_structures/trees.nw"
template <class I, class J>
const tree_iterator<I, J> & 
tree_iterator<I, J>::operator= (const iterator & src) { 
    assert(src.is_valid()); 

    if ( operator!=(src) ) {
        current=src.current; 
        father=src.father;
        map.resize(0); 
        for (int i=0; i<(src.map).size(); ++i)
            map.push_back( (src.map)[i] ); 
    }

    return (*this);
}


#line 1613 "data_structures/trees.nw"
template <class I, class J>
void tree_iterator<I, J>::reset ()
{
    for (int i=map.size(); ( (i>0)&&( (*current).get_father()!=NULL) ); --i)
        current=(*current).get_father();

    map.resize(0);
    father=NULL;
}


#line 1643 "data_structures/trees.nw"
template <class I, class J>
void tree_iterator<I, J>::make_consistent ()
{ 
    iterator aux(*this); 

    for ((*this).reset(); operator!=(aux); operator++() ) {}
}




#line 332 "data_structures/trees.nw"
template <class I, class J>
class tree {

private:


#line 382 "data_structures/trees.nw"
   vertex<I, J> *root;


#line 339 "data_structures/trees.nw"
public:


#line 373 "data_structures/trees.nw"
    typedef tree_iterator<I, J> iterator;


#line 424 "data_structures/trees.nw"
          // constructors

    tree () 
        { root=new vertex<I, J>(); }

    tree (const vertex<I, J> &source)
        { assert( (source.get_father()==NULL)&&(source.get_uedge().get_name()=="") ); 
          root=new vertex<I, J>(source); }

    tree (const tree<I, J> & );

    ~tree ()
        { delete root; }

    // member functions

    bool empty () const
        { return (root==NULL); }

    iterator begin () const
        { return tree_iterator<I, J>(root); }     

    iterator end () const;

    int size () const;

    tree<I, J> subs_switch ( std::string , std::string );

    void print_tree () const;


#line 344 "data_structures/trees.nw"
};


#line 1129 "data_structures/trees.nw"
template <class I, class J>
tree<I, J>::tree (const tree<I, J> & src) {
    root=new vertex<I, J> ( *(src.begin()) );
    iterator itr1(src.begin());
    iterator itr2(root);

    for (int i=0; i<(*itr1).no_of_sons(); ++i)
            (*itr2).new_son("", "");
    while ( itr1.has_next(src.end()) ) {
        ++itr1;
        ++itr2;
        (*itr2)=(*itr1);
        for (int i=0; i<(*itr1).no_of_sons(); ++i)
            (*itr2).new_son("", "");
    }
}


#line 1169 "data_structures/trees.nw"
template <class I, class J>
tree_iterator<I, J>
tree<I, J>::end () const { 
    vertex<I, J> * node=root;
    std::vector<unsigned int> map(0);
    
    // the user must check that the tree on which he wants to call this function is not empty
    while ( (*node).no_of_sons()!=0 ) {
        map.push_back( (*node).no_of_sons()-1 );
        node=(*node).son_no( (*node).no_of_sons()-1 ); 
    }
    return tree_iterator<I, J>(node, map);
}    


#line 1202 "data_structures/trees.nw"
template <class I, class J>
int tree<I, J>::size () const {
    int my_size=1;
    
    if (empty()) return 0;
    for (tree_iterator<I, J> itr(begin()); itr!=end(); ++itr) {
        ++my_size;
    }
    
    return my_size;
}


#line 1259 "data_structures/trees.nw"
template <class I, class J>
tree<I, J>
tree<I, J>::subs_switch (std::string n1, std::string n2) {
    if ( (empty())||(n1==n2) ) return (*this);

    tree_iterator<I, J> itr1(begin());
    
    while ( ((*itr1).get_name()!=n1)&&(itr1.has_next(end())) ) 
        ++itr1;
    if ((*itr1).get_name()!=n1) return (*this);
    
    tree_iterator<I, J> itr2(begin());
    
    while ( ((*itr2).get_name()!=n2)&&(itr2.has_next(end())) ) 
        ++itr2;
    if ((*itr2).get_name()!=n2) return (*this);

    std::vector<unsigned int> map1(itr1.map);
    std::vector<unsigned int> map2(itr2.map);    
    bool flag=true;
    for (int i=0; (i<map1.size())&&(i<map2.size())&&(flag) ; ++i) {
        if (map1[i]!=map2[i]) flag=false;
    }
    if (flag) return (*this);

    vertex<I, J> * f1=(*itr1).get_father();
    unsigned int i=map1[map1.size()-1];
    vertex<I, J> * f2=(*itr2).get_father();
    unsigned int j=map2[map2.size()-1];

    vertex<I, J> * aux=(*f1).son_no(i);
    (*f1).descendents[i]=(*f2).son_no(j);
    (*f2).descendents[j]=aux;
    (*((*f1).descendents[i])).father=f1;
    (*((*f2).descendents[j])).father=f2;

    return (*this);
}


#line 1312 "data_structures/trees.nw"
template <class I, class J>
void tree<I, J>::print_tree() const {
    tree_iterator<I, J> itr(begin());

    for (; itr.has_next(end()); ++itr) {
        (*itr).print_node();
    }
    (*itr).print_node();    
}




#line 301 "data_structures/trees.nw"
}

#endif

