/*
  This file contains the implementation of the BIOSEG PostgreSQL type.  The
  implementation is a copy of the SEG type from the contrib directory in the
  PostgreSQL source, which has been changed to represent ranges of integers.
  The anticipated use of this type is to represent contiguous intervals of a
  biological sequence.

  SEG code written Gene Selkov, Jr.

  Changes for BIOSEG written by Kim Rutherford <kmr@xenu.org.uk>.
*/

#include "postgres.h"

#include <stdlib.h>
#include <errno.h>
#include <limits.h>

#include "access/gist.h"
#include "access/skey.h"
#include "utils/builtins.h"

#include "biosegdata.h"

#ifdef INTERBASE_COORDS
#define BIOSEG bioseg0
#define BIOSEG_PREFIX(x) bioseg0 ## x
#else
#define BIOSEG bioseg
#define BIOSEG_PREFIX(x) bioseg ## x
#endif

PG_MODULE_MAGIC;

extern int  BIOSEG_PREFIX(_yyparse)();
extern void BIOSEG_PREFIX(_yyerror)(const char *message);
extern void BIOSEG_PREFIX(_scanner_init)(const char *str);
extern void BIOSEG_PREFIX(_scanner_finish)(void);

/*
** Input/Output routines
*/
SEG   *BIOSEG_PREFIX(_in)(char *str);
char  *BIOSEG_PREFIX(_out)(SEG * seg);
SEG   *BIOSEG_PREFIX(_create)(int32 lower, int32 upper);
int32  BIOSEG_PREFIX(_lower)(SEG * seg);
int32  BIOSEG_PREFIX(_upper)(SEG * seg);
int32  BIOSEG_PREFIX(_center)(SEG * seg);

/*
** GiST support methods
*/
bool           BIOSEG_PREFIX(_gist_consistent)(GISTENTRY *entry, SEG * query, StrategyNumber strategy);
GISTENTRY     *BIOSEG_PREFIX(_gist_compress)(GISTENTRY *entry);
GISTENTRY     *BIOSEG_PREFIX(_gist_decompress)(GISTENTRY *entry);
float         *BIOSEG_PREFIX(_gist_penalty)(GISTENTRY *origentry, GISTENTRY *newentry, float *result);
GIST_SPLITVEC *BIOSEG_PREFIX(_gist_picksplit)(GistEntryVector *entryvec, GIST_SPLITVEC *v);
bool           BIOSEG_PREFIX(_gist_leaf_consistent)(SEG * key, SEG * query, StrategyNumber strategy);
bool           BIOSEG_PREFIX(_gist_internal_consistent)(SEG * key, SEG * query, StrategyNumber strategy);
SEG           *BIOSEG_PREFIX(_gist_union)(GistEntryVector *entryvec, int *sizep);
SEG           *BIOSEG_PREFIX(_gist_binary_union)(SEG * r1, SEG * r2, int *sizep);
bool          *BIOSEG_PREFIX(_gist_same)(SEG * b1, SEG * b2, bool *result);


/*
** R-tree support functions
*/
bool   BIOSEG_PREFIX(_same)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_contains_int)(SEG * a, int *b);
bool   BIOSEG_PREFIX(_contains)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_contained)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_overlap)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_left)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_over_left)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_right)(SEG * a, SEG * b);
bool   BIOSEG_PREFIX(_over_right)(SEG * a, SEG * b);
void   BIOSEG_PREFIX(_rt_size)(SEG * a, int32 *sz);
int32  BIOSEG_PREFIX(_size)(SEG * a);

static SEG *BIOSEG_PREFIX(_union)(SEG * a, SEG * b);

/*
** Various operators
*/
int32 BIOSEG_PREFIX(_cmp)(SEG * a, SEG * b);
bool  BIOSEG_PREFIX(_lt)(SEG * a, SEG * b);
bool  BIOSEG_PREFIX(_le)(SEG * a, SEG * b);
bool  BIOSEG_PREFIX(_gt)(SEG * a, SEG * b);
bool  BIOSEG_PREFIX(_ge)(SEG * a, SEG * b);
bool  BIOSEG_PREFIX(_different)(SEG * a, SEG * b);
Datum BIOSEG_PREFIX(_joinsel)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_sel)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_contsel)(PG_FUNCTION_ARGS);
Datum BIOSEG_PREFIX(_contjoinsel)(PG_FUNCTION_ARGS);


static int get_dots(char **str);
static int get_int(char **str, int32 *result);


/*****************************************************************************
 * Input/Output functions
 *****************************************************************************/

int MAX_DIGITS = 10;

SEG *
  BIOSEG_PREFIX(_in)(char *str)
{
  int32 lower;
  int32 upper;

  if (!get_int(&str, &lower)) {
    return NULL;
  }

  if (str[0] == 0) {
    upper = lower;
  } else {
    if (!get_dots(&str)) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("number followed by something other than ..: %s", str)));
      return NULL;
    }

    if (!get_int(&str, &upper)) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("number\"..\" followed by something other than a number: %s", str)));
      return NULL;
    }

    if (lower > upper) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("lower limit of range greater than upper")));
      return NULL;
    }

    if (str[0] != 0) {
      ereport(ERROR,
              (errcode(ERRCODE_SYNTAX_ERROR),
               errmsg("bad bioseg representation"),
               errdetail("garbage at end of string: %s", str)));
      return NULL;
    }
  }

  {
    SEG *result = palloc(sizeof(SEG));

    result->lower = lower;
    result->upper = upper;
    return result;
  }
}

#ifdef INTERBASE_COORDS
#define MIN_LOWER INT_MIN
#else
#define MIN_LOWER 1l
#endif

int get_int(char **strp, int32 *result) {
  char *return_pos;
  long int long_result = -1;

  if (!(*strp)[0]) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("end of string found when expecting an integer")));
    return 0;
  }

  errno = 0;

  long_result = strtol(*strp, &return_pos, 0);

  if (errno == ERANGE && (long_result == LONG_MAX || long_result == LONG_MIN)) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("integer at: %s is out of range", *strp)));
    return 0;
  }

  if (errno != 0 && long_result == 0) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("unable to read an integer: %s", strerror(errno))));
    return 0;
  }

  if (*strp == return_pos) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("no integer found at: %s", *strp)));
    return 0;
  }

  if (long_result < MIN_LOWER) {
    // when INTERBASE_COORDS is set, on machines where
    // sizeof(long int) == sizeof(int32) this block won't be executed because
    // strtol() will return ERANGE
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("integer %ld at: %s is out of range - must be >= %ld",
                       long_result, *strp, MIN_LOWER)));
    return 0;
  }

  if (long_result > INT_MAX) {
    // on machines where sizeof(long int) == sizeof(int32) this won't happen
    // because strtol() will return ERANGE
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("integer %ld at: %s is out of range - must be <= %d",
                       long_result, *strp, INT_MAX)));
    return 0;
  }

  *strp = return_pos;
  *result = long_result;

  return 1;
}

int
  get_dots(char **strp)
{
  if ((*strp)[0] == '.') {
    (*strp)++;
    if ((*strp)[0] == '.') {
      (*strp)++;
      if ((*strp)[0] == '.') {
        // allow for "10...20"
        (*strp)++;
      }
      return 1;
    } else {
      return 0;
    }
  } else {
    return 0;
  }
}

char *
  BIOSEG_PREFIX(_out)(SEG * bioseg)
{
  char *result;
  char *p;

  if (bioseg == NULL)
    return (NULL);

  p = result = (char *) palloc(40);

  if (bioseg->lower == bioseg->upper)
    {
      /*
       * indicates that this interval was built by bioseg_in off a single point
       */
      sprintf(p, "%d", bioseg->lower);
    }
  else
    {
      sprintf(p, "%d..%d", bioseg->lower, bioseg->upper);
    }

  return (result);
}

SEG *
  BIOSEG_PREFIX(_create)(int32 lower, int32 upper)
{
  SEG *result = palloc(sizeof(SEG));

  if (lower > upper) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("lower limit of range greater than upper")));
    return NULL;
  }

  if (lower < MIN_LOWER) {
    ereport(ERROR,
            (errcode(ERRCODE_SYNTAX_ERROR),
             errmsg("bad bioseg representation"),
             errdetail("lower bound (%ld) must be >= %ld",
                       (long int) lower, MIN_LOWER)));
    return 0;
  }

  result->lower = lower;
  result->upper = upper;

  return result;
}



int32
  BIOSEG_PREFIX(_center)(SEG * bioseg)
{
  return ((int32) bioseg->lower + (int32) bioseg->upper) / 2;
}

int32
  BIOSEG_PREFIX(_lower)(SEG * bioseg)
{
  return bioseg->lower;
}

int32
  BIOSEG_PREFIX(_upper)(SEG * bioseg)
{
  return bioseg->upper;
}


/*****************************************************************************
 * GiST functions
 *****************************************************************************/

/*
** The GiST Consistent method for biosegments
** Should return false if for all data items x below entry,
** the predicate x op query == FALSE, where op is the oper
** corresponding to strategy in the pg_amop table.
*/
bool
  BIOSEG_PREFIX(_gist_consistent)(GISTENTRY *entry,
                     SEG * query,
                     StrategyNumber strategy)
{
  /*
   * if entry is not leaf, use bioseg_gist_internal_consistent, else use
   * bioseg_gist_leaf_consistent
   */
  if (GIST_LEAF(entry))
    return (BIOSEG_PREFIX(_gist_leaf_consistent)((SEG *) DatumGetPointer(entry->key), query, strategy));
  else
    return (BIOSEG_PREFIX(_gist_internal_consistent)((SEG *) DatumGetPointer(entry->key), query, strategy));
}

/*
** The GiST Union method for biosegments
** returns the minimal bounding bioseg that encloses all the entries in entryvec
*/
SEG *
  BIOSEG_PREFIX(_gist_union)(GistEntryVector *entryvec, int *sizep)
{
  int  numranges;
  int  i;
  SEG *out = (SEG *) NULL;
  SEG *tmp;

#ifdef GIST_DEBUG
  fprintf(stderr, "union\n");
#endif

  numranges = entryvec->n;
  tmp = (SEG *) DatumGetPointer(entryvec->vector[0].key);
  *sizep = sizeof(SEG);

  for (i = 1; i < numranges; i++)
    {
      out = BIOSEG_PREFIX(_gist_binary_union)(tmp, (SEG *)
                                 DatumGetPointer(entryvec->vector[i].key),
                                 sizep);
      tmp = out;
    }

  return (out);
}

/*
** GiST Compress and Decompress methods for biosegments
** do not do anything.
*/
GISTENTRY *
  BIOSEG_PREFIX(_gist_compress)(GISTENTRY *entry)
{
  return (entry);
}

GISTENTRY *
  BIOSEG_PREFIX(_gist_decompress)(GISTENTRY *entry)
{
  return (entry);
}

/*
** The GiST Penalty method for biosegments
** As in the R-tree paper, we use change in area as our penalty metric
*/
float *
  BIOSEG_PREFIX(_gist_penalty)(GISTENTRY *origentry, GISTENTRY *newentry, float *result)
{
  SEG   *ud;
  SEG   *origrange;
  SEG   *newrange;
  int32 tmp1;
  int32 tmp2;

  origrange = (SEG *) DatumGetPointer(origentry->key);
  newrange = (SEG *) DatumGetPointer(newentry->key);
  ud = BIOSEG_PREFIX(_union)(origrange, newrange);
  BIOSEG_PREFIX(_rt_size)(ud, &tmp1);
  BIOSEG_PREFIX(_rt_size)(origrange, &tmp2);
  if (tmp1 == tmp2)
    {
      BIOSEG_PREFIX(_rt_size)(newrange, &tmp1);
      *result = ((tmp2 - tmp1) * 100000.0) / tmp2;
    }
  else
    {
      *result = tmp1 - tmp2 + 100000.0;
    }

#ifdef GIST_DEBUG
  fprintf(stderr, "Penalty of putting %d..%d into %d..%d is %g\n", newrange->lower, newrange->upper, origrange->lower, origrange->upper, *result);
#endif

  return (result);
}


/*
** The GiST PickSplit method for segments
*/
GIST_SPLITVEC *
  BIOSEG_PREFIX(_gist_picksplit)(GistEntryVector *entryvec,
                    GIST_SPLITVEC *v)
{
  OffsetNumber  i;
  SEG          *datum_alpha;
  SEG          *datum_l;
  SEG          *datum_r;
  int32         lowest;
  int32         highest;
  int32         midpoint;
  int32         split;
  int           nbytes;
  int           firsttime;
  long long int midpointsum;
  int           sumcount;
  OffsetNumber *left;
  OffsetNumber *right;
  OffsetNumber  maxoff;

  maxoff = entryvec->n - 1;
  nbytes = (maxoff + 2) * sizeof(OffsetNumber);
  v->spl_left = (OffsetNumber *) palloc(nbytes);
  v->spl_right = (OffsetNumber *) palloc(nbytes);

  midpointsum = 0;
  sumcount = 0;
  firsttime = true;
  lowest = 0;
  highest = 0;

  for (i = FirstOffsetNumber; i <= maxoff; i = OffsetNumberNext(i))
    {
      datum_alpha = (SEG *) DatumGetPointer(entryvec->vector[i].key);
      midpoint = datum_alpha->lower + ((datum_alpha->upper - datum_alpha->lower) / 2);
      midpointsum += midpoint;
      sumcount++;
      if (firsttime || (midpoint < lowest))
        {
          lowest = midpoint;
        }
      if (firsttime || (midpoint > highest))
        {
          highest = midpoint;
        }
      firsttime = false;
    }

  split = midpointsum / sumcount;
  left = v->spl_left;
  v->spl_nleft = 0;
  right = v->spl_right;
  v->spl_nright = 0;

  datum_l = (SEG *) palloc(sizeof(*datum_l));
  datum_l->lower = lowest;
  datum_l->upper = lowest;
  datum_r = (SEG *) palloc(sizeof(*datum_r));
  datum_r->lower = highest;
  datum_r->upper = highest;

  /*
   * Now split up the regions between the two seeds.
   */

  for (i = FirstOffsetNumber; i <= maxoff; i = OffsetNumberNext(i))
    {
      datum_alpha = (SEG *) DatumGetPointer(entryvec->vector[i].key);
      midpoint = datum_alpha->lower + ((datum_alpha->upper - datum_alpha->lower) / 2);

      /* pick which page to add it to */
      if (midpoint <= split)
        {
          datum_l = BIOSEG_PREFIX(_union)(datum_l, datum_alpha);
          *left++ = i;
          v->spl_nleft++;
        }
      else
        {
          datum_r = BIOSEG_PREFIX(_union)(datum_r, datum_alpha);
          *right++ = i;
          v->spl_nright++;
        }
    }
  *left = *right = FirstOffsetNumber; /* sentinel value, see dosplit() */

  v->spl_ldatum = PointerGetDatum(datum_l);
  v->spl_rdatum = PointerGetDatum(datum_r);

#ifdef GIST_DEBUG
  fprintf(stderr, "Picksplit summary to %d..%d (%d locations) and %d..%d (%d locations), with pivot %d\n", datum_l->lower, datum_l->upper, v->spl_nleft,
datum_r->lower, datum_r->upper, v->spl_nright, split);
#endif

  return v;
}

/*
** Equality methods
*/
bool *
  BIOSEG_PREFIX(_gist_same)(SEG * b1, SEG * b2, bool *result)
{
  if (BIOSEG_PREFIX(_same)(b1, b2))
    *result = TRUE;
  else
    *result = FALSE;

#ifdef GIST_DEBUG
  fprintf(stderr, "same: %s\n", (*result ? "TRUE" : "FALSE"));
#endif

  return (result);
}

/*
** SUPPORT ROUTINES
*/
bool
  BIOSEG_PREFIX(_gist_leaf_consistent)(SEG * key,
                          SEG * query,
                          StrategyNumber strategy)
{
  bool retval;

  switch (strategy)
    {
    case RTLeftStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_left)(key, query);
      break;
    case RTOverLeftStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_over_left)(key, query);
      break;
    case RTOverlapStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_overlap)(key, query);
      break;
    case RTOverRightStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_over_right)(key, query);
      break;
    case RTRightStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_right)(key, query);
      break;
    case RTSameStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_same)(key, query);
      break;
    case RTContainsStrategyNumber:
    case RTOldContainsStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_contains)(key, query);
      break;
    case RTContainedByStrategyNumber:
    case RTOldContainedByStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_contained)(key, query);
      break;
    default:
      retval = FALSE;
    }

#ifdef GIST_QUERY_DEBUG
  if (retval)
    {
      fprintf(stderr, "leaf_consistent, %d..%d %d %d..%d matches\n", query->lower, query->upper, strategy, key->lower, key->upper);
    }
  else
    {
      fprintf(stderr, "leaf_consistent, %d..%d %d %d..%d\n", query->lower, query->upper, strategy, key->lower, key->upper);
    }
#endif

  return (retval);
}

bool
  BIOSEG_PREFIX(_gist_internal_consistent)(SEG * key,
                              SEG * query,
                              StrategyNumber strategy)
{
  bool retval;

  switch (strategy)
    {
    case RTLeftStrategyNumber:
      retval = (bool) !BIOSEG_PREFIX(_over_right)(key, query);
      break;
    case RTOverLeftStrategyNumber:
      retval = (bool) !BIOSEG_PREFIX(_right)(key, query);
      break;
    case RTOverlapStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_overlap)(key, query);
      break;
    case RTOverRightStrategyNumber:
      retval = (bool) !BIOSEG_PREFIX(_left)(key, query);
      break;
    case RTRightStrategyNumber:
      retval = (bool) !BIOSEG_PREFIX(_over_left)(key, query);
      break;
    case RTSameStrategyNumber:
    case RTContainsStrategyNumber:
    case RTOldContainsStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_contains)(key, query);
      break;
    case RTContainedByStrategyNumber:
    case RTOldContainedByStrategyNumber:
      retval = (bool) BIOSEG_PREFIX(_overlap)(key, query);
      break;
    default:
      retval = FALSE;
    }

#ifdef GIST_QUERY_DEBUG
  if (retval)
    {
      fprintf(stderr, "internal_consistent, %d..%d %d %d..%d matches\n", query->lower, query->upper, strategy, key->lower, key->upper);
    }
  else
    {
      fprintf(stderr, "internal_consistent, %d..%d %d %d..%d\n", query->lower, query->upper, strategy, key->lower, key->upper);
    }
#endif

  return (retval);
}

SEG *
  BIOSEG_PREFIX(_gist_binary_union)(SEG * r1, SEG * r2, int *sizep)
{
  SEG *retval;

  retval = BIOSEG_PREFIX(_union)(r1, r2);
  *sizep = sizeof(SEG);

  return (retval);
}


bool
  BIOSEG_PREFIX(_contains)(SEG * a, SEG * b)
{
#ifdef INTERBASE_COORDS
  return ((a->lower <= b->lower) && (a->upper >= b->upper) &&
          a->lower != b->upper && a->upper != b->lower);
#else
  return ((a->lower <= b->lower) && (a->upper >= b->upper));
#endif
}

bool
  BIOSEG_PREFIX(_contained)(SEG * a, SEG * b)
{
  return (BIOSEG_PREFIX(_contains)(b, a));
}

/*****************************************************************************
 * Operator class for R-tree indexing
 *****************************************************************************/

bool
  BIOSEG_PREFIX(_same)(SEG * a, SEG * b)
{
  return BIOSEG_PREFIX(_cmp)(a, b) == 0;
}

/*      bioseg_overlap -- does a overlap b?
 */
bool
  BIOSEG_PREFIX(_overlap)(SEG * a, SEG * b)
{
#ifdef INTERBASE_COORDS
  return (
          ((a->upper >= b->upper) && (a->lower < b->upper) &&
           b->lower != a->upper)
          ||
          ((b->upper >= a->upper) && (b->lower < a->upper) &&
           a->lower != b->upper)
          );
#else
  return
    ((a->upper >= b->upper) && (a->lower <= b->upper))
      ||
    ((b->upper >= a->upper) && (b->lower <= a->upper));
#endif
}

/*      bioseg_overleft -- is the right edge of (a) located at or left of the right edge of (b)?
 */
bool
  BIOSEG_PREFIX(_over_left)(SEG * a, SEG * b)
{
  return (a->upper <= b->upper);
}

/*      bioseg_left -- is (a) entirely on the left of (b)?
 */
bool
  BIOSEG_PREFIX(_left)(SEG * a, SEG * b)
{
  return (a->upper < b->lower);
}

/*      bioseg_right -- is (a) entirely on the right of (b)?
 */
bool
  BIOSEG_PREFIX(_right)(SEG * a, SEG * b)
{
  return (a->lower > b->upper);
}

/*      bioseg_overright -- is the left edge of (a) located at or right of the left edge of (b)?
 */
bool
  BIOSEG_PREFIX(_over_right)(SEG * a, SEG * b)
{
  return (a->lower >= b->lower);
}


static SEG *
  BIOSEG_PREFIX(_union)(SEG * a, SEG * b)
{
  SEG *n;

  n = (SEG *) palloc(sizeof(*n));

  /* take max of upper endpoints */
  if (a->upper > b->upper)
    {
      n->upper = a->upper;
    }
  else
    {
      n->upper = b->upper;
    }

  /* take min of lower endpoints */
  if (a->lower < b->lower)
    {
      n->lower = a->lower;
    }
  else
    {
      n->lower = b->lower;
    }

  return (n);
}

void
  BIOSEG_PREFIX(_rt_size)(SEG * a, int32 *size)
{
  if (a == (SEG *) NULL)
    *size = 0;
  else
#ifdef INTERBASE_COORDS
    *size = (int32) (a->upper - a->lower);
#else
    *size = (int32) (a->upper - a->lower + 1);
#endif
}

int32
  BIOSEG_PREFIX(_size)(SEG * a)
{
#ifdef INTERBASE_COORDS
  return a->upper - a->lower;
#else
  return a->upper - a->lower + 1;
#endif
}


/*****************************************************************************
 *                                 Miscellaneous operators
 *****************************************************************************/
int32
  BIOSEG_PREFIX(_cmp)(SEG * a, SEG * b)
{
  if (a->lower < b->lower)
    return -1;
  if (a->lower > b->lower)
    return 1;

  if (a->upper < b->upper)
    return -1;
  if (a->upper > b->upper)
    return 1;

  return 0;
}

bool
  BIOSEG_PREFIX(_lt)(SEG * a, SEG * b)
{
  return BIOSEG_PREFIX(_cmp)(a, b) < 0;
}

bool
  BIOSEG_PREFIX(_le)(SEG * a, SEG * b)
{
  return BIOSEG_PREFIX(_cmp)(a, b) <= 0;
}

bool
  BIOSEG_PREFIX(_gt)(SEG * a, SEG * b)
{
  return BIOSEG_PREFIX(_cmp)(a, b) > 0;
}

bool
  BIOSEG_PREFIX(_ge)(SEG * a, SEG * b)
{
  return BIOSEG_PREFIX(_cmp)(a, b) >= 0;
}

bool
  BIOSEG_PREFIX(_different)(SEG * a, SEG * b)
{
  return BIOSEG_PREFIX(_cmp)(a, b) != 0;
}

bool
  BIOSEG_PREFIX(_contains_int)(SEG * a, int *b)
{
#ifdef INTERBASE_COORDS
  return ((a->lower < *b) && (a->upper > *b));
#else
  return ((a->lower <= *b) && (a->upper >= *b));
#endif
}

/*
** These are lower than those in geo_selfuncs.c - found by trail and error.
*/
Datum
  BIOSEG_PREFIX(_sel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(2.0e-4);
}

Datum
  BIOSEG_PREFIX(_joinsel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(1.0e-5);
}

Datum
  BIOSEG_PREFIX(_contsel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(1.0e-4);
}

Datum
  BIOSEG_PREFIX(_contjoinsel)(PG_FUNCTION_ARGS)
{
  PG_RETURN_FLOAT8(5e-6);
}
