#!/usr/bin/perl -w

use strict;


#******************************************************************************
#  lscel.pl : list cell files with chiptype detail
#
#  Author:  Astrid Deschênes <astrid.deschenes@crchul.ulaval.ca>,
#           Copyright (C) 2005, Quebec Genomic Center
#
#******************************************************************************
#
#                            PUBLIC DOMAIN NOTICE
#                            Quebec Genomic Center
#
#   This software/database is freely available to the public for use or
#   reproduction.
#
#   Although all reasonable efforts have been taken to ensure the accuracy
#   and reliability of the software and data, the Quebec Genomic Center
#   do not and cannot warrant the performance or results that may be obtained
#   by using this software or data.
#   The Quebec Genomic Center disclaim all warranties, express or implied,
#   including warranties of performance, merchantability or fitness for
#   any particular purpose.
#
#   Please cite the author in any work or product based on this material.
#
#******************************************************************************
#   using sample on bash prompt
#
#   s1. list cel under current dir and subdir, sorted by chiptype :
#   > lscel.sh | sort
#
#   s2. list cel under current dir and subdir, which have HG-U133_Plus_2 in 
#       chiptype and write it in a file :
#   > lscel.sh | grep HG-U133_Plus_2 > HG-U133_Plus_2.txt
#
#   s3. list cel under current dir and subdir, which have HG-U133_Plus_2 in 
#       chiptype write it in a file sorted by CEL full filename :
#   > lscel.sh | grep HG-U133_Plus_2 | sort  > HG-U133_Plus_2.txt
#
#******************************************************************************

{
    my @celList;    # array which will containt the cel list
    &list_CEL('.', \@celList); # populate the cel list
    
    # Foreach cel, get the celtype using CELUTIL lib
    # Print the celtype and cel name to the standard output 
    (scalar @celList > 0) && printf("%-20s %-50s\n", 'CHIPTYPE', 'CEL');
    foreach my $oneFile (sort @celList) {
        chomp $oneFile;
        # System call using CELUTIL lib to get the celtype of a cel file
        my $result = readpipe("celutil \'$oneFile\' qry -Dt 2>/dev/null");
        if ($? == 0) {
            chomp $result;
            $result =~ s/\x0d//g; # Erase the Window Special Caracter
            $result =~ s/\t$//;   # Erase the TAB
            printf("%-20s %-50s\n", "$result", "$oneFile"); 
        }
    }

    exit 0;
}

##################################################################
# Recursive directory reading
# to populate an array reference with cel filenames
# Parameters :
#   1- $dir : directory name
#   2- $refCelList : array reference to a cel list
##################################################################
sub list_CEL {
    my $dir = shift;            # directory name
    my $refCelList = shift;     # array reference to a cel list

    # Read directory's content
    opendir DIR, $dir or return;
    my @contents = map "$dir/$_", sort grep !/^\.\.?$/, readdir DIR;
    closedir DIR;

    # Recursive call of the function foreach subdirectory
    # Populate the list with filenames corresponding to cel files
    foreach (@contents) {
        if (!-l && -d ) {
            &list_CEL($_, $refCelList);  # Recursive call for a subdirectory
        } elsif ($_ =~ /\.CEL.*$/) {
            push @{$refCelList}, $_ if (-f); # Add the cel filename to the list
        }
    }
    return 0;
}