/*
________________________________________________________________________________

  cpp Implementation: CELFile, 2005
________________________________________________________________________________

                            PUBLIC DOMAIN NOTICE
                            Quebec Genomic Center

  This software/database is freely available to the public for use or
  reproduction.

  Although all reasonable efforts have been taken to ensure the accuracy
  and reliability of the software and data, the Quebec Genomic Center
  do not and cannot warrant the performance or results that may be obtained
  by using this software or data.
  The Quebec Genomic Center disclaim all warranties, express or implied,
  including warranties of performance, merchantability or fitness for
  any particular purpose.

  Please cite the author in any work or product based on this material.
________________________________________________________________________________

 Author:  david paladini <david.paladini.bioinfo@gmail.com>,
          Copyright (C) 2005, Quebec functional Genomic Center

 File Description: celFile utility class implant :
                   - equality operation between to CELFile class
                   - Based on CEl File Parser class CCELFileData of Affy, add
                     writing capabilities in both ASCII and XDA (binary) CEL
                     File format.
________________________________________________________________________________
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <istream>
#include <fstream>

#ifndef WIN32
#include <sys/mman.h>
#endif

#ifdef _INCLUDE_UNISTD_HEADER_
#include <unistd.h>
#endif

#include "CELFile.h"
#include "CELFileData.h"
#include "FileIO.h"

#define CELL_FILE_MAGIC_NUMBER 64
#define CELL_FILE_VERSION_NUMBER 4
#define CELL_TEXT_FILE_VERSION_NUMBER 3

//______________________________________________________________________________

using namespace affxcel_ext;
using namespace affxcel;
using namespace std;
#define ENDL "\r\n"
int CCELFile::CheckDosfile()
{
  // replace '\r' by '\r\n'
  string header = m_HeaderData.GetHeader();
  char* start = (char*)header.c_str();
  char* cur   = start;
  int i = 0;
  while ((cur= strchr(cur,'\r'))!=NULL)
  {
    if ( i == 0 )
    {
      if (cur[1] == '\n' ) return true;
    }
    header.replace(((cur + i++)-start), 1,ENDL);
    cur++;
  }
  m_HeaderData.SetHeader(header.c_str());

  return false;
}
//______________________________________________________________________________

bool CCELFile::WriteProp(ostream &_OStr,
                         ePropFmt _format,
                         bool _GetHeader,
                         bool _GetVersion,
                         bool _GetCols,
                         bool _GetRows,
                         bool _GetNumCells,
                         bool _GetAlg,
                         bool _GetParams,
                         bool _GetChipType,
                         bool _GetCellMargin,
                         bool _GetNumOutliers,
                         bool _GetNumMasked,
                         bool _GetParameters,
                         bool _GridCorners,
                         bool _GetDatHeader)
{
  char tabsep = '\t';
  if( _format == eList )
  {
    if ( _GetHeader      )
    {
      _OStr << "Header="              << m_HeaderData.GetHeader()      << ENDL;
    }
    else
    {
      if ( _GetVersion     ) _OStr << "Version="                << m_HeaderData.GetVersion()     << ENDL;
      if ( _GetCols        ) _OStr << "Cols="                   << m_HeaderData.GetCols()        << ENDL;
      if ( _GetRows        ) _OStr << "Rows="                   << m_HeaderData.GetRows()        << ENDL;
      if ( _GetAlg         ) _OStr << "Algorithm="              << m_HeaderData.GetAlg()         << ENDL;
      if ( _GetParams      ) _OStr << "AlgorithmParameters="    << m_HeaderData.GetParams()      << ENDL;
      if ( _GetChipType    ) _OStr << "ChipType="               << m_HeaderData.GetChipType()    << ENDL;
      if ( _GetNumCells    ) _OStr << "IntensitiesNumberCells=" << m_HeaderData.GetNumCells()    << ENDL;
      if ( _GetCellMargin  ) _OStr << "CellMargin="             << m_HeaderData.GetCellMargin()  << ENDL;
      if ( _GetNumOutliers ) _OStr << "OutliersNumberCells="    << m_HeaderData.GetNumOutliers() << ENDL;
      if ( _GetNumMasked   ) _OStr << "MaskedNumberCells="      << m_HeaderData.GetNumMasked()   << ENDL;
      if ( _GetParameters  )
      {
//        _OStr << "Parameters="      << m_HeaderData.GetParameters()   << ENDL;
      }
      if ( _GridCorners    )
      {
        GridCoordinatesType &corners = m_HeaderData.GridCorners();
        _OStr << "GridCornerUL=" << corners.upperleft.x  << " " << corners.upperleft.y  << ENDL;
        _OStr << "GridCornerUR=" << corners.upperright.x << " " << corners.upperright.y << ENDL;
        _OStr << "GridCornerLR=" << corners.lowerright.x << " " << corners.lowerright.y << ENDL;
        _OStr << "GridCornerLL=" << corners.lowerleft.x  << " " << corners.lowerleft.y  << ENDL;
      }
      if ( _GetDatHeader  )  _OStr << "DatHeader="      << m_HeaderData.GetDatHeader()   << ENDL;
    }
  }
  else
  {
    if ( ( _format == eTabTitleOnly ) || ( _format == eTabTitleAndData ) )
    {
      if ( _GetVersion     ) _OStr << "Version"                << tabsep;
      if ( _GetCols        ) _OStr << "Cols"                   << tabsep;
      if ( _GetRows        ) _OStr << "Rows"                   << tabsep;
      if ( _GetNumCells    ) _OStr << "IntensitiesNumberCells" << tabsep;
      if ( _GetAlg         ) _OStr << "Algorithm"              << tabsep;
      if ( _GetParams      ) _OStr << "AlgorithmParameters"    << tabsep;
      if ( _GetChipType    ) _OStr << "ChipType"               << tabsep;
      if ( _GetCellMargin  ) _OStr << "CellMargin"             << tabsep;
      if ( _GetNumOutliers ) _OStr << "OutliersNumberCells"    << tabsep;
      if ( _GetNumMasked   ) _OStr << "MaskedNumberCells"      << tabsep;
      if ( _GridCorners    )
      {
        GridCoordinatesType &corners = m_HeaderData.GridCorners();
        _OStr << "GridCornerUL" << tabsep;
        _OStr << "GridCornerUR" << tabsep;
        _OStr << "GridCornerLR" << tabsep;
        _OStr << "GridCornerLL" << tabsep;
      }
      if ( _GetDatHeader  )  _OStr << "DatHeader" << tabsep;
      _OStr << ENDL;
    }
    if ( ( _format == eTabDataOnly ) || ( _format == eTabTitleAndData ) )
    {
      if ( _GetVersion     ) _OStr << m_HeaderData.GetVersion()     << tabsep;
      if ( _GetCols        ) _OStr << m_HeaderData.GetCols()        << tabsep;
      if ( _GetRows        ) _OStr << m_HeaderData.GetRows()        << tabsep;
      if ( _GetNumCells    ) _OStr << m_HeaderData.GetNumCells()    << tabsep;
      if ( _GetAlg         ) _OStr << m_HeaderData.GetAlg()         << tabsep;
      if ( _GetParams      ) _OStr << m_HeaderData.GetParams()      << tabsep;
      if ( _GetChipType    ) _OStr << m_HeaderData.GetChipType()    << tabsep;
      if ( _GetCellMargin  ) _OStr << m_HeaderData.GetCellMargin()  << tabsep;
      if ( _GetNumOutliers ) _OStr << m_HeaderData.GetNumOutliers() << tabsep;
      if ( _GetNumMasked   ) _OStr << m_HeaderData.GetNumMasked()   << tabsep;
      if ( _GridCorners    )
      {
        GridCoordinatesType &corners = m_HeaderData.GridCorners();
        _OStr << corners.upperleft.x  << " " << corners.upperleft.y  << tabsep;
        _OStr << corners.upperright.x << " " << corners.upperright.y << tabsep;
        _OStr << corners.lowerright.x << " " << corners.lowerright.y << tabsep;
        _OStr << corners.lowerleft.x  << " " << corners.lowerleft.y  << tabsep;
      }
      if ( _GetDatHeader  )  _OStr << m_HeaderData.GetDatHeader()   << tabsep;
      _OStr << ENDL;
    }
  }
}

//______________________________________________________________________________

bool CCELFile::WriteIntensityProp(ostream &_OStr)
{
   return WriteTextIntensityData(_OStr);
}

//______________________________________________________________________________

bool CCELFile::Write(ostream &_OStr, int _XdaFormat)
{
    CheckDosfile();
    if (_XdaFormat) // write binary CEL file
    {
        // Write the header, intensity data and footer
        WriteXDAHeader(_OStr);
        WriteXDAIntensityData(_OStr);
        WriteXDAFooter(_OStr);
    }
    else  // write ASCII CEL file
    {
        // Write the header, intensity data and footer
        WriteTextHeader(_OStr);
        WriteTextIntensityData(_OStr);
        WriteTextFooter(_OStr);
    }
    return true;
}

//______________________________________________________________________________

bool CCELFile::WriteTextHeader(std::ostream &_OStr)
{
    // write the CEL section info
    _OStr << "[CEL]" << ENDL;
    _OStr << "Version=" << CELL_TEXT_FILE_VERSION_NUMBER << ENDL;
    // write the HEADER section info
    _OStr << ENDL << "[HEADER]" << ENDL;
    _OStr << m_HeaderData.GetHeader().c_str() << ENDL;
    return true;
}

//______________________________________________________________________________

bool CCELFile::WriteTextIntensityData(std::ostream &_OStr)
{
    CELFileEntryType entry;
    int headerNumCells = m_HeaderData.GetNumCells();

    // write the INTENSITY section info
    _OStr << "[INTENSITY]" << ENDL;
    _OStr << "NumberCells=" << headerNumCells << ENDL;
    _OStr << "CellHeader=X\tY\tMEAN\tSTDV\tNPIXELS" << ENDL;

    // write the INTENSITY data
    for (int icel=0; icel<headerNumCells; icel++)
    {
        char buffer[255];
        GetEntry(icel, entry);
        sprintf ( buffer,
                  "%3d\t%3d\t%.1f\t%.1f\t%3d",
                  IndexToX(icel),
                  IndexToY(icel),
                  entry.Intensity,
                  entry.Stdv,
                  entry.Pixels );
        _OStr << buffer << ENDL;
    }

    return true;
}

//______________________________________________________________________________

bool CCELFile::WriteTextFooter(std::ostream &_OStr)
{
    // write the footer.
    int headerNumCells = m_HeaderData.GetNumCells();

    // write MASKS section info
    _OStr  << ENDL << "[MASKS]" << ENDL;
    _OStr  << "NumberCells=" << m_HeaderData.GetNumMasked() << ENDL;
    _OStr  << "CellHeader=X\tY" << ENDL;
    // write mask data
    for (int icel=0; icel<headerNumCells; icel++)
    {
        if (IsMasked(icel) == true)
        {
            _OStr << IndexToX(icel) << "\t" << IndexToY(icel) << ENDL;
        }
    }

    // write the OUTLIERS section info
    _OStr  << ENDL << "[OUTLIERS]" << ENDL;
    _OStr  << "NumberCells=" << m_HeaderData.GetNumOutliers() << ENDL;
    _OStr  << "CellHeader=X\tY" << ENDL;
    // write the outliers data
    for (int icel=0; icel<headerNumCells; icel++)
    {
        if (IsOutlier(icel) == true)
        {
            _OStr << IndexToX(icel) << "\t" << IndexToY(icel) << ENDL;
        }
    }
    // The "MODIFIED" section specifies which cells were modified by the user.
    // This feature was dropped in MAS 4 thus the number of cells in this
    // section should always be 0. The TAGS are:
    _OStr  << ENDL << "[MODIFIED]" << ENDL;
    _OStr  << "NumberCells=0" << ENDL;
    _OStr  << "CellHeader=X\tY\tORIGMEAN" << ENDL;

    return true;
}

//______________________________________________________________________________

bool CCELFile::WriteXDAHeader(std::ostream &_OStr)
{
    int magic     = CELL_FILE_MAGIC_NUMBER;
    int version   = CELL_FILE_VERSION_NUMBER;
    int nSubGrids = 0;

    // write the magic number.
    CCELFileIO::WRITE_INT    (_OStr, magic                         );
    // write the version
    CCELFileIO::WRITE_INT    (_OStr, version                       );

    // write the dimensions of the array
    CCELFileIO::WRITE_INT    (_OStr, m_HeaderData.GetRows()        );
    CCELFileIO::WRITE_INT    (_OStr, m_HeaderData.GetCols()        );
    CCELFileIO::WRITE_INT    (_OStr, m_HeaderData.GetNumCells()    );

    // write the other members.
    CCELFileIO::WRITE_STRING (_OStr, m_HeaderData.GetHeader()      );
    CCELFileIO::WRITE_STRING (_OStr, m_HeaderData.GetAlg()         );
    CCELFileIO::WRITE_STRING (_OStr, m_HeaderData.GetParams()      );

    CCELFileIO::WRITE_INT    (_OStr, m_HeaderData.GetCellMargin()  );
    CCELFileIO::WRITE_UINT   (_OStr, m_HeaderData.GetNumOutliers() );
    CCELFileIO::WRITE_UINT   (_OStr, m_HeaderData.GetNumMasked()   );

    CCELFileIO::WRITE_INT    (_OStr, nSubGrids                     );

    return true;
}

//______________________________________________________________________________

bool CCELFile::WriteXDAIntensityData(std::ostream &_OStr)
{
    int headerNumCells = m_HeaderData.GetNumCells();
    CELFileEntryType entry;

    // write the intensity data
    for (int icel=0; icel<headerNumCells; icel++)
    {
        GetEntry(icel, entry);
        CCELFileIO::WRITE_FLOAT (_OStr, entry.Intensity);
        CCELFileIO::WRITE_FLOAT (_OStr, entry.Stdv     );
        CCELFileIO::WRITE_SHORT (_OStr, entry.Pixels   );
    }

    return true;
}

//______________________________________________________________________________

bool CCELFile::WriteXDAFooter(std::ostream &_OStr)
{
    int headerNumCells = m_HeaderData.GetNumCells();

    // write the masked data
    for (int icel=0; icel<headerNumCells; icel++)
    {
        if (IsMasked(icel) == true)
        {
            // write the coordinate.
            CCELFileIO::WRITE_SHORT (_OStr, IndexToX(icel));
            CCELFileIO::WRITE_SHORT (_OStr, IndexToY(icel));
        }
    }

    // write the outlier data
    for (int icel=0; icel<headerNumCells; icel++)
    {
        if (IsOutlier(icel) == true)
        {
            // write the coordinate.
            CCELFileIO::WRITE_SHORT (_OStr, IndexToX(icel));
            CCELFileIO::WRITE_SHORT (_OStr, IndexToY(icel));
        }
    }

    return true;
}

//______________________________________________________________________________

void CCELFileIO::WRITE_INT(std::ostream &_OStr, int _Value)
{
    _OStr.write((char*)&_Value, INT_SIZE);
}

//______________________________________________________________________________

void CCELFileIO::WRITE_UINT(std::ostream &_OStr, unsigned int _Value)
{
    _OStr.write((char*)&_Value, UINT_SIZE);
}

//______________________________________________________________________________

void CCELFileIO::WRITE_SHORT(std::ostream &_OStr, unsigned short _Value)
{
    _OStr.write((char*)&_Value, SHORT_SIZE);
}

//______________________________________________________________________________

void CCELFileIO::WRITE_FLOAT(std::ostream &_OStr, float _Value)
{
    _OStr.write((char*)&_Value, FLOAT_SIZE);
}

//______________________________________________________________________________

void CCELFileIO::WRITE_STRING(std::ostream &_OStr, std::string _Value)
{
    WRITE_INT(_OStr,_Value.size());
    _OStr.write(_Value.c_str(), _Value.size());
}

//______________________________________________________________________________
