/*
________________________________________________________________________________

  cpp Implementation: CELUtilApp, 2005
________________________________________________________________________________

                            PUBLIC DOMAIN NOTICE
                            Quebec Genomic Center

  This software/database is freely available to the public for use or
  reproduction.

  Although all reasonable efforts have been taken to ensure the accuracy
  and reliability of the software and data, the Quebec Genomic Center
  do not and cannot warrant the performance or results that may be obtained
  by using this software or data.
  The Quebec Genomic Center disclaim all warranties, express or implied,
  including warranties of performance, merchantability or fitness for
  any particular purpose.

  Please cite the author in any work or product based on this material.
________________________________________________________________________________

 Author:  david paladini <david.paladini.bioinfo@gmail.com>,
          Copyright (C) 2005, Quebec functional Genomic Center

 File Description: celUtilApp class implant application controler to read,
                   compare and convert CEL files
________________________________________________________________________________
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <stdio.h>
#include <cstdlib>

#include "CELUtilApp.h"

using namespace affxcel_ext;
using namespace std;
//______________________________________________________________________________

int CCELUtilApp::PrintHelp()
{
  // write celutil usage
  std::cerr << "Usage: " << m_Runable << " (-OPTION)|(FILE [[COMMAND][([FILE][-OPTION]...)|([-OPTION]...)]])\n";
  std::cerr << "-h                print this help                           \n";
  std::cerr << "--help            print this help                           \n";
  std::cerr << "[COMMAND] no command                                        \n";
  std::cerr << "Usage: " << m_Runable << " FILE                             \n";
  std::cerr << "                  read FILE and print it to stdout          \n";
  std::cerr << "                                                            \n";
  std::cerr << "[COMMAND] cmp                                               \n";
  std::cerr << "Usage: " << m_Runable << " FILE cmp FILE                    \n";
  std::cerr << "cmp               compare CEL file. no option               \n";
  std::cerr << "                  return 0 if matrices are identical.       \n";
  std::cerr << "                  Otherwise return 1                        \n";
  std::cerr << "                                                            \n";
  std::cerr << "[COMMAND] cnv                                               \n";
  std::cerr << "Usage: " << m_Runable << " FILE cnv FILE                    \n";
  std::cerr << "cnv               convert CEL file. default write ASCII     \n";
  std::cerr << "                  format uncompressed output file           \n";
  std::cerr << "[-OPTION]...                                                \n";
  std::cerr << "-x                write XDA (bynary) format output file     \n";
  std::cerr << "-c                compress output file                      \n";
  std::cerr << "                                                            \n";
  std::cerr << "[COMMAND] qry                                               \n";
  std::cerr << "Usage: " << m_Runable << " FILE qry [ ([FILE][-OPTION]...)|([-OPTION]...) ] \n";
  std::cerr << "qry               query CEL file properties                 \n";
  std::cerr << "[FILE]                                                      \n";
  std::cerr << "                  Append queried properties to the file     \n";
  std::cerr << "[-OPTION]...                                                \n";
  std::cerr << "output format                                               \n";
  std::cerr << "-l                write properties in list format. default format\n";
  std::cerr << "-T                writes properties name only in tabbed format\n";
  std::cerr << "-D                writes properties data only in tabbed format\n";
  std::cerr << "-d                writes properties name and data in tabbed format\n";
  std::cerr << "prop                                                        \n";
  std::cerr << "-W                write complete Header. Available only the \n";
  std::cerr << "                  for list format (see -l option)           \n";
  std::cerr << "-v                write Version                             \n";
  std::cerr << "-c                write Cols count                          \n";
  std::cerr << "-r                write Rows count                          \n";
  std::cerr << "-C                write Cells count                         \n";
  std::cerr << "-a                write Algo                                \n";
  std::cerr << "-p                write Params                              \n";
  std::cerr << "-t                write Chip Type                           \n";
  std::cerr << "-m                write Cell Margin                         \n";
  std::cerr << "-O                write Outliers number                     \n";
  std::cerr << "-M                write Masked number                       \n";
  std::cerr << "-P                write Parameters                          \n";
  std::cerr << "-g                write Grid Corners coordonates            \n";
  std::cerr << "-w                write Dat Header                          \n";
  std::cerr << "-i                write intensity section. This option      \n";
  std::cerr << "                  invalid other properties options set.     \n";
  std::cerr << "                                                            \n";
  std::cerr << "samples :                                                   \n";
  std::cerr << m_Runable << " myfile.CEL                                    \n";
  std::cerr << "                  read myfile.CEL and print it to stdout    \n";
  std::cerr << "                                                            \n";
  std::cerr << m_Runable << " myfile.CEL cnv celfile -xc                    \n";
  std::cerr << "                  convert myfile.CEL to XDA compressed file \n";
  std::cerr << "                  celfile.CEL.gz                            \n";
  std::cerr << "                                                            \n";
  std::cerr << m_Runable << " myfile.CEL cmp celfile.CEL.gz                 \n";
  std::cerr << "                  compare intensities matrices of myfile.CEL\n";
  std::cerr << "                  and celfile.CEL.gz                        \n";
  std::cerr << "                                                            \n";
  std::cerr << m_Runable << " myfile.CEL qry -t                             \n";
  std::cerr << "                  retrieves chiptype in CEL File header and \n";
  std::cerr << "                  print it to stdout in list format         \n";
  std::cerr << "                                                            \n";
  std::cerr << m_Runable << " myfile.CEL qry celfile.tab -Dt                \n";
  std::cerr << "                  retrieves chiptype in CEL File header and \n";
  std::cerr << "                  append it to celfile.tab in tab format    \n";
  std::cerr << "                  without properties names                  \n";
  return true;
}

//______________________________________________________________________________

int CCELUtilApp::Failure(string _Msg,int _PrintHelp)
{
  if(!_Msg.empty())
  {
    cerr << _Msg << endl;
  }
  if(_PrintHelp)
  {
    PrintHelp();
  }
  return false;
}

//______________________________________________________________________________

int CCELUtilApp::IsOptionSet (char _Option)
{
  return (m_Option.empty())?false:(strchr(m_Option.c_str(),_Option)!=NULL);
}

//______________________________________________________________________________

int CCELUtilApp::ReadCel(string _FileName, CCELFile& _Cel, bool _ReadHeaderOnly)
{
  // set filename to read
  _Cel.SetFileName(_FileName.c_str());

  // read file if exist
  if ( ! _Cel.Exists() )
    return Failure("File " + _Cel.GetFilePath() + " not found");

  if (_ReadHeaderOnly)
    return _Cel.ReadHeader();
  else return _Cel.Read();
}

//______________________________________________________________________________

int CCELUtilApp::WriteCel(string _FileName,
                          CCELFile& _Cel,
                          int _XdaFormat,
                          int _Compressed)
{
  ostream *ostr;

  // select/open output stream
  if ((_FileName=="-")||(_FileName==""))
  {
    ostr = &cout;
  }
  else
  {
    if (_Compressed)
    {
      ostr = (ostream *)new ogzstream((_FileName + ".CEL.gz").c_str());
    }
    else
    {
      ostr = (ostream *)new ofstream((_FileName + ".CEL" ).c_str());
    }
    if (ostr->fail())
      return Failure("write " + _Cel.GetFilePath() + " error");
  }

  // write cel file to output
  _Cel.Write(*ostr,_XdaFormat);

  // release output stream
  if (!((_FileName=="-")||(_FileName=="")))
  {
    if (_Compressed)
    {
      ((ogzstream*)ostr)->close();
    }
    else
    {
      ((ofstream*)ostr)->close();
    }
    delete ostr;
  }

  return true;
}

//______________________________________________________________________________

int CCELUtilApp::WriteCelProp( std::string _FileName,
                               affxcel_ext::CCELFile& _Cel,
                               affxcel_ext::ePropFmt _format,
                               bool _GetHeader,
                               bool _GetVersion,
                               bool _GetCols,
                               bool _GetRows,
                               bool _GetNumCells,
                               bool _GetAlg,
                               bool _GetParams,
                               bool _GetChipType,
                               bool _GetCellMargin,
                               bool _GetNumOutliers,
                               bool _GetNumMasked,
                               bool _GetParameters,
                               bool _GridCorners,
                               bool _GetDatHeader,
                               bool _GetIntensities )
{
  ostream *ostr;

  // select/open output stream
  if ((_FileName=="-")||(_FileName==""))
  {
    ostr = &cout;
  }
  else
  {
    ostr = (ostream *)new ofstream((_FileName).c_str(), ios_base::app);
  }

  if( _GetIntensities )
  {
    _Cel.WriteIntensityProp(*ostr);
  }
  else
  {
    // write cel file to output
    _Cel.WriteProp(*ostr,
                    _format,
                    _GetHeader,
                    _GetVersion,
                    _GetCols,
                    _GetRows,
                    _GetNumCells,
                    _GetAlg,
                    _GetParams,
                    _GetChipType,
                    _GetCellMargin,
                    _GetNumOutliers,
                    _GetNumMasked,
                    _GetParameters,
                    _GridCorners,
                    _GetDatHeader);
  }

  // release output stream
  if (!((_FileName=="-")||(_FileName=="")))
  {
   ((ofstream*)ostr)->close();
    delete ostr;
  }

  return true;
}


//______________________________________________________________________________

int CCELUtilApp::CmdRead()
{
  CCELFile cel;
  if ( !ReadCel(m_FileNameA, cel) )
  {
    return Failure("reading error");
  }
  if( !WriteCel("-", cel, false, false) )
  {
    return Failure("writing error");
  }

  return true;
}

//______________________________________________________________________________

int CCELUtilApp::CmdCompare()
{
  CCELFile celA;
  CCELFile celB;
  // read files to compare
  if (!ReadCel(m_FileNameA, celA))
  {
    return Failure("reading error");
  }
  if (!ReadCel(m_FileNameB, celB))
  {
    return Failure("reading error");
  }
  // evaluation of intensity values
  cout << (celA == celB) << endl;

  return true;
}

//______________________________________________________________________________

int CCELUtilApp::CmdConvert()
{
  CCELFile cel;
  if (!ReadCel(m_FileNameA, cel))
  {
    return Failure("reading error");
  }
  if ( !WriteCel ( m_FileNameB, cel, IsOptionSet('x'), IsOptionSet('c') ) )
  {
    return Failure("writing error");
  }
}

//______________________________________________________________________________

int CCELUtilApp::CmdQuery ()
{
  CCELFile cel;
  bool readpartial = !IsOptionSet('i');

  if (!ReadCel(m_FileNameA, cel, readpartial))
  {
    return Failure("reading error");
  }
  ePropFmt PropFmt = affxcel_ext::eList;
  if ( IsOptionSet('l') ) PropFmt = affxcel_ext::eList;
  if ( IsOptionSet('T') ) PropFmt = affxcel_ext::eTabTitleOnly;
  if ( IsOptionSet('D') ) PropFmt = affxcel_ext::eTabDataOnly;
  if ( IsOptionSet('d') ) PropFmt = affxcel_ext::eTabTitleAndData;

  if (!WriteCelProp(m_FileNameB,
                    cel,
                    PropFmt,
                    IsOptionSet('W'),  // GetHeader
                    IsOptionSet('v'),  // GetVersion
                    IsOptionSet('c'),  // GetCols
                    IsOptionSet('r'),  // GetRows
                    IsOptionSet('C'),  // GetNumCells
                    IsOptionSet('a'),  // GetAlg
                    IsOptionSet('p'),  // GetParams
                    IsOptionSet('t'),  // GetChipType
                    IsOptionSet('m'),  // GetCellMargin
                    IsOptionSet('O'),  // GetNumOutliers
                    IsOptionSet('M'),  // GetNumMasked
                    IsOptionSet('P'),  // GetParameters
                    IsOptionSet('g'),  // GridCorners
                    IsOptionSet('w'),  // GetDatHeader
                    IsOptionSet('i')))                                                // GetIntensities
  {
    return Failure("writing error");
  }
}

//______________________________________________________________________________

int CCELUtilApp::Init(int argc, char *argv[])
{
  argv[0]!=NULL ? m_Runable   = argv[0] : m_Runable   = "";

  // parse command line
  if ( argc < 2) return Failure("wrong number of arguments",true);

  // init
  argv[1]!=NULL ? m_FileNameA = argv[1] : m_FileNameA = "";
  if (m_FileNameA.c_str()[0]=='-')
  {
    m_Command = eHelp;
    return true;
  }
  if (m_FileNameA.find("/", 0) == string::npos) m_FileNameA.insert(0, "./");

  argv[3]!=NULL ? m_FileNameB = argv[3] : m_FileNameB = "";
  argv[4]!=NULL ? m_Option    = argv[4] : m_Option    = "";

  if (m_FileNameB.c_str()[0]=='-')
  {
    m_Option = m_FileNameB;
    m_FileNameB = "";
  }
  else
  {
    if (m_FileNameB.find("/", 0) == string::npos) m_FileNameB.insert(0, "./");
  }
  m_Command   = eRead;

  if (argc>2) // read cel file
  {
    // 4 or 5 argument expected
    if ((argc<4)&&(argc>5)) return Failure("wrong number of arguments",true);
    // command parse
    if (!strcmp(argv[2],"cmp")) m_Command = eCompare;
    else if (!strcmp(argv[2],"cnv")) m_Command = eConvert;
    else if (!strcmp(argv[2],"qry")) m_Command=eQuery;
    else return Failure("wrong command arguments",true);
  }
}

//______________________________________________________________________________

int CCELUtilApp::Run()
{
  switch(m_Command)
  {
    case eHelp    : return PrintHelp ();
    case eCompare : return CmdCompare();
    case eConvert : return CmdConvert();
    case eQuery   : return CmdQuery  ();
    default       : return CmdRead   ();
  }
}

