/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef NODELIST_H
#define NODELIST_H

#include <vector>
#include <iostream>
#include <iomanip>
#include <string>
#include <time.h>
#include <stdexcept>
#include <fstream>
#include <algorithm>
#include <functional>
#include "Node.h"
#include "lib/StringUtils.h"

void ipause();

	/**
	 * The node item is an object that keeps track of a node for the nodelist
	 */
class NodeItem {
public:
    //default constructor
    NodeItem();
    //copy constructor
    NodeItem(const NodeItem &);
    //destructor
    ~NodeItem();
    //assignment operator
    NodeItem & operator = (const NodeItem &);
    void setName(string n);
    void setNode(const Node &);
    void setNode(const InputNode &);
    void setNode(const OutputNode &);
    void setNode(const BooleanNode &);
    void setNode(const DelayNode &);
    void setNode(const SustainNode &);
    bool removeNode();
    string getName() const {return name;};
    Node & getNode();
    bool hasNode() const;
protected:
    string name;
private:
    Node *node;
};


	/**
	 * The Nodelist tracks and maintains the nodes within the network
	 */
class NodeList {
public:
    NodeList();
    friend istream &operator >>(istream &in, NodeList &nl);
    void visPrint(ostream &out, NodeList &nl);
    void bitsPrint(NodeList &nl);
    int find(string n);
    int add(NodeItem &i);
    Node &getNode(int i) {return nodes[i].getNode();};
    NodeItem &getNodeItem(int i) { return nodes[i];}
    Node &getInputNode(int i) {return inputs[i].getNode();};
    string getName(int i) {return nodes[i].getName();};
    NodeList &setRunTime(int r) {runTime = r; return *this;};
    int getRunTime() {return runTime;};
    string getRunName() {return runName;};
    NodeList &setTransient(int r) {transient = r; return *this;};
    int getTransient() {return transient;};
    NodeList &setWindow( int w) { window = w; return *this;};
    NodeList &setScalar( int s) { scalar = s; return *this;};
    int getWindow(void) {return window;};
    int getScalar(void) {return scalar;};
    int getNextAnalPt( void );
    void sortAnalPts( void );
    NodeList &printNames(ostream &out);
    void printNodes(ostream &out);
    void printInputs(ostream &out);
    vector<NodeItem> outputs;
    vector<NodeItem> outCalc;
    int inOutCalc( string );
    int getNodeIndex( string );
    vector <float> avg; // Gong added in 11/11/05 to compute the average value for each Boolean node
    vector < vector<int> > bits;
    int outCalcSize() {return static_cast<int>(outCalc.size());}
    int getNumOfInputs() {return inputs.size();}
    int outputSize() {return static_cast<int>(outputs.size());};
    int ptsSize() { return analPts.size(); };//rename this more appropriately
    void printAnalPts( void );
    void setMutants( void );
    void setRunName( const string & str ) { runName = str; };
    void calcAvg( void );
    void printAvg( ostream &);
    void calcCurrAvg( void );
    void printAvgSep( const string & );
    void printDosages( ostream &  );
    void printBits( ostream & );
    void printInputBits( ostream & );
    void initBits( bool);
    void initInputBits( );
    void setNoisyInputs(void) { noisyInputs = true; };
    bool isNoisyInputs(void) { return noisyInputs; };
	void addAnalPt( const int & );
    vector<int> dosageRec;
	vector<int> analPts;
	int avgPeriod;
    int nodeSize() {return static_cast<int>(nodes.size());};
protected:
    vector < vector<int> > inputBits;
    void addOutput(NodeItem &i);
    void addInput(NodeItem &i);
    void addOutCalc(NodeItem &i);// Gong added to compute the average
    void addNode(NodeItem &i);
    void addMutant( Mutant & );
    vector<Mutant> mutants;//tracks mutant nodes
    vector<NodeItem> nodes;//tracks all nodes
    vector<NodeItem> inputs;//tracks input nodes
    int runTime;
	string runName;
    int transient;
    int window;
    int scalar;
    bool noisyInputs;
        
};
#endif
