/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <iostream>
#include <fstream>
#include <errno.h>
#include <iomanip>
#include <vector>
#include <string>
#include <map>
#include "../../lib/Matrix.h"
#include "Patterns.h"

using namespace std;

	/**
	 * Patterns constructor
	 */
Patterns::Patterns(void)
{
	;;;
}

	/**
	 * Patterns destructor
	 */
Patterns::~Patterns(void)
{
	;;;
}

	/**
	 * Patterns analysis to be run during a simulation
	 */
void Patterns::midPatterns(string & avgFileName_pt, string & labelsFileName, string & specAvgFile, string & in_patternNodesFile)
{
	ifstream inAvg, inSpecNodes, inNodeLabels;
	ofstream specAvg;
	inAvg.open(avgFileName_pt.c_str());
	Matrix m;
	m.readMat2(inAvg);
	inNodeLabels.open( labelsFileName.c_str() );
	if (!inNodeLabels) 
	{
		cerr << "Unable to open " << labelsFileName << " to read labels!" << endl;
		exit(1);
	}
	m.readLabels(inNodeLabels);
	specAvg.open(specAvgFile.c_str());
	if (specAvg.fail()) 
	{
		cerr << "Unable to open " << specAvg << " for file with selected columns" << endl;
		exit(1);
	}

	inSpecNodes.open(in_patternNodesFile.c_str());
	if (inSpecNodes.fail()) 
	{
		cerr << "Unable to open " << in_patternNodesFile << " for file with selected columns" << endl;
		exit(1);
	}
	Matrix specM;
	specM.readLabels(inSpecNodes);
	vector<string> labels;
	labels = specM.getLabels();
	for( unsigned int i = 0; i < labels.size(); ++i )
	{
		specM.addRow(m.getCol(labels[i]));
	}
	specM.transpose();
	specM.printTheMat(specAvg);
}

	/**
	 * Patterns analysis to be run after a simulation
	 */
void Patterns::postPatterns(string & inSpecsStr, string & inNodesStr, string & inInNodesStr, string & inAvgStr,  string & inInAvgStr, string & outputPath)
{
	const string DUMPPATH = " > /dev/null 2>&1";
	ifstream inNodes, inSpecs, inInNodes, inAvg, inInAvg;
	string inputPath = "";
	
	
	inNodesStr =  inputPath + inNodesStr; 
    inNodes.open(inNodesStr.c_str());
    if (inNodes.fail()) 
    {
	   cerr << "Unable to open " << inNodesStr << " file with nodes." << endl;
	   exit(1);
    }

    inSpecsStr =  inputPath + inSpecsStr; 
    inSpecs.open(inSpecsStr.c_str());
    if (inSpecs.fail()) 
    {
	    cerr << "Unable to open " << inSpecs << " file settings." << endl;
	    exit(1);
    }

    inInNodesStr =  inputPath + inInNodesStr; 
    inInNodes.open(inInNodesStr.c_str());
    if (inInNodes.fail()) 
    {
	    cerr << "Unable to open " << inInNodesStr << " file with input nodes." << endl;
	    exit(1);
    }

    inAvgStr =  inputPath + inAvgStr; 
    inAvg.open(inAvgStr.c_str());
    if (inAvg.fail()) 
    {
	    cerr << "Unable to open " << inAvgStr << " file with node averages." << endl;
	    exit(1);
    }

    inInAvgStr = inputPath + inInAvgStr;  
    inInAvg.open(inInAvgStr.c_str());
    if (inInAvg.fail()) 
    {
	    cerr << "Unable to open " << inInAvgStr << " file with input node averages." << endl;
	    exit(1);
    }

    outputPath = inputPath + outputPath; 

    Matrix nodeAvg;
    Matrix inputAvg;

    nodeAvg.readMat2(inAvg);
    nodeAvg.readLabels(inNodes);

    inputAvg.readMat2(inInAvg);
    inputAvg.readLabels(inInNodes);

    StringUtils su;
    string line = "";
    int cnt = 0;
    map<string, pair<int,int> > def;
    map<string, map<string, pair<int,int> > > nodes;

    while( !inSpecs.eof() )
    {
        getline(inSpecs, line );
        if( (line[0] != '*') && (line != "") )  //Make sure the line is not a comment
        {
            vector<string> parts;
            cnt = su.SplitString( line, ":", parts );
            if( (parts.size() != 2) && (parts.size() != 3) )
            {
                cerr << "Wrong format of settings!" << endl;
                cout << "Size: " << parts.size() << endl;
                exit(1);
            }
            else
            {
                //container for the default setting: descr,lower b,higher b 
                vector<string> ranges; //holds the string of ranges
                if( parts[0] == "default" ) //default settings
                {
                    for( unsigned int i = 1; i < parts.size(); ++i )
                    {
                        //read in the descriptor and the ranges
                        cnt = su.SplitString( parts[i], ";", ranges );
                        for( unsigned int j = 0; j < ranges.size(); ++j )
                        {
                            string descr = "";
                            string lowerStr = "";
                            string higherStr = "";
                            vector<string> tmp;
                            cnt = su.SplitString( ranges[j], " ", tmp );
                            lowerStr = tmp[0];
                            higherStr = tmp[1];
                            descr = tmp[2];
                            pair<int,int> range( atoi(lowerStr.c_str()), atoi(higherStr.c_str()));
                            def[descr] = range;
                            tmp.clear();
                        }
                        ranges.clear();
                    } 


                } //end if( parts[0] == "default" ) 
                else if( parts[0] == "node" ) //settings for specific nodes
                {
                    map<string, pair<int,int> > tmpMap;
                    string nodeName = parts[1];
                    for( unsigned int i = 2; i < parts.size(); ++i )
                    {
                        //read in the descriptor and the ranges
                        cnt = su.SplitString( parts[i], ";", ranges );
                        for( unsigned int j = 0; j < ranges.size(); ++j )
                        {
                            string descr = "";
                            string lowerStr = "";
                            string higherStr = "";
                            vector<string> tmp;
                            cnt = su.SplitString( ranges[j], " ", tmp );
                            lowerStr = tmp[0];
                            higherStr = tmp[1];
                            descr = tmp[2];
                            pair<int,int> range( atoi(lowerStr.c_str()), atoi(higherStr.c_str()));
                            tmpMap[descr] = range;
                            tmp.clear();
                        }
                        ranges.clear();
                    }
                    nodes[nodeName] = tmpMap;               
                }//end else if( parts[0] == "node" )
                else
                {
                    cerr << "Wrong format of settings!" << endl;
                    exit(1);
                }//end else
            }// end else
        }// end if
    } // end while loop   
    
    //get the nodes from  the matrix
    vector<string> nodeLabels;
    nodeLabels = nodeAvg.getLabels();
    Matrix cnvMat;
    //loop through the maps of nodes, and if a node not found, assing the default range
    for( unsigned int i = 0; i < nodeLabels.size(); ++i )
    {
        map<string, map<string, pair<int,int> > >::iterator iter;
        iter = nodes.find(nodeLabels[i]);
        if ( iter == nodes.end() ) //node not found
        {
            if( def.size() == 0 )
            {
                cerr << "Default setting required" << endl;
                exit(1);
            }
            nodes[nodeLabels[i]] = def;
            iter = nodes.find(nodeLabels[i]);
        }
        
        
        //convert the avg matrix into the range matrix
        vector<int> col;
        col = nodeAvg.getCol(i);
        vector<int> cnvCol;
        for( unsigned int j = 0; j < col.size(); ++j )
        {
            map<string, pair<int,int> >::iterator iter2;
            for(iter2 = iter->second.begin(); iter2 != iter->second.end(); iter2++) //search through ranges
            {
                string desc = iter2->first;
                int descInt = atoi(desc.c_str());
                int low = iter2->second.first;
                int high = iter2->second.second;
                //cout << "Node: " << nodeLabels[i] << " des: " << descInt << ": " << low << " - " << high << endl;
                if( (col[j] >= low) && (col[j] <= high) )
                {
                    cnvCol.push_back(descInt);
                    //cout << "Node: " << nodeLabels[i] << " -> " << descInt << endl;
                }
            }   
        }// end for loop
        cnvMat.addRow(cnvCol);
        cnvCol.clear();
    }// end for loop

    cnvMat.transpose(); 
    
    string ultPath = outputPath;// + "/patterns";
    string cmd = "mkdir " + ultPath + DUMPPATH;
    system(cmd.c_str());
    
    map< string, vector<int> > patterns;
    patterns = cnvMat.groupMat();
    cout << "Found: " << patterns.size() << " patterns" << endl;
    
    vector<string> inputLabels;
    inputLabels = inputAvg.getLabels();
    
    ofstream out2;
    string outFile2 = ultPath + "/all_vectors_stats.csv";
    out2.open( outFile2.c_str() );
    if (out2.fail()) 
    {
        cerr << "Unable to open " << outFile2 << " file for output." << endl;
        exit(1);
    }
    out2 << "Global Output" << "   " << "Count" << "   "; 
    for( unsigned int i = 0; i < inputLabels.size(); ++i )
        out2 << inputLabels[i] << "  ";

    for( unsigned int i = 0; i < nodeLabels.size(); ++i )
        out2 << nodeLabels[i] << "  ";
    out2 << endl;
    ofstream out3;
    string outFile3 = ultPath + "/all_vectors_std.csv";
    out3.open( outFile3.c_str() );
    if (out3.fail()) 
    {
        cerr << "Unable to open " << outFile3 << " file for output." << endl;
        exit(1);
    }
    out3 << "Global Output" << "   " << "Count" << "   "; 
    for( unsigned int i = 0; i < inputLabels.size(); ++i )
        out3 << inputLabels[i] << "  ";

    for( unsigned int i = 0; i < nodeLabels.size(); ++i )
        out3 << nodeLabels[i] << "  ";
    out3 << endl;
    
    Matrix patternMat;
    Matrix patternInMat;
    map<string, vector<int> >::iterator patternIter;
    
    for( patternIter = patterns.begin(); patternIter != patterns.end(); patternIter++ ) //loop through patterns
    {
        string pattern = patternIter->first;
        vector<int> matIndx = patternIter->second;
        for( unsigned int i = 0; i < matIndx.size(); ++i )
        {
            //merge the two vectors
            vector<int> tmp, nAvg;
            tmp = inputAvg.getRow(matIndx[i]);
	    patternInMat.addRow(tmp);
	    nAvg = nodeAvg.getRow(matIndx[i]);
	    for( unsigned int j = 0; j < nAvg.size(); ++j )
            {
                tmp.push_back(nAvg[j]);
            }
	    patternMat.addRow(tmp);
        }

        //create folder with the pattern name
        string patternPath = ultPath + "/" + pattern;
        string cmd = "mkdir " + patternPath + DUMPPATH;
        system(cmd.c_str());
        
        //print output
        ofstream out;
        string outFile = patternPath + "/vectors_graph.csv";
        out.open( outFile.c_str() );
        if (out.fail()) 
        {
            cerr << "Unable to open " << outFile << " file for output." << endl;
            exit(1);
        }
        patternMat.printTheMat(out);
        out.close(); 
        //print output for maple PCA
        ofstream PCAout;
        string PCAoutFile = patternPath + "/" + pattern +"_IN.mtb";
        PCAout.open( PCAoutFile.c_str() );
        if (PCAout.fail()) 
        {
            cerr << "Unable to open " << PCAoutFile << " file for PCA data output." << endl;
            exit(1);
        }
        patternInMat.printTheMat_MapleTransposedPCAFrmt(PCAout,pattern);
        PCAout.close();
	out2 << pattern << "  " << patternMat.size() << "  ";
        vector<int> avgVec;
        avgVec = patternMat.getAvgVec();
   
        for( unsigned int i = 0; i < avgVec.size(); ++i )
            out2 << avgVec[i] << "  ";
        out2 << endl;
        out3 << pattern << "  " << patternMat.size() << "  ";
        vector<double> stdVec;
        stdVec = patternMat.getStdVec();
   
        for( unsigned int i = 0; i < stdVec.size(); ++i )
            out3 << stdVec[i] << "  ";
        out3 << endl;

        patternMat.clear();
        patternInMat.clear();
  
    }
	
    out2.close();
}
