/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */ 

#include <iostream>
#include <assert.h>
#include <cstring>
#include "Node.h"
#include "NodeList.h"

using namespace std;

// Debug switch
bool verbose = false;
bool outputAll = false;
bool globalUp = false;
//bool rinputs = false;
bool calc = false;

Node::Node()
{
	/**
	 * Node constructor that initializes variables
	 */
    current = previous = false;
    timeT = 0;
}

void Node::setMut( string m )
{
	/**
	 * sets the mutation to string m
	 */
 mutation = m;
 if( m == "OFF" )
     current = previous = false;
 else if( m == "ON" )
     current = previous = true;
    

}

InputNode::InputNode()
    : Node()
{
	/**
	 * Input node constructor
	 */
    NC_amount=0;
    noisy = false;
}

void InputNode::amountInit( int win )
{
	/**
	 * sets the first n bool on ammount to false
	 */
    for( int i = 0; i < win; ++i )
        amount.push_back(false);
}


	/**
	 * Takes in val and sets noisy equal to it
	 * noisy is the boolean that determines if
	 * the Input node has noise interference
	 */
void InputNode::setNoise( bool val )
{
    noisy = val;
}

	/**
	 *	Sets this input node's level to level
	 *	for length equal to time
	 */
void InputNode::setLevels( int level, int time)
{
	int temp=start[0];
	start.clear();
	start.push_back(temp);
	dosage.clear();
	dosage.push_back(level);
	duration.clear();
	duration.push_back(time);

}

	/**
	 * Takes in val and sets manual equal to it
	 */
void InputNode::setManual( bool val )
{
    manual = val;
}

	/**
	 * Sets inputBitStr equal to s
	 */
void InputNode::setInputBitString( const string & s, NodeList & nl )
{
	cerr << "Size: " << s.size() << " RunTime: " << nl.getRunTime() << endl;
	if( int(s.size()) == int(nl.getRunTime()) )
	{
		inputBitStr = s;
	}
	else
	{
		cerr << "Specified input string needs to be the same size as the specified run time!" << endl;
		exit(1);
	}
}



	/**
	 * Sets previous to current and determines 
	 * the next value for current
	 *
	 * If dosages are used then it checks the start dosage
	 * and duration vectors to mantain the proper levels
	 */
bool InputNode::evaluate(NodeList &nl, int t)
{
	// Check to see if the node has already been evaluated
	if (timeT >= t) return previous;
	if (verbose) cerr << "> Evaluating Input Node " << endl;
	timeT = t;
	// Update previous value to the old current value
	previous = current;
	if( manual )
	{
		current = (inputBitStr[timeT]=='0')?false:true;
	}
	else
	{
		// Check to see if it is timeT to start the dosage

		int istart, idosage, iduration;
		if(!(start.size()==0))
		{
			istart = start[start.size()-1];
			idosage = dosage[dosage.size()-1];
			iduration = duration[duration.size()-1];
		}
		else
		{
			istart=0;
			idosage=0;
			iduration=0;
		}
		if(verbose) cout<<"Time is "<<t<<endl<<"start:"<<istart<<endl<<"dosage:"<<idosage
			<<endl<<"duration:"<<iduration<<endl<<endl;
		
		if ((timeT >= istart) && (timeT <= (istart + iduration))) 
		{
			// The code approximates the target dosage by going above and below
			// target percentage. This spreads the dosage out across the duration
			// timeT.

			if(verbose) cout << "start: " << istart << endl;
			if( noisy ==true) //Noisy inputs
			{
				if(verbose) cout << "Running with noisy inputs" << endl; 
				amountInit(nl.getWindow());
				if( timeT == istart )
				{      
					if( idosage == 100 )
					{
						current = true;
						amount[0] = current;
					}
					else
					{
						current = false;
						amount[0] = current;
					}
				}       
				else
				{
					int window = nl.getWindow();
					int tot = 0;
					int period; 
					period = ( (window > ( timeT - istart + 1 )) ? (timeT - istart +1) : window );
					for( int i = 0; i <= period; ++i )
					{    
						if( amount.size() > 0 )
							if( amount[i]==true )
								++tot;
					}
					int injection = (100 * tot) / period;
					int r,m,scalar;
					scalar = nl.getScalar();

					m = scalar*2 + 1;
					r = (rand() % m) - scalar;

					if (injection < (idosage +  r ) ) 
					{
						current = true;
					} 
					else 
					{
						current = false;
					}

					if( amount.size() > 0 )
					{
						//Insert the current value to the begining of the window

						for( int j = window; j > 0; --j )
  							amount[j] = amount[j-1];
						amount[0] = current;
					}
					else
						amount.push_back(current);


					if( timeT == (istart + iduration) )
					{
						start.pop_back();
						dosage.pop_back();
						duration.pop_back();
					}
				}
			}
			else //Non-noisy inputs
			{
				// The code approximates the target dosage by going above and below
				// target percentage. This spreads the dosage out across the duration
				// time.
				if(verbose) cout << "Running with Fixed inputs" << endl; 
				if( current && (timeT == 1) )
					NC_amount += 1;
				int injection = (100 * NC_amount) / (timeT - istart + 1);
				if (injection < idosage) {
					NC_amount += 1;
					current = injectionValue;
				} else {
					current = !injectionValue;
				}
				if( timeT == (istart + iduration) )
				{
					start.pop_back();
					dosage.pop_back();
					duration.pop_back();
					NC_amount = 0;
				}

			}

		} 
		else 
		{
			// Past the duration of the dosage
			current = !injectionValue;
		}
		// Change this to run from input values
	}
    if (verbose) cerr << "\tInput Value is " << previous << endl;
    return previous;
}


	/**
	 * Checks the consistency of starting times for input dosages. 
	 */
void InputNode::checkDosages()
{
    if( start.size() > 1 )
    {
        //cout << "Start:"
        if( (start[start.size()-1] > start[start.size()-2]) && 
                ( start[start.size()-1] > (start[start.size()-2]+duration[duration.size()-2]) ) )
        {}
        else
        {
            cerr << "Multiple start timeTs for input dosages have to be in increasing order, and the intervals ";
            cerr << "can not overlap" << endl;
            exit(99);
        }
    }
}

	/**
	 * Reverses the contents of the arrays start,
	 * duration, and dosage so that they go in the
	 * correct order and not in reverse
	 */
void InputNode::reverseParams( void )
{
    vector<int> tmp;
    tmp.clear();
    int size = start.size();
    tmp = start;
    start.clear();
    for( int i = 0;  i < size; ++i )
    {
        start.push_back(tmp[tmp.size()-(i+1)]);
    }
    tmp.clear();
    
        tmp = dosage;
        dosage.clear();
    for( int i = 0; i < size; ++i )
    {
        dosage.push_back(tmp[tmp.size()-(i+1)]);
    }
    tmp.clear();
        
    tmp = duration;
    duration.clear();
    for( int i = 0; i < size; ++i )
    {
        duration.push_back(tmp[tmp.size()-(i+1)]);
    }
    tmp.clear();
}


	/**
	 * Prints the Parameters
	 */
void InputNode::printParams()
{
    for( unsigned int i = 0; i < start.size(); ++i )
    {
       cout << "Start: " << i << ": "  << start[i] << endl;
    }
    for( unsigned int i = 0; i < dosage.size(); ++i )
    {
        cout << "Dosage: " << i << ": "  << dosage[i] << endl;
    }
    for( unsigned  int i = 0; i < duration.size(); ++i )
    {
        cout << "Duration: " << i << ": "  << duration[i] << endl;
    }
}


	/**
	 * OutputNode constructor
	 */
OutputNode::OutputNode()
    : Node()
{
    source = -1;
}


	/**
	 * Output Node Implementation of evaluate:
	 * The output node's value is set to the value
	 * of the node that it is dependant on
	 */
bool OutputNode::evaluate(NodeList &nl, int t)
{
    	// Don't evaluate nodes that have no input source
    	if (source == -1) return current;
    	// Don't evaluate nodes that have already been processed
    	if (timeT >= t) return current;
    	if (verbose) cerr << "> Evaluating Output Node " << endl;
    	timeT = t;
    	previous = current;
    	// The output node's value equals its input
	if (verbose) 
		cerr << "\tOutput Value depends on node " << source << " " << nl.getName(source) << endl;
	current = nl.getNode(source).evaluate(nl, t);
	if (verbose) 
		cerr << "\tValue of " << source << " " << nl.getName(source) << " is " << current << endl;
	if (verbose) 
		cerr << "\tOutput value is " << current << endl;
	return current;
}
	/**
	 * Boolean Node Constructor
	 */
BooleanNode::BooleanNode()
    : Node()
{
   target = true;
}

	/**
	 * Adds a new parent node to sources
	 */
BooleanNode &BooleanNode::addSource(int item)
{
   sources.push_back(item);
   return *this;
}

	/**
	 * adds d to the decision table
	 */
BooleanNode &BooleanNode::addDecision(int d)
{
   table.push_back(d);
   return *this;
}

	/**
	 * Sets the value equal to the appropriate
	 * value based on the sources.
	 *
	 * retreives the correct table in the form of an integer,
	 * builds the current state table in the form of an integer
	 * using bit shift operations and then compares the two 
	 * to determine if the current state of the inputs mandates
	 * that this node be true.
	 */
bool BooleanNode::evaluate(NodeList &nl, int t)
{
    int update = 0;
    bool result;
        // Don't evaluate a node that has already been evaluated
    if (timeT >= t) 
        return previous;

    if (verbose) cerr << "> Evaluating Boolean Node " << endl;

    timeT = t;
    if( mutation == "ON" )
        previous = true;
    else if( mutation == "OFF" )
        previous = false;
    else
    {
	    if (verbose)cerr<<"..."<<endl;
        previous = current;
        // Determine the value of each input
        for (int i = 0; i < sourceSize(); i++) {
            update <<= 1;
            if (verbose) cerr << "\tBoolean Value depends on node " << sources[i] << " "  << nl.getName(sources[i]) << endl;
            result = nl.getNode(sources[i]).evaluate(nl, t);
            if (verbose) cerr << "\tValue of " << sources[i] << " "  << nl.getName(sources[i]) << " is " << current << endl;
            if (result) update |= 1;
        }
        // Find current pattern of input in target table
        current = !target;
	int j = tableSize();
        for (int i = 0; i < j; i++) {
            if (update == table[i]) {
                current = target;
                break;
            }
        }
        if (verbose) cerr <<"\tBoolean Node is " << previous << endl;
    }
    return previous;
}

	/**
	 * Delay Nodes put off changes in input by a fixed amount
	 */
DelayNode::DelayNode(int d)
    : Node()
{
    source = -1;
    pipeline = NULL;
    setDelay(d);
}

	/**
	 * Sets the delay of the delaynode to d
	 */
DelayNode &DelayNode::setDelay(int d)
{
   if (d < 1) {
      cerr << "Attempt to establish a delay node with a value of " << d << endl;
      exit(2);
   }
   if (pipeline) delete [] pipeline;
   delay = d;
   pipeline = new int[d];
   assert(pipeline != NULL);
   // Prefill the pipeline with the previous value
   for (int i = 0; i < d; i++) {
      pipeline[i] = previous;
   }
   return *this;
}

	/**
	 * Destructor for delaynode
	 */
DelayNode::~DelayNode()
{
    if (pipeline) delete [] pipeline;
}


	/**
	 * Copy Constructor for delaynode
	 */
DelayNode::DelayNode(const DelayNode &r)
{
   if (this != &r) {
      if (pipeline) delete [] pipeline;
      pipeline = new int[r.delay];
      assert(pipeline != NULL);
      memcpy(pipeline, r.pipeline, r.delay * sizeof(int));
      delay = r.delay;
      source = r.source;
   }
}


	/**
	 * Assignment for delaynode
	 */
const DelayNode &DelayNode::operator =(const DelayNode &r)
{
   if (this != &r) {
      if (pipeline) delete [] pipeline;
      pipeline = new int[r.delay];
      assert(pipeline != NULL);
      memcpy(pipeline, r.pipeline, r.delay * sizeof(int));
      delay = r.delay;
      source = r.source;
   }
   return *this;
}

	/**
	 * Evaluates delay nodes.
	 */
bool DelayNode::evaluate(NodeList &nl, int t)
{
    // Don't evaluate a node that has no input source
    if (source == -1) return previous;
    // Don't evaluate a node that already has been evaluated
    if (timeT >= t) return previous;
    if (verbose) cerr << "> Evaluating Delay Node " << endl;
    int timeDiff = t-timeT;
    timeT = t;
    previous = current;
    // Find the input value and add it to the delay pipeline
    if (verbose) cerr << "\tDelay Value depends on node " << source << " "  << nl.getName(source) << endl;
    bool update = nl.getNode(source).evaluate(nl, t);
    if (verbose) cerr << "\tValue of " << source << " "  << nl.getName(source) << " is " << update << endl;
    for (int j = 0; j < timeDiff; j++) {
        current = pipeline[0];
        // Make room for new value
        for (int i = 0; i < delay-1; i++) {
            pipeline[i] = pipeline[i+1];
        }
        pipeline[delay-1] = update;
    }
    if (verbose) cerr << "\tDelay node value is " << previous << endl;
    return previous;
}


	/**
	 * Sustain Node receive a signal, and continue to produce an output for
	 * a period of timeT regardless of the input state.
	 */
SustainNode::SustainNode(int d)
{
    start = 0;
    duration = d;
    source = -1;
}

	/**
	 * Evaluates sustain node.
	 */
bool SustainNode::evaluate(NodeList &nl, int t)
{
    // Don't evaluate a node that has no input source
    if (source == -1) return previous;
    // Don't evaluate a node that already has been evaluated
    if (timeT >= t) return previous;
    if (verbose) 
		cerr << "> Evaluating Sustain Node " << endl;
    timeT = t;
    previous = current;
    // Are we waiting for the input to change?
    if (start == 0) {
        if (verbose) 
			cerr << "\tSustain Value depends on node " << source << " "  
				 << nl.getName(source) << endl;
        bool update = nl.getNode(source).evaluate(nl,t);
        if (verbose) 
			cerr << "\tValue of " << source << " "  << nl.getName(source) << " is " 
				 << update << endl;
        if (current != update) {
            start = timeT;
            current = update;
        }
    } else if (timeT >= (start + duration)) {
        // Time has expired, switch back to original value
        start = 0;
        current = !current;
    }
    if (verbose) cerr << "\tSustain node value is " << previous << endl;
    return previous;
}


