/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef NODE_H
#define NODE_H
// Node
#include <vector>
#include <string>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

using namespace std;

extern bool verbose;
extern bool outputAll;
extern bool globalUp;
extern bool rinputs;
extern bool calc;

void ipause();

class NodeList;//

	/**
	 * @section DESCRIPTION
	 * The Node is an abstraction of the different
	 * types of other nodes
	 */
class Node {
public:
    Node();
	///virtual destructor for node
    virtual ~Node(){};
	///sets current and previous to v and returns *this
    Node &setValue(bool v){current = v; previous = v; return *this;};
	///returns previous
    bool getValue(){return previous;};
    void setMut ( string m );
    virtual bool evaluate(NodeList &nl, int t)=0;
	virtual string getType(void)=0;
protected:
    string mutation;
    bool current;   // Current value at this time
    bool previous;  // Prior value value before this time
    int timeT;       // Last time this node was updated
};

/**
 * @section DESCRIPTION
 * Input nodes are the receptors for detecting the presences 
 * of chemicals These chemicals may not be solidly available 
 * during a period of timeT, but come as a percentage during 
 * a period of timeT.
 */
class InputNode : public Node {
public:
    InputNode();
    virtual bool evaluate(NodeList &nl, int t);
	virtual string getType(void){return "Input";};
    //bool evaluate_nonnoisy(NodeList &nl, int t);
	///pushes s onto start and returns *this
    InputNode &setStart(int s) {start.push_back(s); return *this;};
	///pushes d onto doage and returns *this
    InputNode &setDosage(int d) {dosage.push_back(d); return *this;};
	///pushes d onto duration and returns *this
    InputNode &setDuration(int d) {duration.push_back(d); return *this;};
	///sets the injectionValue to the opposite of previous and returns *this
    InputNode &setInjectionValue() {injectionValue = !previous; return *this;};
	void setLevels(int,int);
    void checkDosages(void);
    void reverseParams(void);
    void printParams(void);
    void setNoise( bool );
    void setManual( bool );
    void setInputBitString( const string &, NodeList & );
	//vector<> getInputBits( void );
private:
    vector<int> start;    // Index to times to start different dosages
    vector<int> dosage;   // Percent chemical is present during time duration
    vector<int> duration; // Length of time dosage is available
    vector<bool> amount;   // Amount injected so far - vector of values of the window size
    string inputBitStr; //manually set input feed
    int NC_amount;      //Amount to keep track of for non-noisy inputs
    void amountInit( int );
    bool injectionValue;
    bool noisy;
    bool manual; //input bit string has been provided
};

/**
 * @section DESCRIPTION
 * An output node is set to a node and tracks that node for 
 * the purposes of analysis
 */
class OutputNode : public Node {
public:
    OutputNode();
	///sets source to s and returns *this
    OutputNode &setSource(int s) {source = s; return *this;};
    virtual bool evaluate(NodeList &nl, int t);
	virtual string getType(void){return "Output";};
private:
    int source;
};

class Mutant {
public:
	///constructor for Mutant
    Mutant( const string & n, const string &m)
    {   
        name = n;
        mutation = m;
    };
	///returns mutation
    string getType (void) { return mutation; };
	///returns name
    string getName (void) { return name; };    
private:
    string name;
    string mutation;
};


/**
 * @section DESCRIPTION
 * A boolean node is a representation for the activity
 * level of protiens within the simulation, they store 
 * their current state, their previous state and the 
 * other nodes wich it is dependant on.
 */
class BooleanNode : public Node {
public:
    BooleanNode();
    virtual bool evaluate(NodeList &nl, int t);
	virtual string getType(void){return "Bool";};
    BooleanNode &addSource(int item);
    BooleanNode &addDecision(int d);
	///sets target to b and returns *this
    BooleanNode &setTarget(bool b){target=b; return *this;};
private:
    vector<int> sources;    // index into list of nodes
	///returns the number of sources
    int sourceSize() {return static_cast<int>(sources.size());};
    vector<int> table;
	///returns the size of the table
    int tableSize() {return static_cast<int>(table.size());};
    bool target;
};

/**
 * @section DESCRIPTION
 * A delay node is a node to reflect an input after
 * a set delay period.
 */
class DelayNode : public Node {
public:
    DelayNode(int d=1);
    ~DelayNode();
    DelayNode(const DelayNode &r);
    const DelayNode &operator =(const DelayNode &);
    virtual bool evaluate(NodeList &nl, int t);
	virtual string getType(void){return "Delay";};
    DelayNode &setDelay(int d);
	///sets source to s and returns *this
    DelayNode &setSource(int s){source=s; return *this;};
private:
    int source;             // index into list of nodes
    int delay;
    int *pipeline;
};

/**
 * @section DESCRIPTION
 * A sustain node is a node that reflects its input 
 * for a prescribed amount of time.
 */
class SustainNode : public Node {
public:
    SustainNode(int d=1);
    virtual bool evaluate(NodeList &nl, int t);
	virtual string getType(void){return "Sustain";};
	///sets duration to d and returns *this
    SustainNode &setDuration(int d) {duration = d; return *this;};
	///sets source to s and returns this
    SustainNode &setSource(int s) {source=s; return *this;};
private:
    int source;
    int duration;
    int start;
};
#endif
