/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
// Node Lists
#include <assert.h>
#include "NodeList.h"
#include <algorithm>
#include <cstring>
#include <typeinfo>

using namespace std;

enum nodeType {UNDEF, IN, OUT, BOOLEAN, MUTANT, ANALPTS, DELAY, SUSTAIN, RUNTIME, NOISE, COMMENT};//rework input parameters
char line[1000000000];
int offset = 100000000;

/**
 * Default Constructor
 */
NodeItem::NodeItem() : node(NULL)
{
}

/**
 * copy constructor
 */
NodeItem::NodeItem(const NodeItem & other) : node(NULL)
{
	//use operator = to assign the data in "other" to this NodeItem
	*this = other;
}

/**
 * Destructor
 */
NodeItem::~NodeItem()
{
	//deallocate memory used to store a node
	removeNode();
}

/**
 * assign one NodeItem to another
 */
NodeItem & NodeItem::operator = (const NodeItem & other)
{
	//if they are not the same object
	if (this != &other)
	{
		//copy name
		name = other.name;
		//copy node
		if (other.node != NULL)
		{
			setNode(*other.node);
		}
		else
		{
			node = NULL;
		}
	}
	//return a NodeItem &
	return *this;
}

/**
 * set the node on this item to a valid InputNode, OutputNode, BooleanNode, DelayNode, or SustainNode
 */
void NodeItem::setNode(const Node & other)
{
	//use RTTI to call the proper function
	if (typeid(other) == typeid(InputNode))
	{
		setNode(static_cast<const InputNode &>(other));
	}
	else if (typeid(other) == typeid(OutputNode))
	{
		setNode(static_cast<const OutputNode &>(other));
	}
	else if (typeid(other) == typeid(BooleanNode))
	{
		setNode(static_cast<const BooleanNode &>(other));
	}
	else if (typeid(other) == typeid(DelayNode))
	{
		setNode(static_cast<const DelayNode &>(other));
	}
	else if (typeid(other) == typeid(SustainNode))
	{
		setNode(static_cast<const SustainNode &>(other));
	}
}

/**
 * set the node to a valid InputNode
 */
void NodeItem::setNode(const InputNode & other)
{
	//deallocate old memory
	removeNode();
	//allocate new memory
	try
	{
		node = new InputNode();
	}
	catch(bad_alloc)
	{
		node = NULL;
	}
	//assign the data if memory allocation was successful
	if (node)
	{
		InputNode * data = static_cast<InputNode *>(node);
		*data = other;
	}
}

/**
 * set the node to a valid OutputNode
 */
void NodeItem::setNode(const OutputNode & other)
{
	//deallocate old memory
	removeNode();
	//allocate new memory
	try
	{
		node = new OutputNode();
	}
	catch(bad_alloc)
	{
		node = NULL;
	}
	//assign the data if memory allocation was successful
	if (node)
	{
		OutputNode * data = static_cast<OutputNode *>(node);
		*data = other;
	}
}

/**
 * set the node to a valid BooleanNode
 */
void NodeItem::setNode(const BooleanNode & other)
{
	//deallocate old memory
	removeNode();
	//allocate new memory
	try
	{
		node = new BooleanNode();
	}
	catch(bad_alloc)
	{
		node = NULL;
	}
	//assign the data if memory allocation was successful
	if (node)
	{
		BooleanNode * data = static_cast<BooleanNode *>(node);
		*data = other;
	}
}

/**
 * set the node to a valid DelayNode
 */
void NodeItem::setNode(const DelayNode & other)
{
	//deallocate old memory
	removeNode();
	//allocate new memory
	try
	{
		node = new DelayNode();
	}
	catch(bad_alloc)
	{
		node = NULL;
	}
	//assign the data if memory allocation was successful
	if (node)
	{
		DelayNode * data = static_cast<DelayNode *>(node);
		*data = other;
	}
}

/**
 * set the node to a valid SustainNode
 */
void NodeItem::setNode(const SustainNode & other)
{
	//deallocate old memory
	removeNode();
	//allocate new memory
	try
	{
		node = new SustainNode();
	}
	catch(bad_alloc)
	{
		node = NULL;
	}
	//assign the data if memory allocation was successful
	if (node)
	{
		SustainNode * data = static_cast<SustainNode *>(node);
		*data = other;
	}
}

/**
 * check if a node has been allocated to this NodeItem
 */
bool NodeItem::hasNode() const
{
	//check for NULL
	if (node == NULL)
	{
		return false;
	}
	else
	{
		return true;
	}
}

/**
 * get the non-NULL node from the NodeItem
 */
Node &NodeItem::getNode()
{
	//throw an exception if the user is trying to derefrence NULL
	if (node == NULL)
	{
		throw runtime_error("No node allocated at this time.");
	}
	//return the node
	return *node;
}

/**
 * remove a node by reclaiming its memory
 */
bool NodeItem::removeNode()
{
	//only call delete if pointer is non-NULL
	if (node != NULL)
	{
		delete node;
		node = NULL;
		return true;
	}
	return false;
}

/**
 * change the name used to refer to this NodeItem
 */
void NodeItem::setName(string s)
{
	name = s;
}

/**
 * Constructor
 */
NodeList::NodeList()
{
	runTime = 100;
	srand(time(NULL));
}

/**
 * Adds item to outputs
 */
void NodeList::addOutput(NodeItem &item)
{
	outputs.push_back(item);
}

/**
 * Adds item to inputs
 */
void NodeList::addInput(NodeItem &item)
{
	inputs.push_back( item );
}

/**
 * Adds item to outCalc
 */
void NodeList::addOutCalc(NodeItem &item)
{
	outCalc.push_back(item);
}

/**
 * adds item to nodes
 */
void NodeList::addNode(NodeItem &item)
{
	nodes.push_back(item);
}

/**
 * Adds item to mutants
 */
void NodeList::addMutant(Mutant &item)
{
	mutants.push_back(item);
}

/**
 * Adds item to analPts
 */
void NodeList::addAnalPt( const int &item )
{
	analPts.push_back(item);
}

/**
 * For every Mutant, find the matching node and set its Mut
 * to the matching Mutant's type
 */
void NodeList::setMutants( void )
{
	if( mutants.size() > 0 )  //If any mutants, then update the nodes
	{
		for( int i = 0; i < nodeSize(); ++i )
		{
			for( unsigned int j = 0; j < mutants.size(); ++j )
			{
				if( getName(i) == mutants[j].getName() )
				{        
					string type = mutants[j].getType();
					nodes[i].getNode().setMut(type);
				}
			}
		}
	} 
}

/**
 * Read a line from the working buffer
 */
void readLine(istream &in)
{
	in.getline(line, sizeof(line));
	offset = 0;

	// cerr << "!!!test: " << line << endl;
}


/**
 * Test if line is exhausted
 */
bool isEmpty()
{
	if (int(offset) < int(sizeof(line)) && line[offset] != '\0') return false;
	return true;
}

/**
 * Test if input is exhausted
 */
bool isEOF(istream &in)
{
	if (isEmpty() && in.eof()) return true;
	return false;
}

/**
 * Read a character from the working buffer
 */
char readChar(istream &in)
{
	if (isEmpty()) {
		readLine(in);
		return '\n';
	}
	return line[offset++];
}

/**
 * Read in a single word, deliminated by either ':', ",", or "\n"
 */
string readWord(istream &in, char &terminator)
{
	string s;
	while (!isEOF(in)) {
		char c = readChar(in);
		switch (c) {
			case ':':
			case ' ':
			case ',':
				//cerr << "test3: " << c << endl;
			case '\n':
				terminator = c;
				return s;
			default:
				s += c;
				//cerr << "test3: " << s << endl;
		}
	}
	terminator = '\n';
	if (s.empty()) 
		s += '*';

	return s;
}

/**
 *  The remainder of the input line is read and discarded.
 */
void skipLine(istream &in)
{
	offset = sizeof(line);
}

/**
 * Read in the line type
 */
nodeType readType(istream &in)
{
	char t;
	string s = readWord(in, t);
	if (s.empty()) return UNDEF;
	s[0] = toupper(s[0]);
	// Input Node
	if (s == "Input") {
		return IN;
		// Output Node
	} else if ( s == "Output") {
		return OUT;
		// Boolean Node
	} else if ( s == "Bool") {
		return BOOLEAN;
		// Delay Node
	} else if( s == "Mutation" ){
		return MUTANT;
	}else if ( s == "Delay") {
		return DELAY;
		// Sustain Node
	} else if ( s == "Sustain") {
		return SUSTAIN;
		// Run time for experiment
	} else if ( s == "RunTime") {
		return RUNTIME;
		// Run time for experiment	
	} else if ( s == "SnapshotAnalysis") {
		return ANALPTS;
	} else if ( s == "Noise") {
		return NOISE;
		// Comment line
	} else if (s[0] == '*') {
		if (t != '\n') skipLine(in);
		return COMMENT;
	} else return UNDEF;
}

/**
 * Read in a chemical node name and return the name
 */
string readName(istream &in)
{
	char t;
	string s = readWord(in, t);
	return s;
}

/**
 * Read in an integer value
 */
int readValue(istream &in, char &t)
{
	char *tptr;
	long i;
	string s = readWord(in, t);
	i = strtol(s.c_str(), &tptr, 10);
	return static_cast<int>(i);
}

/**
 * Read in a float value
 */
float readFloat( istream &in )
{
	char t;//, *tptr;
	float i;
	string s = readWord(in, t);
	i = atof(s.c_str());
	return static_cast<float>(i);
}

/**
 * Read the intial value for a chemical node: either True or False
 */
bool readInitial(string & src, string & inputStr )
{
	ifstream in;
	in.open(src.c_str());
	if (in.fail()) 
	{
		cerr << "Unable to open " << src << " for input string file" << endl;
		exit(1);
	}
	else
	{
		string tmp;
		getline(in ,tmp);
		inputStr = tmp;
		if (toupper(tmp[0]) == '0') return false;
		else if (toupper(tmp[0]) == '1') return true;
	}
	return true;
}

/** 
 * Read the intial value for a chemical node: either True or False
 */
bool readInitial(string & init )
{
	if (toupper(init[0]) == 'F'){
		//cerr << "Init: False" << endl;
		return false;
	}
	return true;
}

/** 
 * Read the intial value for a chemical node: either True or False
 */
string readInitialSrc(istream & in )
{
	char t;
	string s = readWord(in, t);
	return s;
}

/** 
 * Read in a chemical node name, locate and return its index
 */
int readSource(istream &in, NodeList &nl, int &source)
{
	char t;
	string s = readWord(in, t);
	source = nl.find(s);
	if (t == ':') return true;
	return false;
}

/**
 * read a word into string & mutant and return true if
 * istream is empty
 */
bool readMutant(istream &in, NodeList &nl, string & mutant)
{
	char t;
	string s = readWord(in, t);
	mutant = s;
	if (t == '\n') return true;
	return false;
}

/** 
 * Read a decision table: a string of T and F characters
 */
int readDecision(istream &in, int &decision)
{
	char t;
	string s = readWord(in, t);
	//cerr << "des: " << s << endl;
	decision = 0;
	for (unsigned int i=0; i < s.size(); i++) {
		decision <<= 1;
		if (toupper(s[i]) == 'T') decision |= 1;
	}
	if (t == '\n') return true;
	return false;
}

/**
 * Comment to be written, function not in use
 */
void setRandInputs( InputNode & n, NodeList &nl )  //This function is not being used
{
	int dosage = rand()%101;
	if( verbose )
		cout << "Rand dosage: " << dosage << endl;
	n.setStart(1);     //For now, the random percentage is applied to the entire runtime
	n.setDosage(dosage);
	n.setDuration(nl.getRunTime()-1);
}

/**
 * Read points of output analysis
 */
void readAnalPts( istream &in, NodeList & nl )
{
	char t;
	int val = 0;
	int period = 0;
	period = readValue( in, t );
	if( t == ':' )
	{
		if( period > 0 )
		{
			nl.avgPeriod = period;
			//		cerr << "AVGPer: " << nl.avgPeriod << endl;
			while( t != '\n' )
			{
				val = readValue( in, t );
				if( (val > 0) && (val < nl.getRunTime()) )
				{
					if( verbose )
						cerr << "\n Iteration:	" << val;		
					nl.addAnalPt( val );
				}
				else if( val < 0 ) 
				{
					cerr << "Please enter a valid positive integer for the point of output "
						<<	"analysis!" << endl;
					exit(1);
				}
				else if( val > nl.getRunTime() ) 
				{
					cerr << "The point of output analysis must be smaller than RunTime!" 
						<< endl;
					exit(1);
				}
			}
		}
		else
		{
			cerr << "Average period has to be > 0 for Output analysis! Terminating ..." 
				<< endl;
			exit(1);
		}
	}
	else
	{
		cerr << "Wrong seperator for average period for Output Analysis" << endl;
		exit(1);
	}

	if( verbose )
		cerr << "\n Added " << nl.ptsSize() << " output analysis points." << endl;
}

/**
 * Reads the start, dosage, and duration from
 * the instream for an InputNode (Tomas Helikar 9/26/05)
 */
bool readParams(istream &in, InputNode & n, bool random, NodeList & nl )
{
	int start, dos, dur;
	char t;
	string s = "";
	bool rval = false;

	while( t != '\n' )
	{

		string st = readWord( in, t );
		start = atoi(st.c_str());
		n.setStart(start);

		string sd = readWord( in, t );
		StringUtils su;
		vector<string> rangeStr;
		int cnt;
		cnt = su.SplitString(sd,"-",rangeStr);
		if( random )
		{
			if( rangeStr.size() == 2 )
			{
				int low = atoi(rangeStr[0].c_str());
				int high = atoi(rangeStr[1].c_str());
				dos = rand()%(high-low+1)+low;
				if( verbose )
					cout << "Percent rand: " << dos << endl;
			}
			else
			{
				cerr << "Wrong format random number range in your specs file!" 
					<< endl;
				exit(1);
			}
		}
		else
		{
			if( rangeStr.size() == 0 )
			{
				dos = atoi(sd.c_str());
			}
			else
			{
				cerr << "Wrong format! Fixed  number in your specs file can not be a range!" << endl;
				exit(1);
			}
		}
		//cerr << "Dos: " << dos << endl; 
		n.setDosage(dos);

		//ofstream stats;
		//stats.open("stats.out", ios::app);
		//stats << " Percentage: " << dos << endl;
		nl.dosageRec.push_back(dos);

		string sdu = readWord( in, t );
		dur = atoi(sdu.c_str());
		n.setDuration(dur);

		if( (t != ',') && (t != '\n') )
		{
			cerr << "Wrong interval seperator!:" << t << endl;
			exit(99);
		} 
	}
	if( t == '\n')
		rval = true;
	return rval;
}

/**
 * Reads in an Inputfile and sets up
 * the network based on that file
 */
istream& operator >>(istream &in, NodeList &nl)
{
	while (!isEOF(in)) {
		NodeItem item;
		switch (readType(in)) {
			case ANALPTS:
				{
					if (verbose) cerr << "Output Analysis: ";

					readAnalPts( in, nl );

					//sort analysis oints
					nl.sortAnalPts( );

					if( verbose )
						if( nl.ptsSize() == 0 )
							cerr << "No points of output analysis were entered "
								<< "-> output analysis will take place after the simulation "
								<< "finishes." << endl;
					//nl.printAnalPts();
					//		exit(1);

				} break;
			case IN: {

						 string name;
						 name = readName(in);
						 if (verbose) 
							 cerr << "Input Node: " << name << endl;
						 item.setName(name);
						 bool random = false;

						 InputNode n;
						 string src = readInitialSrc( in );
						 if( src == "File" )//input handling should be done elsewhere
						 {
							 int dosage = 0;
							 n.setNoise(false);
							 n.setManual(true);
							 string inputStr = "";
							 char t;
							 string inSrc = readWord(in,t);
							 if( readWord( in,t ) == "R") 
							 {
								 //if( rinputs )
								 random = true;
								 string sd = readWord( in, t );
								 StringUtils su;
								 vector<string> rangeStr;
								 int cnt;
								 cnt = su.SplitString(sd,"-",rangeStr);
								 if( rangeStr.size() == 2 )
								 {
									 int dos;
									 int low = atoi(rangeStr[0].c_str());
									 int high = atoi(rangeStr[1].c_str());
									 dos = rand()%(high-low+1)+low;
									 if( verbose )
										 cout << "Percent rand: " << dos << endl;
									 char dosStr [10];
									 sprintf( dosStr,"%d", dos );
									 string tmpStr = "stats/" + nl.getRunName() + "/";
									 inSrc = tmpStr + inSrc;
									 inSrc += dosStr;
									 inSrc += ".txt";
									 dosage = dos;
									 cerr << "Src: " << inSrc << endl;
								 }
								 else
								 {
									 cerr << "Wrong format random number range in your specs file!" 
										 << endl;
									 exit(1);
								 }

							 }
							 else
							 {
								 dosage = -1;
								 //n.setDuration(nl.getRunTime()-1);
							 }
							 n.setValue(readInitial( inSrc, inputStr ));
							 n.setInputBitString(inputStr, nl);
							 n.setDosage(dosage);
							 nl.dosageRec.push_back(dosage);
						 }
						 else
						 {
							 n.setManual(false);
							 //string src = readInitialSrc( in );
							 //		 string s= "False";
							 //		 n.setValue(readInitial(s));

							 char t;
							 //Read in if the dosage is random or fixed
							 string tmp = readWord( in, t );

							 // cout << "Read: " << tmp << endl;
							 if( strcmp(tmp.c_str(), "R" ) == 0 )
							 {
								 //if( rinputs )
								 random = true;
							 }
							 //Check for fixed or random dosages
							 else if( strcmp(tmp.c_str(), "F" ) == 0 )
								 random = false;
							 else
							 {
								 cerr << "Have to specify if the input is fixed or random" 
									 << endl;
								 exit(1);
							 }

							 tmp = readWord( in, t );

							 //Check for noise 
							 if( strcmp(tmp.c_str(), "N" ) == 0 )
							 {
								 if( nl.isNoisyInputs() )
								 {
									 n.setNoise(true);
								 }
							 }
							 else if( strcmp(tmp.c_str(), "F" ) == 0 )
							 {
								 n.setNoise(false);
							 }
							 else
							 {
								 cerr << "Have to specify if the input is fixed or noisy" << endl;
								 exit(1);
							 }

							 bool done;                       

							 //Ability to have different dosages throughout the simulation 
							 //This section modified by Tomas Helikar 9/26/05
							 do 
							 {                               
								 done = readParams(in, n, random, nl);
								 n.checkDosages();
							 } while(!done);
							 n.reverseParams();
							 //}
							 n.setInjectionValue();

							 n.setValue(readInitial(src));
					 }
					 item.setNode(n);
					 nl.add(item);    // Add the node to the end of the list
					 nl.addInput(item);
					 if(calc)
					 {
						 nl.initInputBits();
					 }
		} break;
		case OUT:{
					if(!outputAll)
					{
						 int s;
						 if (verbose) cerr << "Output Node: ";
						 item.setName(readName(in));
						 OutputNode n;
						 string src = readInitialSrc( in );
						 n.setValue(readInitial(src));
						 //n.setValue(readInitial(in));
						 readSource(in, nl, s);
						 n.setSource(s);
						 item.setNode(n);
						 nl.addOutput(item);
						 nl.add(item);    // Add the node to the end of the lisit
					 }
				 } 
				 break;
		case BOOLEAN: {
						  if (verbose) cerr << "Boolean Node: ";
						  string name;
						  name = readName(in);
						  item.setName(name);
						  BooleanNode n;
						  string src = readInitialSrc( in );
						  //n.setValue(readInitial(src));
						  bool init = readInitial(src);
						  n.setValue(init);
						  bool done;
						  do {
							  int s;
							  done = readSource(in, nl, s);
							  n.addSource(s);
						  } while(!done);
						  do {
							  int d;
							  done = readDecision(in, d);
							  n.addDecision(d);
						  } while(!done);
						  n.setTarget(!n.getValue());
						  item.setNode(n);
						  nl.add(item);          // Add the node to the end of the list
						  if(calc) 
						  {
							  nl.addOutCalc(item);   
							  // Gong added in 11/11/05 to add node into the net
							  nl.avg.push_back(0);   
							  // Gong added in 11/11/05 to initialize the Average vector
							  nl.initBits(init);
						  }

						  //if outputall, create an output for every boolean.
						  if(outputAll)
						  {
							  OutputNode o;
							  NodeItem i;
							  string oname="Out_"+name;
							  i.setName(oname);
							  o.setSource(nl.find(name));			  
							  o.setValue(init);
							  i.setNode(o);
							  nl.addOutput(i);
							  nl.add(i);
						  }
					  } break;
		case MUTANT:{
						string mutation;
						char t;
						mutation = readWord(in, t);

						bool done = false;
						do {
							string name;
							done = readMutant(in, nl, name);
							Mutant mut(name, mutation);
							nl.addMutant(mut);
						} while(!done);
					} break;
		case DELAY: {
						int s;
						char t;
						if (verbose) cerr << "Delay Node: ";
						item.setName(readName(in));
						DelayNode n;
						string src = readInitialSrc( in );
						n.setValue(readInitial(src));
						//n.setValue(readInitial(in));
						readSource(in, nl, s);
						n.setSource(s);
						n.setDelay(readValue(in, t));
						item.setNode(n);
						nl.add(item);    // Add the node to the end of the list
					} break;
		case SUSTAIN: {/*
						  int s;
						  char t;
						  if (verbose) cerr << "Sustained Node: ";
						  item.setName(readName(in));
						  cout<<"Adding Sustain node:";
						  SustainNode n;
						  string src = readInitialSrc( in );
						  cout<<"Source:"<<src<<" ";
						  n.setValue(readInitial(src));
						  //n.setValue(readInitial(in));
						  readSource(in, nl, s);
						  cout<<"Source:"<<s<<" ";
						  n.setSource(s);
						  int temp = readValue(in, t);
						  //n.setDuration(readValue(in, t));
						  cout<<"Duration:"<<temp<<" ";
						  n.setDuration(temp);
						  item.setNode(n);
						  cout<<"Name:"<<item.getName()<<endl;
						  nl.add(item);    // Add the node to the end of the list
*/
						  char t=':';
						  item.setName(readWord(in,t));
						  SustainNode n;
						  string temp = readWord(in,t);
						  if(temp=="False")
							  n.setValue(false);
						  else if(temp=="True")
							  n.setValue(true);
						  temp = readWord(in,t);
						  n.setSource(nl.find(temp));
						  n.setDuration(readValue(in,t));
						  
					  } break;
		case RUNTIME: 
					  {
						  char t;
						  nl.setTransient(readValue(in, t));
						  nl.setRunTime(readValue(in, t));
						  break;
					  }
		case NOISE:
					  {
						  nl.setNoisyInputs();
						  char t;
						  nl.setWindow (readValue(in, t));
						  nl.setScalar (readValue(in, t));
						  break;
					  }
		case COMMENT:
					  // Comments are ignored
					  break;
		case UNDEF:

					  //use function to exit and output that input file is incorrect
					  break;
	}
}
return in;
}

/**
 * Prints Analysis Points for output
 */
void NodeList::printAnalPts( void )
{
	for( int i = 0; i < int(analPts.size()); i++ )
		cerr << "Point: " << analPts[i] << endl;
}

/**
 *  Locate a given node by name. If it does not exist, create a NodeItem
 *  to be filled in later.
 */
int NodeList::find(string n)
{
	for (int i = 0; i < nodeSize(); i++) {
		if (nodes[i].getName() == n) {
			if (verbose) cerr << " Reference to node " << n << " found at offset " << i << endl;
			return i;
		}
	}
	if (verbose) cerr << "Adding forward reference to node " << n << " at offset " << nodeSize() << endl;
	NodeItem item;
	item.setName(n);
	addNode(item);
	return nodeSize()-1;
}

/**
 * Returns the index of the node with name n on the Outcalc array
 * if that node is on the outcalc array, otherwise it returns -1
 */
int NodeList::inOutCalc(string n)
{
	int pos = -1;
	bool rval = false;
	int j = outCalcSize();
	for (int i = 0; (i < j)&&(!rval); i++) {
		if (outCalc[i].getName() == n) 
		{
			if (verbose) cerr<<" Reference to node "<<n<<" found for average calculation "<<i<<endl;
			rval = true;
			pos = i;
		}
	}
	return pos;
}

/**
 * Returns the index number of node n
 */
int NodeList::getNodeIndex(string n)
{
	int pos = -1;
	bool rval = false;
	for (int i = 0; i < int(nodes.size()); i++) {
		if (nodes[i].getName() == n) 
		{
			if (verbose) 
				cerr << " Reference to node " << n << endl;
			rval = true;
			pos = i;
		}
		if( rval )
			break;
	}
	return pos;
}

/** 
 * Locate a given node by name. If it exists, replace the node reference.
 * Otherwise, add the item to the list.
 */
int NodeList::add(NodeItem &item)
{
	for (int i = 0; i < nodeSize(); i++) {
		if (nodes[i].getName() == item.getName()) {
			if (verbose) 
				cerr << "Replacing node " << item.getName() 
					<< " with definition at " << i << endl;
			nodes[i].setNode(item.getNode());
			return i;
		}
	}
	if (verbose) 
		cerr << "Adding node " << item.getName() << " at offset " 
			<< nodeSize() << endl;
	addNode(item);
	return nodeSize()-1;
}

/**
 * Prints required nodes to visout.
 */
void NodeList::visPrint(ostream &out, NodeList &nl)
{
	if (outputAll) 
	{
		for (int i = 0; i < nl.nodeSize(); i++) 
		{
			if(nl.getNode(i).getType()=="Bool")
			{
				if (nl.getNode(i).getValue()) 
				{
					out << "* ";
				} 
				else 
				{
					out << ". ";
				}
			}
		}
	} 
	else 
	{
		for (int i = 0; i < nl.outputSize(); i++) 
		{
			if (nl.outputs[i].getNode().getValue()) 
			{
				out << "* ";
			} 
			else {
				out << ". ";
			}
		}
	}
	out<<endl;


}

/**
 * Properly records bits for later use
 */

void NodeList::bitsPrint(NodeList &nl)
{
	//int j = nl.nodeSize();
	for (int i = 0; i < nl.nodeSize(); i++) 
	{
		int pos = nl.inOutCalc( nl.getName(i));
		if ( pos > -1 )   //If > -1 then Node found for avg calculations    
		{ 
			if (nl.getNode(i).getValue()) 
			{
				nl.bits[pos].push_back(1);
				if(verbose)
				{
					cerr<<"Printing 1 for Node "<<nl.getName(i)<<" to bit files"<<endl;
					cerr<<"Value is "<<nl.getNode(i).getValue()<<endl;
				}	
			} 
			else 
			{
				nl.bits[pos].push_back(0); 
				if(verbose)
				{
					cerr<<"Printing 0 for Node "<<nl.getName(i)<<" to bit files"<<endl;
					cerr<<"Value is "<<nl.getNode(i).getValue()<<endl;
				}	
			}

		}

	}
	//j = int(nl.inputs.size());
	for( int i = 0; i < int(nl.inputs.size()); ++i )
	{
		int pos = nl.getNodeIndex( nl.inputs[i].getName());
		//cerr << "Pos: " << pos << endl;
		if ( pos > -1 )   //If > -1 then Node found for avg calculations    
		{
			if (nl.getNode(pos).getValue()) 
			{
				int val = 1;
				nl.inputBits[i].push_back(val);
				//cerr << "Pushing: one" << endl;
			} 
			else 
			{
				int val = 0;
				nl.inputBits[i].push_back(val); 
				//cerr << "Pushing: zero" << endl;
			}
		}
	}
}


/**
 * Sorts Analysis points
 */
void NodeList::sortAnalPts()
{
	sort( analPts.begin(), analPts.end() );
	reverse( analPts.begin(), analPts.end() );

}

/**
 * gets the next analysis point
 */
int NodeList::getNextAnalPt()
{
	int val;
	if( !analPts.empty() )
	{
		val = analPts.back();
		analPts.pop_back();
	}
	else
		val = 0;

	return val;
}

/**
 * Initilizes the bit files
 */
void NodeList::initBits( bool val )
{
	vector<int> tmp;
	if( val == true )
		tmp.push_back(1);
	else
		tmp.push_back(0);
	bits.push_back(tmp);
}

/**
 * Initializes the input bit files
 */
void NodeList::initInputBits( )
{
	//init with 0's
	bool val = false;
	vector<int> tmp;
	if( val == true )
		tmp.push_back(1);
	else
		tmp.push_back(0);
	inputBits.push_back(tmp);
}

/**
 * Print all node names
 */
NodeList &NodeList::printNames(ostream &out)
{
	char c[10] = "Output";
	if (outputAll) 
	{
		strcpy(c,"Bool");
	}
		int j = 0;
		for (int i = 0; i < nodeSize(); i++) 
		{
			if(getNode(i).getType()==c)
			{
				out << "Column " << setw(5) << j << ": " << getNodeItem(i).getName() << endl;
				++j;
			}
		}

		out << setw(8) << " ";
		for (int i = 0; i < j; i++) 
		{
			out << setw(2) << (i/100) % 10;
		}
		out << endl;

		out << setw(8) << " ";
		for (int i = 0; i < j; i++) 
		{
			out << setw(2) << (i/10) % 10;
		}
		out << endl;

		out << setw(8) << " ";
		for (int i = 0; i < j; i++) 
		{
			out << setw(2) << i % 10;
		}
		out << endl;

		return *this;
}

/**
 * Calculate the average activity level for nodes
 * for the last n iterations where n is the value
 * of transient.
 */
void NodeList::calcAvg( void )
{
	for( int i = 0; i < outCalcSize(); ++i )
	{
		for( int j = transient; j < runTime; ++j )
			if( bits[i][j] == 1 )
			{
				avg[i] = avg[i]+1;
			}
		avg[i] = avg[i]/(runTime-transient);
	}

}

/**
 * Calculates the average activity level of nodes
 * based on the average period
 */
void NodeList::calcCurrAvg( void )
{
	int tmp;
	for( int i = 0; i < outCalcSize(); ++i )
	{
		if( avgPeriod > int(bits[i].size()) )
			tmp = 0;
		else
			tmp = bits[i].size() - avgPeriod;

		for( int j = tmp; j < int(bits[i].size()); ++j )
			if( bits[i][j] == 1 )
			{
				avg[i] = avg[i]+1;
			}
		if( tmp == 0 )
			avg[i] = avg[i]/(bits[i].size());
		else
			avg[i] = avg[i]/avgPeriod;
	}

}

/**
 * Prints the average activity levels to ostream out
 */
void NodeList::printAvg( ostream & out )
{
	float decAvg;	
	int Avg;
	for( int i = 0; i < outCalcSize(); ++i )
	{
		//      out << outCalc[i].getName() << ":  " << avg[i] << endl;
		decAvg = (avg[i]*100);
		Avg = int( decAvg + 0.5);
		//cerr << Avg << endl;		
		out << Avg << "  ";//setw(6);   decAvg = (int)(avg[i]*100);
	}
	out << endl;
}



/**
 * Prints nodes to ostream out
 */
void NodeList::printNodes( ostream & out)
{
	string nodeName;
	for( int i = 0; i < outCalcSize(); ++i )
	{
		nodeName = outCalc[i].getName();
		out << nodeName << endl;
	}
}

/**
 * Prints inputnodes to ostream out
 */
void NodeList::printInputs( ostream & out)
{
	string nodeName;
	for( unsigned int i = 0; i < inputs.size(); ++i )
	{
		nodeName = inputs[i].getName();
		out << nodeName << endl;
	}
}

/**
 * Prints each node's average to an individual file in the directory path
 */
void NodeList::printAvgSep( const string & path )
{
	ofstream out;
	float decAvg;
	int Avg;
	for( int i = 0; i < outCalcSize(); ++i )
	{
		string nodeName = outCalc[i].getName();
		string fileName = path + nodeName + ".mtb";        
		//string fileName = "../" + runName + "/NodesAvg/" + nodeName + ".mtb";        
		out.open( fileName.c_str(), ios::app );
		if (!out) 
		{
			cerr << "Unable to open " << fileName << " for node avg output" << endl;
			exit(1);
		}
		decAvg = (avg[i]*100);
		Avg = int( decAvg + 0.5);
		out << Avg << "      Dosages: ";
		for( unsigned int j = 0; j < dosageRec.size(); ++j )
		{
			out << dosageRec[j] << " ";
		}
		out << endl;
		out.close();
	}
}

/**
 * returns the recent average activity for 'node'
 * for the last n iterations where n is the 
 * value of transient
 */
int NodeList::getNodeAvg(string node, int transient)
{
	int rval;
	float total=0;
	bool set=false;
	if(transient>runTime)
		transient=runTime;
	for(int i=0; i<outCalcSize(); ++i)
	{
		if(node==outCalc[i].getName())
		{
			for(int j=runTime; j>(runTime-transient); --j)
			{
				if(bits[i][j]==1)
					++total;
			}
			set=true;
			rval=int(((total*100/transient)+0.5));
		}
	}
	if(!set)
		rval=-1;

	return rval;
}

/**
 * Prints the dosages to ostream out
 */
void NodeList::printDosages( ostream & out )
{
	//ofstream out;
	//string fileName = "stats/" + runName + "/all_dosage.mtb";        
	//string fileName = "../" + runName + "/NodesAvg/" + nodeName + ".mtb";        
	//out.open( fileName.c_str(), ios::app );
	for( int j = 0; j < int(dosageRec.size()); ++j )
		out << dosageRec[j] << " ";
	out << endl;

}


/**
 * Print bits by iterations  -> 1 line = state of the system
 */
void NodeList::printBits( ostream & out )
{
	for( int j = 1; j < runTime; ++j )
	{
		for( int i = 0; i < outCalcSize(); ++i )
		{
			out << bits[i][j] << "  ";
			//cerr << outCalc[i].getName() << ":  ";
		}
		out << endl;
		//out << "    ";
	}
	out << endl;
}

/**
 * Prints input node bits to ostream out
 */
void NodeList::printInputBits( ostream & out )
{ 
	int numOfInputs = getNumOfInputs();
	for( int j = 1; j <= runTime-1; ++j )
	{
		for( int i = 0; i < numOfInputs; ++i )
		{
			out << inputBits[i][j] << "  ";
			//cerr << inputs[i].getName() << ":  ";
		}
		out << endl;
		//out << "    ";
	}
	out << endl;

}

string NodeList::getInputNames(int node)
{
	/*
	 * getInputNames returns a string of input node
	 * names in the order they are in the NodeList
	 */
	int j=0;
	string arr = "";
	for(int i=0; i<nodeSize(); ++i)
	{
		if(getNode(i).getType()=="Input")
		{
			if(j==node)
			{
				arr=getNodeItem(i).getName();
				i=nodeSize();
			}
			++j;
		}
	}
	return arr;
}

string NodeList::getOutNames(int i)
{
	/*
	 * getOutNames returns a string of output node
	 * names in the order they are in the NodeList
	 */
	return outCalc[i].getName();
}

void NodeList::setInputs(string node, int level)
{
	/*
	 * setInputs sets a node's input
	 */
	for(int j=0; j<nodeSize(); ++j)
	{
		if(getNode(j).getType()=="Input"&&getNodeItem(j).getName()==node)
		{
			static_cast<InputNode &>(getNode(j)).setLevels(level, runTime);
		}
	}
}

int NodeList::getOutLevels(int j)
{
	/*
	 * getOutLevels returns an array of integers
	 * reflecting the output levels of the output nodes
	 */
	float fAvg=(avg[j]*100);
	int i = int(fAvg+0.5);
	return i;
}

bool NodeList::isBool(string s)
{
	/**
	 * returns true if s is a boolean node in the network
	 */
	bool rval = false;
	for(int i=0; i<nodeSize(); ++i)
	{
		if(getNode(i).getType()=="Bool"&&nodes[i].getName()==s)
		{
			rval = true;
		}
	}
	return rval;

}
