/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <time.h>
#include "NodeList.h"
#include <iostream>
#include <fstream>
#include <errno.h>
#include <cstring>
#include "NodeList.h"
#include "Node.h"
#include "lib/Matrix.h"
#include "Simulation.h"

using namespace std;

Simulation::Simulation(void)
{
	/**
	 * Simulation constructor that initializs variables
	 */
	noOutFiles=false;
	usePlugins=true;
	ROOTDIR = "../CCoutput";
	SIMDIR = ROOTDIR+"/stats";
	NODESAVGDIR = "NodesAvg";
	BITSDIR = "bits";
	LOGICDIR = "logic";
	EXTSDIR = "exts";
	DUMP = " > /dev/null 2>&1";
	n = 1; //number of times to run simulation
	ext = ".csv";
	bitsFileName = "";
	inputBitsFileName = "";
	in_specsFile = "";                        
	in_logicFile = "";
	in_patternSpecsFile = "";
	in_patternNodesFile = "";
	iNodeList = "";
	TT2ND = false;
	rand_nodeInputs=false;
	ND2TT = false;
	patternAnal = false;
	rand_init = false;
	visMode = false;
	analysisMode = false;
	iTables = "";
	mutRate_nodes = 0;
	mutRate_logic = 0;
	rand_logicMut=false;
	cycleSize = 0;
	outputFile = "";
	visualFile = "";
	progStart=clock();
	noBits = false;
}

Simulation::~Simulation(void)
{
	/**
	 * Simulation destructor, currently empty
	 */
	;;;
}

void Simulation::setup(int argc, char** argv)
{	
	/**
	 * Setup, takes in argv, its length and 
	 * cycles through argv to initialize flags, 
	 * it then creates apropriate directories 
	 * for the Simulation.
	 */
	for (int i = 1; i < argc; i++) 
	{

		if (strcmp(argv[i],"-inodelist") == 0)
		{
			i++;
			iNodeList = argv[i];
		}
		else if (strcmp(argv[i],"-noBits") == 0)
		{
			i++;
			noBits=true;
		}
		else if (strcmp(argv[i],"-noPrint") == 0)
		{
			i++;
			noOutFiles=true;
		}
		else if (strcmp(argv[i],"-noPlugins") == 0)
		{
			i++;
			usePlugins=false;
		}
		else if (strcmp(argv[i],"-itables") == 0)
		{
			i++;
			iTables = argv[i];
		}
		else if (strcmp(argv[i],"-TT2ND") == 0)
		{
			TT2ND = true;
		}
		else if (strcmp(argv[i],"-ND2TT") == 0)
		{
			ND2TT = true;
		}
		else if (strcmp(argv[i],"-o") == 0)
		{
			i++;
			outputFile = argv[i];
		}
		else if (strcmp(argv[i],"-h") == 0||strcmp(argv[i],"-help")==0) 
		{
			print_help();
			exit(1);
		}
		else if (strcmp(argv[i],"-ispecs") == 0) 
		{
			i++;
			in_specsFile = argv[i];            
		} 
		else if (strcmp(argv[i],"-ilogic") == 0) 
		{
			i++;
			in_logicFile = argv[i];

		} 
		else if (strcmp(argv[i],"-vis") == 0) 
		{
			i++;
			visualFile= argv[i];
			visMode=true;
		}  
		else if (strcmp(argv[i],"-n") == 0) 
		{
			i++;
			n = atoi(argv[i]);
		}
		else if (strcmp(argv[i],"-calc") == 0) 
		{
			i++;
			runName = argv[i];
			calc = true;
			outputAll = true;
		} 
		else if (strcmp(argv[i],"-v") == 0) 
		{
			verbose = true;
		}
		else if (strcmp(argv[i],"-A") == 0) 
		{
			outputAll = true;
		}
		else if (strcmp(argv[i],"-rInputs") == 0) 
		{
			rand_nodeInputs = true;
			i++;
			mutRate_nodes = atoi(argv[i]);
		}
		else if (strcmp(argv[i],"-r_mut") == 0) 
		{
			rand_logicMut = true;
			i++;
			mutRate_logic = atoi(argv[i]);
		} 
		else if (strcmp(argv[i],"-randInit") == 0) 
		{
			rand_init = true; //random initial conditions
		} 
		else if (strcmp(argv[i],"-analysis") == 0)
		{
			analysisMode = true;//analysis mode
		}
		else 
		{
			cerr << "Invalid argument: " << argv[i] << endl;
			print_help();
			exit(1);
		}
	} //end for 

	if (!visout) 
	{
		cerr << "No output file given for visual mode" << endl;
		exit(1);
	}

	cmd = "mkdir " + ROOTDIR + DUMP;
	system(cmd.c_str());


	calcModeSetup();
	visModeSetup();

}

void Simulation::calcModeSetup(void)
{

	/**
	 * creates the appropriate directories for calc mode
	 */
	if(calc&&!noOutFiles)
	{
		cmd = "mkdir " +  SIMDIR + DUMP; 
		system(cmd.c_str());
		runDir = SIMDIR + "/" + runName;
		cmd = "mkdir " +  runDir + DUMP; 
		system(cmd.c_str());
		cmd = "mkdir " + runDir + "/" + NODESAVGDIR + DUMP;  
		system(cmd.c_str());
		cmd = "mkdir " + runDir + "/" + BITSDIR + DUMP;  
		system(cmd.c_str());
		cmd = "mkdir " + runDir + "/" + LOGICDIR + DUMP;  
		system(cmd.c_str());
		inputFileCheck();
		cmd = "cp " + in_specsFile + " " + SIMDIR + "/" + runName + "/specs.txt";  
		system(cmd.c_str());	
		cmd = "cp " + in_logicFile + " " + SIMDIR+"/"+runName+"/logic/logic.txt";
		system(cmd.c_str());
		avgFileName = SIMDIR + "/" + runName + "/" + "allNodes_avg" + ext;
	}
}

void Simulation::visModeSetup(void)
{
	/**
	 * opens the appropriate file for vis mode
	 */
	if(visualFile!="")
	{
		cmd = ROOTDIR+"/"+visualFile;
		visout.open(cmd.c_str());
		if (!visout) 
		{
			cerr << "Unable to open " << visualFile << " for input" << endl;
			exit(1);
		}
		inputFileCheck();
	}

}

void Simulation::inputFileCheck(void)
{
	/**
	 * exits if the nessicary input files have not been provided
	 */
	if ( (in_logicFile == "" && !rand_init) || in_specsFile == "" ) 
	{
		cerr << "The input file for logic and specs needs to be given" << endl;
		exit(1);
	}
}

void Simulation::load(void)
{
	/**
	 * load; currently unimplimented
	 */
	;;;
}

void Simulation::inputGenerator(void)
{
	/**
	 * Uses the generator class to create a logic or truth table file
	 */
	PluginWrapper p;
	p.open("Plugins/FileConverter/FileConverter.so");
	p.getPlugin()->setSIMDIR(SIMDIR);
	p.getPlugin()->setRunName(runName);
	p.getPlugin()->setDUMP(DUMP);
	p.getPlugin()->setROOTDIR(ROOTDIR);
	p.getPlugin()->setOutFile(outputFile);
	p.getPlugin()->setup("Plugins");
	p.getPlugin()->setLogic(in_logicFile);
	p.getPlugin()->setSpecs(in_specsFile);
	if(TT2ND)
	{
		p.getPlugin()->midSim();
	}
	if(ND2TT)
	{
		p.getPlugin()->postSim();
	}


}

void Simulation::run(void)
{
	/**
	 * This function calls the differnt modes of 
	 * ChemChains based upon which flags were used.
	 * The different modes are the File Converter,
	 * the analysis mode, and the main simulation.
	 */
	if ((TT2ND==true)||(ND2TT==true))
	{
		inputGenerator();

	}
	else if (analysisMode==true)
	{
		load();

	}
	else if (calc||visMode)
	{
		simulate();
	}
	else
	{
		cerr<<"Choose a mode"<<endl;
	}

}

void Simulation::simulate(void)
{

	if(usePlugins)
	{
		pManager.setROOTDIR(ROOTDIR);
		pManager.setOutFile(outputFile);
		pManager.setSIMDIR(SIMDIR);
		pManager.setRunName(runName);
		pManager.setDUMP(DUMP);
		pManager.setLogic(in_logicFile);
		pManager.setSpecs(in_specsFile);
		pManager.loadPlugins("Plugins");
	}
	/**
	 * This function represents the bulk of the program
	 * It creates a Boolean Network in the form of a NodeList
	 * and evaluates the network a set number of times,
	 * making some analysis throughout.
	 */


	//plugin manager initilization activities
	if(usePlugins)
	{
		pManager.initialize();
		if(pManager.fileChange())
		{
			in_specsFile=pManager.getSpecs();
			in_logicFile=pManager.getLogic();
		}
	}

	for( int run = 1; run <= n; ++run )
	{
		cerr << "Running Simulation: " << run << endl;
		//Convert integer to string
		char sim [10];
		sprintf( sim,"%d", run );

		//********* Early Simulation Operations *********//
		// Any operations that need to take place at the beginning of a new 
		// simulation - i.e., generate new logic

		//plugin manager pre simulation activities
		if(usePlugins)
		{
			pManager.preSim();
			if(pManager.fileChange())
			{
				in_specsFile=pManager.getSpecs();
				in_logicFile=pManager.getLogic();
			}
		}

		//********* End of Early Simulation Operations *********//


		//open specs and logic files
		ifstream in_specs, in_logic;
		in_specs.open(in_specsFile.c_str());
		if (in_specs.fail()) 
		{
			cerr << "Unable to open " << in_specsFile << " for specs file" << endl;
			exit(1);
		}
		in_logic.open(in_logicFile.c_str());
		if (in_logic.fail()) 
		{
			cerr << "Unable to open " << in_logicFile << " for logic file" << endl;
			exit(1);
		}

		//create nodelist
		NodeList NList;
		NList.setRunName( runName );  

		// Process logic and specs files
		if (verbose) cerr << "Node Descriptions" << endl;
		in_logic >> NList;
		in_specs >> NList;
		in_logic.close();
		in_specs.close();

		//checks to make sure that every node named 
		//as an input to another node exists
		if(verbose) cerr<<"Checking Network Consistency"<<endl;
		bool inconsistent=false;
		for (int i = 0; i < NList.nodeSize(); i++) 
		{ 
			if(!NList.getNodeItem(i).hasNode())
			{
				cerr<<"Node "<<NList.getNodeItem(i).getName()<<" Not Declared"<<endl;
				inconsistent=true;
			}

		}
		if(inconsistent)
		{
			cerr<<"Check your network specs"<<endl;
			exit(1);
		}

		NList.setMutants();

		//sets up for calc mode anlysis'
		if(calc && !noOutFiles)
		{
			string nodeLabelsFileName = SIMDIR + "/" + runName + "/node_labels.csv";
			ofstream outNodeLabels;
			outNodeLabels.open( nodeLabelsFileName.c_str() );
			if (!outNodeLabels) 
			{
				cerr << "Unable to open " << nodeLabelsFileName 
					<< " to output labels!" << endl;
				exit(1);
			} 

			string inputLabelsFileName = SIMDIR + "/" + runName + "/input_labels.csv";
			ofstream outInputLabels;
			outInputLabels.open( inputLabelsFileName.c_str() );
			if (!outInputLabels) 
			{
				cerr << "Unable to open " << inputLabelsFileName 
					<< " to output labels!" << endl; 
				exit(1);
			}
			cerr << "	Printing list of nodes ... ";
			NList.printNodes(outNodeLabels);    
			cerr << "done" << endl;
			cerr << "	Printing list of stimuli ... ";
			NList.printInputs(outInputLabels);    
			cerr << "done" << endl;
			outNodeLabels.close();
			outInputLabels.close();
		}

		NList.setOutputAll(outputAll);
		if(visMode) NList.printNames(visout);

		// Check for errors in descriptionsInput:IN1:TruInput:IN1:True:R:F:1:5-10:29,31:45-50:39,71:95-100:29e:R:F:1:5-10:29,31:45-50:39,71:95-100:29
		// Execute Simulation
		if (verbose) cerr << "----------- Begin Simulation ------------" << endl;
		int t = 1;

		if( calc && ( NList.ptsSize() > 0 ) )
		{
			cmd = "mkdir " + SIMDIR + "/" + runName + "/SnapshotAnalysis"+DUMP;  
			system(cmd.c_str());
		}
		int analPt = NList.getNextAnalPt();

		//this is the main loop representing an iteration
		while (t < NList.getRunTime()) 
		{
			if(visMode) visout << setw(8) << t << " ";
			if (verbose) cerr << ">>>>> Time " << t << endl;
			if (verbose) cerr << "Processing " << NList.outputSize() << " output nodes" << endl;

			for (int i = 0; i < NList.outputSize(); i++) 
			{

				if (verbose) 
					cerr << "Output Node " << NList.outputs[i].getName() << endl;
				bool result = NList.outputs[i].getNode().evaluate(NList, t);
				if (verbose)
				{
					cerr << "Node "<< NList.outputs[i].getName()<<" Evaluated to "<<result<<endl;
				}
			}

			if (visMode) NList.visPrint(visout, NList);
			if (calc) NList.bitsPrint(NList);

			ipause();
			if(usePlugins)
			{
				pManager.setT(t);
				pManager.setAnalysisPoint(analPt);
			}
			//Perform incremental output analysis during the simulation 
			//based on the OutputAnalysis option
			if( calc && (( NList.ptsSize() > 0 )  || (analPt > 0)) )
			{
				if( (analPt > 0) && ( (t+1) == analPt ) )
				{
					stringstream ss;
					ss << analPt;
					string folder;
					ss >> folder;

					cmd = "mkdir " + SIMDIR + "/" + runName + "/SnapshotAnalysis/" + folder+DUMP;  
					system(cmd.c_str());
					string avgFileName_pt=SIMDIR+"/"+runName+"/SnapshotAnalysis/"+folder+"/allNodes_avg"+ext;
					outAvg.open(avgFileName_pt.c_str(), ios::app);
					if (!outAvg) 
					{
						cerr << "Unable to open " << avgFileName_pt 
							<< " for Avg. input" << endl;
						exit(1);
					}

					NList.calcCurrAvg();
					NList.printAvg( outAvg );
					outAvg.close();

					cmd = "mkdir " + SIMDIR + "/" + runName + "/SnapshotAnalysis/" + folder + "/NodesAvg"+DUMP;
					system(cmd.c_str());

					string name = SIMDIR+"/"+runName + "/SnapshotAnalysis/" + folder+"/NodesAvg/";
					NList.printAvgSep(name);


					analPt = NList.getNextAnalPt();
				}
			}
			if(usePlugins)
				pManager.midSim(NList);

			t += 1;
		}//end for loop representing simulation step

		//post iteration calculation work
		if( calc && !noOutFiles)
		{
			if(!noBits)
			{
				bitsFileName = SIMDIR + "/" + runName + "/" + BITSDIR + "/" + "network_bits" + sim + ext;
				outBits.open(bitsFileName.c_str() );
				if (!outBits) 
				{
					cerr << "Unable to open " << bitsFileName 
						<< " for Bit output" << endl;
					exit(1);
				}

				inputBitsFileName = SIMDIR + "/" + runName + "/" + BITSDIR + "/" + "input_bits" + sim + ext;
				outInputBits.open( inputBitsFileName.c_str() );
				if (!outInputBits) 
				{
					cerr << "Unable to open " << inputBitsFileName 
						<< " for Bit output" << endl;
					exit(1);
				}
			}
			outAvg.open(avgFileName.c_str(), ios::app);
			if (!outAvg) 
			{
				cerr << "Unable to open " << avgFileName 
					<< " for Avg. input" << endl;
				exit(1);
			}

			string dosageFileName = SIMDIR + "/" + runName + "/input_dosages" + ext;        
			outDosage.open(dosageFileName.c_str(), ios::app);
			if (!outDosage) 
			{
				cerr << "Unable to open " << dosageFileName 
					<< " for dosage data output!" << endl;
				exit(1);
			}

			cerr << "	Calculating averages ... ";
			NList.calcAvg();
			cerr << "done" << endl;

			cerr << "	Printing averages ... ";
			NList.printAvg(outAvg);
			outAvg.close();
			cerr << "done" << endl;

			cerr << "	Printing averages for individual nodes ... ";
			string path = SIMDIR + "/" + runName + "/NodesAvg/";        
			NList.printAvgSep( path );
			cerr << "done" << endl;

			cerr << "	Printing dosages ... ";
			NList.printDosages( outDosage );
			outDosage.close();
			cerr << "done" << endl;

			if(!noBits)
			{
				cerr << "	Printing output bit files ... ";
				NList.printBits(outBits);
				outBits.close();
				cerr << "done" << endl;
				cerr << "	Printing input bit files ... ";
				NList.printInputBits(outInputBits);
				outInputBits.close();
				cerr << "done" << endl;
			}




			//********* Late Simulation Operations *********//
			// Any operation/calculation at the end of each simulation
			// i.e., to calculate stats in real time instead of waiting 
			// till the end of the simmulation cycle

			if(usePlugins)
			{
				pManager.setT(t);
				pManager.postSim();
			}

			//********* End of Late Simulation Operations *********//


		} // end if( calc )

	} //end for loop 
	//***************************************** End of Simulation ****************************************//
	cout<<"Simulation runs completed!"<<endl;

	pManager.close();
}	


void Simulation::analysis()
{
	/**
	 * This method runs differnt analysis through
	 * the analysis object based upon which 
	 * analysis flags were used at the command line
	 */

	//to be implemented
}


void Simulation::ipause()
{
	/**
	 * This function pauses the program,
	 * used only in verbose mode
	 */
	if (verbose) {
		char c;
		cout << "Press any key to continue." << endl;
		cin.get(c);
	}
}


void Simulation::print_help()
{
	/**
	 * A help function that lists the flags and their uses
	 */
	string space = "	";
	int width1 = 20;
	int width2 = 45;
	cout<<"Usage: ChemChain(version 1.20) [OPTIONS]"<< endl;
	cout<<endl;
	cout<<endl<<"Modes:"<<endl;
	cerr<<space<<left<<setw(width1)<<"-calc xxx"<<setw(width2)
		<<"Calculation mode and experiment name"<<endl;
	cout<<space<<left<<setw(width1)<<"-ND2TT"<<setw(width2)
		<<"Convert network descriptor file into individual truth tables (Requires -ilogic)"<<endl;
	cout<<space<<left<<setw(width1)<<"-TT2ND"<<setw(width2)
		<<"Convert truth tables into a network descriptor file (Requires -itables and -inodelist flags)"<<endl;
	cout<<space<<left<<setw(width1)<<"-vis xxx"<<setw(width2)
		<<"Instantiates the visual mode and sets the output file name"<<endl;

	cout<<endl<<"Files:"<<endl;
	cout<<space<<left<<setw(width1)<<"-o xxx"<<setw(width2)
		<<"File Converter output"<<endl;
	cout<<space<<left<<setw(width1)<<"-ispecs xxx"<<setw(width2)
		<<"Specification file name (REQUIRED)"<<endl;
	cout<<space<<left<<setw(width1)<<"-ilogic xxx"<<setw(width2)
		<<"Network logic file name (REQUIRED)"<< endl;
	cout<<space<<left<<setw(width1)<<"-n xxx"<<setw(width2)
		<<"Number of consecutive simulations"<<endl;
	cerr<<space<<left<<setw(width1)<<"-itables xxx"<<setw(width2)
		<<"File path for list of truthtables, and the nodelist"<<endl;
	cerr<<space<<left<<setw(width1)<<"-inodelist xxx"<<setw(width2)
		<<"File for List of Nodes, Used with TT2ND"<<endl;

	cout<<endl<<"General Flags: "<<endl;
	cerr<<space<<left<<setw(width1)<<"-v"<<setw(width2)
		<<"Verbose mode"<<endl;
	cerr<<space<<left<<setw(width1)<<"-A"<<setw(width2)
		<<"Evaluates and prints all nodes in the network (used with -vis)"<<endl;
	cerr<<space<<left<<setw(width1)<<"-noBits"<<setw(width2)
		<<"Suppress calculcation of bitfiles"<<endl;
	cerr<<space<<left<<setw(width1)<<"-noPrint"<<setw(width2)
		<<"Suppress printing outputfiles"<<endl;
	cerr<<space<<left<<setw(width1)<<"-noPlugins"<<setw(width2)
		<<"Suppress running plugins"<<endl;
	cerr<<space<<left<<setw(width1)<<"-randInit"<<setw(width2)
		<<"Generate random network initial conditions for each simulation"<<endl;

}
