/**
 * @section LICENSE
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef NODELIST_H
#define NODELIST_H

#include <vector>
#include <iostream>
#include <iomanip>
#include <string>
#include <time.h>
#include <stdexcept>
#include <fstream>
#include <algorithm>
#include <functional>
#include "Node.h"
#include "lib/StringUtils.h"

void ipause();

	/**
	 * @section DESCRIPTION
	 * The node item functions as a wrapper to a node
	 */
class NodeItem {
public:
    //default constructor
    NodeItem();
    //copy constructor
    NodeItem(const NodeItem &);
    //destructor
    ~NodeItem();
    ///assignment operator
    NodeItem & operator = (const NodeItem &);
    void setName(string n);
    void setNode(const Node &);
    void setNode(const InputNode &);
    void setNode(const OutputNode &);
    void setNode(const BooleanNode &);
    void setNode(const DelayNode &);
    void setNode(const SustainNode &);
    bool removeNode();
	///return name of the contained node
    string getName() const {return name;};
    Node & getNode();
    bool hasNode() const;
protected:
    string name;
private:
    Node *node;
};


	/**
	 * @section DESCRIPTION
	 * The Nodelist tracks and maintains the nodes within the network
	 */
class NodeList {
public:
    NodeList();
    friend istream &operator >>(istream &in, NodeList &nl);
    void visPrint(ostream &out, NodeList &nl);
    void bitsPrint(NodeList &nl);
    int find(string n);
    int add(NodeItem &i);
	///returns the i'th node on the array of nodes
    Node &getNode(int i) {return nodes[i].getNode();};
	///returns the i'th node item on the array of node items
    NodeItem &getNodeItem(int i) { return nodes[i];}
	///returns the i'th Input node
    Node &getInputNode(int i) {return inputs[i].getNode();};
	///returns the name of the i'th node on the nodelist
    string getName(int i) {return nodes[i].getName();};
	///sets the runtime to r and returns *this
    NodeList &setRunTime(int r) {runTime = r+1; return *this;};
	///returns the runtime
    int getRunTime() {return runTime;};
	///returns the name of the simulation
    string getRunName() {return runName;};
	///sets transient to r and returns *this
    NodeList &setTransient(int r) {transient = r; return *this;};
	///returns transient
    int getTransient() {return transient;};
	///sets window to w and returns *this
    NodeList &setWindow( int w) { window = w; return *this;};
	///sets scalar to s and returns *this
    NodeList &setScalar( int s) { scalar = s; return *this;};
	///returns window
    int getWindow(void) {return window;};
	///returns scalar
    int getScalar(void) {return scalar;};
    int getNextAnalPt( void );
    void sortAnalPts( void );
    NodeList &printNames(ostream &out);
    void printNodes(ostream &out);
    void printInputs(ostream &out);
    vector<NodeItem> outputs;
    vector<NodeItem> outCalc;
    int inOutCalc( int );
	void inOutInit(void);
	vector<int> inOutIndex;
    int getNodeIndex( string );
    vector <float> avg; // Gong added in 11/11/05 to compute the average value for each Boolean node
    vector < vector<bool> > bits;
	///return the size of the ouput calculations
    int outCalcSize() {return static_cast<int>(outCalc.size());}
	///returns the number of inputs
    int getNumOfInputs() {return inputs.size();}
	///returns the number of outputs
    int outputSize() {return static_cast<int>(outputs.size());};
	///returns the number of analysis points
    int ptsSize() { return analPts.size(); };//rename this more appropriately
    void printAnalPts( void );
    void setMutants( void );
	///sets runName to str
    void setRunName( const string & str ) { runName = str; };
	bool readParams(istream &, InputNode &, bool , NodeList & nl );
    void calcAvg( void );
    void printAvg( ostream &);
    void calcCurrAvg( void );
    void printAvgSep( const string & );
    void printDosages( ostream &  );
    void printBits( ostream & );
    void printInputBits( ostream & );
    void initBits( bool);
    void initInputBits( );
	///set noisyInputs to true to allow for noise
    void setNoisyInputs(void) { noisyInputs = true; };
	///returns noisyInputs
    bool isNoisyInputs(void) { return noisyInputs; };
	void addAnalPt( const int & );
    vector<int> dosageRec;
	vector<int> analPts;
	int avgPeriod;
	///return the total number of nodes
    int nodeSize() {return static_cast<int>(nodes.size());};
	int getNodeAvg(string, int);
	void setOutputAll(bool b){outputAll=b;};
	string getInputNames(int);
	string getOutNames(int);
	void setInputs(string , int);
	int  getOutLevels(int);
	bool isBool(string);
protected:	
    vector < vector<bool> > inputBits;
    void addOutput(NodeItem &i);
    void addInput(NodeItem &i);
    void addOutCalc(NodeItem &i);// Gong added to compute the average
    void addNode(NodeItem &i);
    void addMutant( Mutant & );
    vector<Mutant> mutants;//tracks mutant nodes
    vector<NodeItem> nodes;//tracks all nodes
    vector<NodeItem> inputs;//tracks input nodes
    int runTime;
	string runName;
    int transient;
    int window;
    int scalar;
    bool noisyInputs;
	bool outputAll;
        
};
#endif
