/**
 * @section LICENSE
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#ifndef PLUGIN_HPP
#define PLUGIN_HPP
#include <iostream>
#include <fstream>
#include <errno.h>
#include <string.h>
#include "Plugin.hpp"

using namespace std;

	/**
	 * @section DESCRIPTION
	 * The Plugin class is a virtual class from which
	 * plugins that are to be loaded dynamically may
	 * be loaded. They inherit from this class so that
	 * we may catch a pointer to an instance of a 
	 * plugin in a pointer to type Plugin and call the 
	 * functions on it.
	 */

class Plugin {
	public:
		Plugin(){};

		virtual ~Plugin(){};

		virtual void setup(string) = 0;
		//returns the version of the plugin
		virtual int getVersion() const = 0;
		//returns the location of the new specs file
		virtual string getSpecs() = 0;
		//returns the location of a new logic file
		virtual string getLogic() = 0;
		//returns true if there is a new specs file
		virtual bool nSpecs()=0;
		//returns true if there is a new logic file
		virtual bool nLogic()=0;
		//returns the name of a node that this plugin wants to see the level of
		virtual string retrieveNodeLevel() = 0;
		//takes in the name of the node and it's current level for analysis
		virtual void sendNodeLevel(string, int) =0;
		//returns the name of an input node and the level it wants the node set to
		virtual pair<string, int> setInputNodeLevel() =0;

		//These three functions are called throughout the simulation
		//so that this plugin may know what is going on and may do
		//any nessicary calculations.
		//
		//this function is called before any simulations occur
		virtual void initialize() = 0;
		//this function is called during simulation setup before each simulation
		virtual void preSim() = 0;
		//this function is called every step during the simulation
		virtual void midSim() = 0;
		//this function is called after each simulation is over
		virtual void postSim() = 0;
		//this function is called after all simulations are over
		virtual void close() = 0;

		//setter methods for data to be shared with the plugin
		void setROOTDIR(string s){ROOTDIR=s;};
		void setSIMDIR(string s){SIMDIR=s;};
		void setOutFile(string s){outFile=s;};
		void setRunName(string s){runName=s;};
		void setT(int i){t=i;};
		void setDUMP(string s){DUMP=s;};
		void setAnalysisPoint(int i){analysisPt=i;};
		void setLogic(string s){logic=s;};
		void setSpecs(string s){specs=s;};

	protected:
		int t;
		string outFile;
		string ROOTDIR;
		string SIMDIR;
		string runName;
		string DUMP;
		string logic;
		string specs;
		string name;
		int analysisPt;


};

// the types of the class factories
typedef Plugin* create_t();
typedef void destroy_t(Plugin*);

#endif
