/**
 *   Copyright (C) 2004-2008 Tomas Helikar & Mathbio Research Group, Department of Mathematics, University of Nebraska at Omaha 
 * 
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <iostream>
#include <fstream>
#include <errno.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <string.h>
#include "Plugin.hpp"
#include "PluginManager.h"

using namespace std;

PluginManager::PluginManager(void)
{
	/**
	 * PluginManager constructor, initializes values
	 */
	t=0;
	lastCalc=0;
	SIMDIR="";
	RunName="";
	DUMP="";
	analysisPt=0;
	logic="";
	specs="";
	outFile="";
	ROOTDIR="";
}

PluginManager::~PluginManager(void)
{
	/**
	 * PluginManager destructor, empty
	 */
	;;;
}

void PluginManager::loadPlugins(string path)
{
	/**
	 * loadPlugins opens path/config.txt and loads all
	 * plugins listed in that file
	 */
	vector<string> names;
	ifstream config;
	string s;
	config.open((path+"/config.txt").c_str());
	if (config.fail())
	{
		cerr<<"Error loading plugins:Config file failed to open!"<<endl;
		exit(1);
	}

	char arr[1024];
	config.getline(arr, 1024);
	do
	{
		s="";
		for(int i =0; arr[i]!='\0'; ++i)
			s+=arr[i];
		if(s[0]!='#'&&s!="")
		{
			PluginWrapper p;
			//int i=int(plugins.size());
			plugins.push_back(p);
			names.push_back(s);
		}

		config.getline(arr, 1024);
	} while(!config.eof());
	config.close();
	
	for(int i = 0; i<int(plugins.size()); ++i)
	{
		plugins[i].open(path+"/"+names[i]+"/"+names[i]+".so");
		plugins[i].getPlugin()->setSIMDIR(SIMDIR);
		plugins[i].getPlugin()->setRunName(RunName);
		plugins[i].getPlugin()->setDUMP(DUMP);
		plugins[i].getPlugin()->setROOTDIR(ROOTDIR);
		plugins[i].getPlugin()->setOutFile(outFile);
		plugins[i].getPlugin()->setup(path);
	}
}

void PluginManager::initialize(void)
{
	/**
	 * initialize is called once while the program is starting up, 
	 * It allows plugins to setup any variables they need to.
	 *
	 */
	for(int i=0; i<int(plugins.size()); ++i)
	{
		Plugin * p = plugins[i].getPlugin();
		p->setLogic(logic);
		p->setSpecs(specs);
		p->initialize();
	}
}

void PluginManager::preSim(void)
{
	/**
	 * preSim is called before each new Simulation, in case any 
	 * plugin wants to change input files every simulation.
	 * It provides to all plugins the specs and logic files,
	 * and then asks them if they would like to make any changes.
	 * If there are modifications to an input file the 
	 * PluginManager takes the new file and provides it to
	 * the simulation, and to all other plugins as well.
	 *
	 * It is important to note that plugins are called in the 
	 * order they are listed in the config file and if pluginOne
	 * makes a change to the files and then pluginTwo makes a change
	 * pluginOne is not consulted. Because of this multiple plugins
	 * performing similar modifications may cause erroneuos behavior.
	 */
	for(int i=0; i<int(plugins.size()); ++i)
	{
		Plugin * p = plugins[i].getPlugin();
		p->preSim();
		if(p->nSpecs())
		{
			specs=p->getSpecs();
			inFileChange=true;
		}
		if(p->nLogic())
		{
			logic=p->getLogic();
			inFileChange=true;
		}
		if(inFileChange)
		{
			for(int j=0; j<i; ++j)
			{
				plugins[j].getPlugin()->setLogic(logic);
				plugins[j].getPlugin()->setSpecs(specs);
			}
		}
	}
}

void PluginManager::midSim(NodeList & n)
{
	/**
	 * midSim is called during every simulation step,
	 * PluginManager then provides to each plugin the current
	 * level of any requested boolean node as well as then
	 * setting the level of any requested Input node.
	 *
	 * As with preSim, the plugins are called in the order
	 * provided in config.txt and multiple plugins doing
	 * similar things may cause errors.
	 */
	for(int i=0; i<int(plugins.size()); ++i)
	{
		Plugin * p = plugins[i].getPlugin();
		p->setT(t);
		p->setAnalysisPoint(analysisPt);

		//find out the levels of any input nodes the
		//plugin might want
		string s= p->retrieveNodeLevel();
		while(s!="")
		{
			int i =0;
			for(int j = 0; j<n.outputSize(); ++j)
			{
				if(n.getOutNames(j)==s)
					i=j;
			}

			if(lastCalc!=t)
			{
				n.calcAvg();
				lastCalc=t;
			}

			p->sendNodeLevel(s,n.getOutLevels(i));
			s=p->retrieveNodeLevel();
		}

		p->midSim();

		//change the input node levels of any input
		//nodes that the plugin might want to change
		pair<string, int> in = p->setInputNodeLevel();
		while(in.first!="")
		{
			n.setInputs(in.first, in.second);
			in = p->setInputNodeLevel();
		}


	}	
}

void PluginManager::postSim(void)
{
	/**
	 * postSim calls the postSim function on each plugin
	 */
	for(int i=0; i<int(plugins.size()); ++i)
	{
		plugins[i].getPlugin()->postSim();
	}
}

void PluginManager::close(void)
{
	/**
	 * close ensures that each plugin is closed
	 */
	for(int i=0; i<int(plugins.size()); ++i)
	{
		plugins[i].getPlugin()->close();
		plugins[i].close();
	}
}
