#!/usr/bin/perl -w

# Amplicon file generator
#
# Michael Zhang, Johan Stenberg
# Copyright Stanford University, 2007
#
# Purpose: Create amplicon coordinates for a list of fragments.
# Usage: perl amplicon_generator.pl -i [fragment (PieceMaker output) input file] -o [amplicon output file]
# Input: 
#    1. Fragment description (Target_Name|Enzymes|[+-] polarity|fragment_number)
#    2. Parent Accession/ID (Should be versioned, e.g. NC_000001.9 for chr. 1)
#    3. Start coordinate of fragment
#    4. Stop coordinate of fragment
#	 5. Flap index (from 5' end)
#	 6. Fragment sequence
# Output:
#    1. Fragment ID
#    2. Parent Accession/ID (Should be versioned, e.g. NC_000001.9 for chr. 1)
#    3. Start coordinate of amplicon
#    4. Stop coordinate of amplicon
#    5. Polarity



use strict;

use FileHandle;
use Getopt::Long;

# add custom library directory to the path
use FindBin;
use lib $FindBin::Bin."/../lib";

use Log::Log4perl qw/:easy/;


# initialize log for perl level to error
Log::Log4perl->easy_init($WARN);
my $logger = get_logger();


my ($fragment_infile, $amplicon_outfile);

GetOptions(
	"i=s" => \$fragment_infile,
	"o=s" => \$amplicon_outfile,
);

# if either input or output files or flank size were not specified, output usage
if (!defined($fragment_infile) || !defined($amplicon_outfile))
{
	&usage();
}

# Create input filehandle
my $in_fh = new FileHandle($fragment_infile);
if (!defined($in_fh))
{
	$logger->error( "Could not open file: $fragment_infile!");
	exit(1);
}

# Create output filehandle
my $out_fh = new FileHandle(">$amplicon_outfile");
if (!defined($out_fh))
{
	$logger->error("Could not open file: $amplicon_outfile!");
	exit(1);
}

# Print target file header
print $out_fh "# created: " . gmtime() . "\n";
print $out_fh "# fragment file: $fragment_infile\n";

# iterate through ROI coordinate input file
while (<$in_fh>) {
	if (/^\#/) {
		next;
	}
	chomp;
	
	my ($target_desc, $g_accession, $frag_start, $frag_stop, $flap_index, $frag_seq) = split /\t/;
	
	my @target_desc_ary = split(/\|/, $target_desc);
	my $polarity = $target_desc_ary[2];
	my $amplicon_name = $target_desc;
	
	# replaces all spaces with underscores
# 	$amplicon_name =~ s/ /_/g;
	# replaces all forward slashes with plus signs
# 	$amplicon_name =~ s/\//+/g;
	# replaces all pipes with exclamation points
# 	$amplicon_name =~ s/\|/!/g;
	
	my ($amplicon_start, $amplicon_stop);
	if ($polarity eq '+')
	{
		$amplicon_start = $frag_start + $flap_index;
		$amplicon_stop = $frag_stop;
	}
	elsif ($polarity eq '-')
	{
		$amplicon_start = $frag_start;
		$amplicon_stop = $frag_stop - $flap_index;
	}
	else
	{
		$logger->error("bad fragment description: $target_desc\n");
	}
	print $out_fh join("\t",($amplicon_name, $g_accession, $amplicon_start, $amplicon_stop, $polarity));
	print $out_fh "\n";
}

sub usage
{
	print "Bad input.  Usage:\n";
	print "\tperl amplicon_generator.pl -i [fragment (PieceMaker output) input file] -o [amplicon output file]\n";
	exit;
}