/*
 *  bayreg/gdl_bayesian_regression.h 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#ifndef __GDL_BAYESIAN_REGRESSION_H__
#define __GDL_BAYESIAN_REGRESSION_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>

__BEGIN_DECLS

/**
 * Here some structure to handle the prior of effect sizes 
 */
typedef struct
{
	double sigmaa;
	double sigmad;
} gdl_bayreg_grid_point;

gdl_bayreg_grid_point * gdl_bayreg_grid_point_clone (const gdl_bayreg_grid_point * p);
gdl_bayreg_grid_point * gdl_bayreg_grid_point_fread (FILE * stream);
int gdl_bayreg_grid_point_fwrite (FILE * stream, const gdl_bayreg_grid_point * p);

typedef struct
{
	size_t size;
	gdl_bayreg_grid_point ** points;
	double * weight;
} gdl_bayreg_grid;

gdl_bayreg_grid * gdl_bayreg_grid_alloc (void);
void gdl_bayreg_grid_free (gdl_bayreg_grid * g);
gdl_bayreg_grid * gdl_bayreg_grid_clone (const gdl_bayreg_grid * g);
gdl_bayreg_grid * gdl_bayreg_grid_fread (FILE * stream);
int gdl_bayreg_grid_fwrite (FILE * stream, const gdl_bayreg_grid * g);
void gdl_bayreg_grid_add (gdl_bayreg_grid * g, const gdl_bayreg_grid_point * point, const double weight);

/**
 * Here some structure to handle the SNP models
 */

typedef struct
{
	double w;
	gdl_bayreg_grid * g;
} gdl_bayreg_additive_model;

gdl_bayreg_additive_model * gdl_bayreg_additive_model_alloc (const double w, gdl_bayreg_grid * g);
void gdl_bayreg_additive_model_free (gdl_bayreg_additive_model * m);
gdl_bayreg_additive_model * gdl_bayreg_additive_model_fread (FILE * stream);
gdl_bayreg_additive_model * gdl_bayreg_additive_model_fscanf (FILE * stream);
int gdl_bayreg_additive_model_fwrite (FILE * stream, const gdl_bayreg_additive_model * m);

typedef struct
{
	double w;
	double c;
	gdl_bayreg_grid * g;
} gdl_bayreg_adddom_model;

gdl_bayreg_adddom_model * gdl_bayreg_adddom_model_alloc (const double w, const double c, gdl_bayreg_grid * g);
void gdl_bayreg_adddom_model_free (gdl_bayreg_adddom_model * m);
gdl_bayreg_adddom_model * gdl_bayreg_adddom_model_fread (FILE * stream);
gdl_bayreg_adddom_model * gdl_bayreg_adddom_model_fscanf (FILE * stream);
int gdl_bayreg_adddom_model_fwrite (FILE * stream, const gdl_bayreg_adddom_model * m);

typedef struct
{
	double w;
	double pres;
	double pdom;
	gdl_bayreg_grid * g;
} gdl_bayreg_resdom_model;

gdl_bayreg_resdom_model * gdl_bayreg_resdom_model_alloc (const double w, const double q, gdl_bayreg_grid * g);
void gdl_bayreg_resdom_model_free (gdl_bayreg_resdom_model * m);
gdl_bayreg_resdom_model * gdl_bayreg_resdom_model_fread (FILE * stream);
gdl_bayreg_resdom_model * gdl_bayreg_resdom_model_fscanf (FILE * stream);
int gdl_bayreg_resdom_model_fwrite (FILE * stream, const gdl_bayreg_resdom_model * m);

typedef struct
{
	size_t size;
	gdl_bayreg_additive_model * puradd;
	gdl_bayreg_adddom_model   * adddom;
	gdl_bayreg_resdom_model   * resdom;
} gdl_bayreg_model;

gdl_bayreg_model * gdl_bayreg_model_alloc (void);
void gdl_bayreg_model_free (gdl_bayreg_model * m);
int gdl_bayreg_model_fwrite (FILE * stream, const gdl_bayreg_model * m);
int gdl_bayreg_model_fprintf (FILE * stream, const gdl_bayreg_model * m);
gdl_bayreg_model * gdl_bayreg_model_fread (FILE * stream);
gdl_bayreg_model * gdl_bayreg_model_fscanf (FILE * stream);
gdl_bayreg_model * gdl_bayreg_model_clone (const gdl_bayreg_model * m);

/**
 * now the workspace to compute the bayes factor * 
 */
typedef struct
{
	size_t * pidx;
	size_t np;
	size_t N;
	size_t P;
	double det;
	gdl_matrix * invOmega;
	gdl_matrix * X;	
	gdl_matrix * Y;
	gdl_matrix * Q;
	gdl_matrix * QSI;
	gdl_vector * S;
	gdl_vector * xt;
	gdl_vector * b;
	gdl_vector * y;
	gdl_vector * yy;
	gdl_vector * e;
	gdl_vector * D;
	// for multiple fit only
	gdl_matrix * Xm;    // the design matrix (last 2 columns = add + dom terms of the candidate SNP)
	gdl_vector * sigm;  // the prior for each column of X
} gdl_bayreg_workspace;

gdl_bayreg_workspace * gdl_bayreg_workspace_alloc (const size_t N, const size_t P);
gdl_bayreg_workspace * gdl_bayreg_workspace_forward_alloc (const size_t N, const size_t P);
void gdl_bayreg_workspace_free (gdl_bayreg_workspace * b);
double gdl_bayreg_single_adddom_bf (gdl_bayreg_workspace * bayes, const double sigmaa, const double sigmad);
double gdl_bayreg_single_puradd_bf (gdl_bayreg_workspace * bayes, const double sigmaa);
double gdl_bayreg_multiple_adddom_bf (gdl_bayreg_workspace * bayes);
double gdl_bayreg_multiple_puradd_bf (gdl_bayreg_workspace * bayes);

size_t gdl_bayreg_model_bf_storage_size (gdl_bayreg_model * m); 
double * gdl_bayreg_model_bf_storage (gdl_bayreg_model * m);
int gdl_bayreg_model_single_bf (const gdl_bayreg_model * m, gdl_bayreg_workspace * w, double bf[]);
int gdl_bayreg_model_forward_bf (const gdl_bayreg_model * m, gdl_bayreg_workspace * w, double bf[]);   
double gdl_bayreg_model_average_bf (const gdl_bayreg_model * m, const double bf[], const gdl_boolean rm_negative);
gdl_vector * gdl_bayreg_model_single_residual (const gdl_bayreg_model * m, gdl_bayreg_workspace * work);
gdl_vector * gdl_bayreg_model_push_snp (const gdl_bayreg_model * model, const size_t which_model, const size_t which_prior, const int G[], const size_t n, gdl_matrix ** Xm, gdl_vector * sigm);

typedef struct
{
	size_t n;
	size_t * m;
	size_t * w;
	double * bf;
} gdl_bayreg_bf_zip;

void gdl_bayreg_bf_zip_free (gdl_bayreg_bf_zip * z);
gdl_bayreg_bf_zip * gdl_bayreg_bf_zip_fread (FILE * stream);
int gdl_bayreg_bf_zip_fwrite (FILE * stream, const gdl_bayreg_bf_zip * z);

gdl_bayreg_bf_zip * gdl_bayreg_model_zip_bf (const gdl_bayreg_model * m, const double bf[]);
double gdl_bayreg_model_average_zip_bf (const gdl_bayreg_model * m, const gdl_bayreg_bf_zip * zip, const double w0, const double w1, const double w2, const double w3, const double wt);
double gdl_bayreg_model_best_zip_bf (const gdl_bayreg_model * m, const gdl_bayreg_bf_zip * zip, size_t * which, const double w0, const double w1, const double w2, const double w3, const double wt);

typedef struct
{
	size_t nvalue;
	size_t size;
	gdl_string ** names;
	size_t * nt;
	double *** values;
} gdl_bayreg_model_theta;

gdl_bayreg_model_theta * gdl_bayreg_model_parameters (const gdl_bayreg_model * m);

__END_DECLS

#endif

