/*
 *  bayreg/model.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h>
#include <stdio.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_machine.h>
#include <gdl/gdl_bayesian_regression.h>

gdl_bayreg_model *
gdl_bayreg_model_alloc (void)
{
	gdl_bayreg_model * m;
	
	m = GDL_CALLOC (gdl_bayreg_model, 1);
	
	return m;	
}

void
gdl_bayreg_model_free (gdl_bayreg_model * m)
{
	if (m)
	{
		gdl_bayreg_additive_model_free (m->puradd);
		gdl_bayreg_adddom_model_free (m->adddom);
		gdl_bayreg_resdom_model_free (m->resdom);
	}	
}

int
gdl_bayreg_model_fwrite (FILE * stream, const gdl_bayreg_model * m)
{
	if (stream && m)
	{
		int status;
		unsigned char has;
		
		has = (m->puradd) ? 'y' : 'n';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (m->puradd)
		{
			status = gdl_bayreg_additive_model_fwrite (stream, m->puradd);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		has = (m->adddom) ? 'y' : 'n';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (m->adddom)
		{
			status = gdl_bayreg_adddom_model_fwrite (stream, m->adddom);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);			
		}
		has = (m->resdom) ? 'y' : 'n';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (m->resdom)
		{
			status = gdl_bayreg_resdom_model_fwrite (stream, m->resdom);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

gdl_bayreg_model *
gdl_bayreg_model_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		unsigned char has;
		gdl_bayreg_model * m = gdl_bayreg_model_alloc ();
		
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='y')
		{
			m->puradd = gdl_bayreg_additive_model_fread (stream);
			GDL_FREAD_STATUS (m->puradd!=0, 1);
		}
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='y')
		{
			m->adddom = gdl_bayreg_adddom_model_fread (stream);
			GDL_FWRITE_STATUS (m->adddom!=0, 1);
		}
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has=='y')
		{
			m->resdom = gdl_bayreg_resdom_model_fread (stream);
			GDL_FWRITE_STATUS (m->resdom!=0, 1);
		}		
		
		gdl_bayreg_model_bf_storage_size (m);
		
		return m;
	}
	return 0;
}

gdl_bayreg_model *
gdl_bayreg_model_fscanf (FILE * stream)
{
	gdl_bayreg_model * m = gdl_bayreg_model_alloc ();
	m->puradd = gdl_bayreg_additive_model_fscanf (stream);
	m->adddom = gdl_bayreg_adddom_model_fscanf (stream);
	m->resdom = gdl_bayreg_resdom_model_fscanf (stream);
	gdl_bayreg_model_bf_storage_size (m);
	return m;
}

int
gdl_bayreg_model_fprintf (FILE * stream, const gdl_bayreg_model * m)
{
	size_t p;
	
	if (m->puradd)
	{
		const gdl_bayreg_grid * g = m->puradd->g;
		fprintf(stream, "[ Additive ] %g\n", m->puradd->w);
		for(p = 0; p < g->size; p++)
			fprintf(stream, "   %d %g %g\n", p, g->weight[p], g->points[p]->sigmaa);
	}
	if (m->adddom)
	{
		const gdl_bayreg_grid * g = m->adddom->g;
		fprintf(stream, "[ Additive+Dominance ] %g\n", m->adddom->w);
		for(p = 0; p < g->size; p++)
			fprintf(stream, "   %d %g %g %g\n", p, g->weight[p], g->points[p]->sigmaa, g->points[p]->sigmad);	
	}
	if (m->resdom)
	{
		const gdl_bayreg_grid * g = m->resdom->g;
		fprintf(stream, "[ Recessive/Dominance ] %g %g %g\n", m->resdom->w, m->resdom->pres, m->resdom->pdom);
		for(p = 0; p < g->size; p++)
			fprintf(stream, "   %d %g %g\n", p, g->weight[p], g->points[p]->sigmaa);		
	}		
}

gdl_bayreg_model *
gdl_bayreg_model_clone (const gdl_bayreg_model * m)
{
	if (m)
	{
		gdl_bayreg_model * c;
		
		c = gdl_bayreg_model_alloc ();
		
		if (m->adddom)
		{
			c->adddom = gdl_bayreg_adddom_model_alloc (m->adddom->w, m->adddom->c, gdl_bayreg_grid_clone (m->adddom->g)); 	
		}
		if (m->puradd)
		{
			c->puradd = gdl_bayreg_additive_model_alloc (m->puradd->w, gdl_bayreg_grid_clone (m->puradd->g));
		}
		if (m->resdom)
		{
			double q = m->resdom->pres/(m->resdom->pres+m->resdom->pdom);
			c->resdom = gdl_bayreg_resdom_model_alloc (m->resdom->w, q, gdl_bayreg_grid_clone (m->resdom->g));
		}
		
		gdl_bayreg_model_bf_storage_size (c);
		
		return c;
	}
	return 0;	
}

size_t
gdl_bayreg_model_bf_storage_size (gdl_bayreg_model * m)
{
	if (!m->size)
	{
		if (m->puradd)
			m->size += m->puradd->g->size;
		if (m->adddom)
			m->size += m->adddom->g->size;
		if (m->resdom)
			m->size += 2*m->resdom->g->size;
		if (!m->size)
		{
			GDL_ERROR_VAL ("The bayesian regression model is of size 0", GDL_EINVAL, 0);
		}
	}
	return m->size;
}

double *
gdl_bayreg_model_bf_storage (gdl_bayreg_model * m)
{
	if (!m->size)
	{
		if (m->puradd)
			m->size += m->puradd->g->size;
		if (m->adddom)
			m->size += m->adddom->g->size;
		if (m->resdom)
			m->size += 2*m->resdom->g->size;
		if (!m->size)
		{
			GDL_ERROR_VAL ("The bayesian regression model is of size 0", GDL_EINVAL, 0);
		}
	}
	return GDL_CALLOC (double, m->size);
}

#define MATRIX_SET(X,i,j,y)(*(X->data+(i*X->tda+j))=y)
#define MATRIX_GET(X,i,j)(*(X->data+(i*X->tda+j)))

static gdl_boolean
can_fit_adddom (const gdl_matrix * X)
{
	size_t i,n = 0;
	for(i = 0; i < X->size1; i++)	
	{
		n += (MATRIX_GET (X, i, 2)!=0.0);
	}
	return n;
}

static size_t
move_matrix_to_recessive_model (gdl_matrix * X, const size_t offset)
{
	size_t i,n = 0;
	for(i = 0; i < X->size1; i++)	
	{
		if (MATRIX_GET (X, i, offset+2)!=0.0)
			MATRIX_SET(X, i, offset+1, 0.0);
		n += (MATRIX_GET (X, i, offset+1) == 2.0);
	}
	return n;
}

static size_t
move_matrix_to_dominant_model (gdl_matrix * X, const size_t offset)
{
	size_t i, n = 0;
	for(i = 0; i < X->size1; i++)
	{	
		if (MATRIX_GET (X, i, offset+2)!=0.0)
			MATRIX_SET(X,i,offset+1,2);
		n += (MATRIX_GET (X, i, offset+1) == 0.0);
	}
	return n;
}

static void
restore_matrix (gdl_matrix * X, const size_t offset)
{
	size_t i;
	for(i = 0; i < X->size1; i++)
	{	
		if (MATRIX_GET (X, i, offset+2)!=0.0)
			MATRIX_SET(X, i, offset+1, 1.0);
	}
}

#undef MATRIX_SET
#undef MATRIX_GET

int
gdl_bayreg_model_single_bf (const gdl_bayreg_model * m,
                            gdl_bayreg_workspace * w, double bf[])
{
	size_t p,i=0;
	
	if (m->puradd)
	{
		const gdl_bayreg_grid * g = m->puradd->g;
		for(p = 0; p < g->size; p++, i++)
		{
			bf[i] = gdl_bayreg_single_puradd_bf (w, g->points[p]->sigmaa);
			//if (bf[i] > 1000) printf("A %d %g\n", i, bf[i]);
		}
	}
	if (m->adddom)
	{
		const gdl_bayreg_grid * g = m->adddom->g;
		gdl_boolean ok = can_fit_adddom (w->X);
		if (ok || (!m->puradd && !ok))
		{
			for(p = 0; p < g->size; p++, i++)
			{
				bf[i] = gdl_bayreg_single_adddom_bf (w, g->points[p]->sigmaa, g->points[p]->sigmad);
				//printf("AD %d %g\n", i, bf[i]);
			}
		}
		else
		{
			for(p = 0; p < g->size; p++, i++)
			{
				bf[i] = GDL_NEGINF;
			}	
		}
	}
	if (m->resdom)
	{
		const gdl_bayreg_grid * g = m->resdom->g;
		size_t can_fit = move_matrix_to_recessive_model (w->X, 0);
		if (can_fit)
		{
			for(p = 0; p < g->size; p++, i++)
			{
				bf[i] = gdl_bayreg_single_puradd_bf (w, g->points[p]->sigmaa);
				//printf("R>D %d %g\n", i, bf[i]);
			}
			move_matrix_to_dominant_model (w->X, 0);
			for(p = 0; p < g->size; p++, i++)
			{
				bf[i] = gdl_bayreg_single_puradd_bf (w, g->points[p]->sigmaa);
				//printf("R<D %d %g\n", i, bf[i]);
			}			
		}
		else
		{
			for(p = 0; p < 2*g->size; p++, i++)
			{
				bf[i] = GDL_NEGINF;
			}	
		}
		restore_matrix (w->X, 0);
	}
	return GDL_SUCCESS;
}

gdl_vector *
gdl_bayreg_model_push_snp (const gdl_bayreg_model * model,
                           const size_t which_model,
                           const size_t which_prior,
                           const int G[],
                           const size_t n,
                           gdl_matrix ** Xm,
                           gdl_vector * sigm)
{
	size_t i,m;
	double sigmaa, sigmad;
	gdl_matrix * xtmp;
	
	m = (which_model==1) ? 2 : 1;
	if (sigm==0)
	{
		sigm = gdl_vector_alloc (m);
	}
	else
	{
		gdl_vector * vtmp = gdl_vector_alloc (sigm->size + m);
		vtmp->size = sigm->size;
		gdl_vector_memcpy (vtmp, sigm);
		vtmp->size = sigm->size + m;
		gdl_vector_free (sigm);
		sigm = vtmp;
	}
	xtmp = gdl_matrix_calloc ((*Xm)->size1, (*Xm)->size2 + m);
	xtmp->size2 = (*Xm)->size2;
	gdl_matrix_memcpy (xtmp, (*Xm));
	xtmp->size2 = (*Xm)->size2 + m;
	gdl_matrix_free ((*Xm));
	*Xm = xtmp;
	
	for(i = 0; i < n; i++)
	{
		switch(G[i])
		{
			case 0:
			case 1:
			case 2:
				gdl_matrix_set ((*Xm), i, (*Xm)->size2 - m, G[i]);
				break;
			default:
				gdl_matrix_set ((*Xm), i, (*Xm)->size2 - m, 0.0);
				break;
		}
		if (m == 2 && G[i] == 1)
		{
			gdl_matrix_set ((*Xm), i, (*Xm)->size2 - 1, 1.0);
		}	
	}
	
	switch (which_model)
	{
		case 0: //additive
			sigmaa = model->puradd->g->points[which_prior]->sigmaa;
			gdl_vector_set (sigm, m-1, sigmaa);
			break;
		case 1: // additive + dominance
			sigmaa = model->adddom->g->points[which_prior]->sigmaa;
			gdl_vector_set (sigm, m-2, sigmaa);
			sigmad = model->adddom->g->points[which_prior]->sigmad;
			gdl_vector_set (sigm, m-1, sigmad);
			break;
		case 2: // recessive
		 	sigmaa = model->resdom->g->points[which_prior]->sigmaa;
		 	gdl_vector_set (sigm, m-1, sigmaa);
		 	// set heterozygotes to 0
		 	{
		 	for(i = 0; i < n; i++)
				if (G[i]==1)
					gdl_matrix_set ((*Xm), i, (*Xm)->size2 - 1, 0.0);
		 	} 
			break;
		case 3:	// dominant
			sigmaa = model->resdom->g->points[which_prior]->sigmaa;
			gdl_vector_set (sigm, m-1, sigmaa);
			// set heterozygotes to 1
		 	{
		 	for(i = 0; i < n; i++)
				if (G[i]==1)
					gdl_matrix_set ((*Xm), i, (*Xm)->size2 - 1, 2.0);
		 	} 
			break;
	}
	
	return sigm;	
}                           
/**
 * Here, we evaluate the strenght of any additional SNP
 * with respect to the current model
 * 
 * The model plus the tested SNP design matrix is stored
 * into m->Xm and prior on effect sizes are stored into
 * m->sigm
 * 
 */
int
gdl_bayreg_model_forward_bf (const gdl_bayreg_model * m, gdl_bayreg_workspace * w, double bf[])
{
	size_t p,i=0;
	
	if (m->puradd)
	{
		const gdl_bayreg_grid * g = m->puradd->g;
		for(p = 0; p < g->size; p++, i++)
		{
			gdl_vector_set (w->sigm, w->sigm->size-2, g->points[p]->sigmaa);
			bf[i] = gdl_bayreg_multiple_puradd_bf (w);
		}
	}
	if (m->adddom)
	{
		const gdl_bayreg_grid * g = m->adddom->g;
		gdl_boolean ok = can_fit_adddom (w->X);
		if (ok || (!m->puradd && !ok))
		{
			for(p = 0; p < g->size; p++, i++)
			{
				gdl_vector_set (w->sigm, w->sigm->size-2, g->points[p]->sigmaa);
				gdl_vector_set (w->sigm, w->sigm->size-1, g->points[p]->sigmad);
				bf[i] = gdl_bayreg_multiple_adddom_bf (w);
				//printf("AD %d %g\n", i, bf[i]);
			}
		}
		else
		{
			for(p = 0; p < g->size; p++, i++)
			{
				bf[i] = GDL_NEGINF;
			}	
		}
	}
	if (m->resdom)
	{
		const gdl_bayreg_grid * g = m->resdom->g;
		size_t can_fit = move_matrix_to_recessive_model (w->Xm, w->Xm->size2-3);
		if (can_fit)
		{
			for(p = 0; p < g->size; p++, i++)
			{
				gdl_vector_set (w->sigm, w->sigm->size-2, g->points[p]->sigmaa);
				bf[i] = gdl_bayreg_multiple_puradd_bf (w);
				//printf("R>D %d %g\n", i, bf[i]);
			}
			move_matrix_to_dominant_model (w->Xm, w->Xm->size2-3);
			for(p = 0; p < g->size; p++, i++)
			{
				gdl_vector_set (w->sigm, w->sigm->size-2, g->points[p]->sigmaa);
				bf[i] = gdl_bayreg_multiple_puradd_bf (w);
				//printf("R<D %d %g\n", i, bf[i]);
			}			
		}
		else
		{
			for(p = 0; p < 2*g->size; p++, i++)
			{
				bf[i] = GDL_NEGINF;
			}	
		}
		restore_matrix (w->Xm, w->Xm->size2-3);
	}
	return GDL_SUCCESS;
}

gdl_vector * 
gdl_bayreg_model_single_residual (const gdl_bayreg_model * m,
                                  gdl_bayreg_workspace * work)
{
	const size_t N = work->X->size1;
	size_t p,which_prior=0,which_model=0;
	double max=0, bf, wm, wp, wt;
	gdl_vector * e = gdl_vector_calloc (N);
	
	/**
	 *  STEP 1 : determine the best model
	 */
	wt=0;
	if (m->puradd) wt += m->puradd->w;
	if (m->adddom) wt += m->adddom->w;
	if (m->resdom) wt += m->resdom->w;
	
	if (m->puradd)
	{
		const double wm = (m->puradd->w)/wt;
		const gdl_bayreg_grid * g = m->puradd->g;
		for(wp = 0, p = 0; p < g->size; p++)
			wp += g->weight[p];
		for(p = 0; p < g->size; p++)
		{
			bf  = gdl_bayreg_single_puradd_bf (work, g->points[p]->sigmaa);
			if (wm*(g->weight[p]/wp)*bf	> max)
			{
				max  = wm*(g->weight[p]/wp)*bf;
				which_model = 0;
				which_prior = p; 
				//printf ("MAX %d %d %g %g %g (%g)\n", which_model, which_prior, wm, g->weight[p]/wp, bf, max);
			}
		}
	}
	if (m->adddom)
	{
		const double wm = (m->adddom->w)/wt; 
		const gdl_bayreg_grid * g = m->adddom->g;
		for(wp = 0, p = 0; p < g->size; p++)
			wp += g->weight[p];
		for(p = 0; p < g->size; p++)
		{
			bf = gdl_bayreg_single_adddom_bf (work, g->points[p]->sigmaa, g->points[p]->sigmad);
			if (wm*(g->weight[p]/wp)*bf	> max)
			{
				max  = wm*(g->weight[p]/wp)*bf;
				which_model = 1;
				which_prior = p;
				//printf ("MAX %d %d %g %g %g (%g)\n", which_model, which_prior, wm, g->weight[p]/wp, bf, max); 
			}	
		}
	}
	if (m->resdom)
	{
		size_t can_fit = move_matrix_to_recessive_model (work->X, 0);
		if (can_fit)
		{
			const double wm   = (m->resdom->w)/wt;
			const double q    = (m->resdom->pres)/(m->resdom->pres+m->resdom->pdom);
			const gdl_bayreg_grid * g = m->resdom->g;
			for(wp = 0, p = 0; p < g->size; p++)
				wp  += g->weight[p];
			for(p = 0; p < g->size; p++)
			{	
				bf = gdl_bayreg_single_adddom_bf (work, g->points[p]->sigmaa, g->points[p]->sigmad);
				if (wm*q*(g->weight[p]/wp)*bf	> max)
				{
					max  = wm*(g->weight[p]/wp)*bf;
					which_model = 2;
					which_prior = p; 
					//printf ("MAX %d %d %g %g %g (%g)\n", which_model, which_prior, wm, g->weight[p]/wp, bf, max);
				}		
			}
			move_matrix_to_dominant_model (work->X, 0);
			for(p = 0; p < g->size; p++)
			{	
				bf = gdl_bayreg_single_adddom_bf (work, g->points[p]->sigmaa, g->points[p]->sigmad);
				if (wm*(1.0-q)*(g->weight[p]/wp)*bf	> max)
				{
					max  = wm*(g->weight[p]/wp)*bf;
					which_model = 3;
					which_prior = p; 
					//printf ("MAX %d %d %g %g %g (%g)\n", which_model, which_prior, wm, g->weight[p]/wp, bf, max);
				}		
			}
		}
		restore_matrix (work->X, 0);
	}
	//printf ("SIZE = %d %d\n", e->size, work->e->size);
	/**
	 * STEP 2 : Compute the residuals for the best model
	 */ 
	switch(which_model)
	{
		case 0:
			bf = gdl_bayreg_single_puradd_bf (work, m->puradd->g->points[which_prior]->sigmaa);
			gdl_vector_memcpy (e, work->e);
			break;
		case 1:
			bf = gdl_bayreg_single_adddom_bf (work, m->adddom->g->points[which_prior]->sigmaa, m->adddom->g->points[which_prior]->sigmad);
			gdl_vector_memcpy (e, work->e);
			break;
		case 2:
			move_matrix_to_recessive_model (work->X, 0);
			bf = gdl_bayreg_single_puradd_bf (work, m->resdom->g->points[which_prior]->sigmaa);
			gdl_vector_memcpy (e, work->e);
			restore_matrix (work->X, 0);
			break;
		case 3:
			move_matrix_to_dominant_model (work->X, 0);
			bf = gdl_bayreg_single_puradd_bf (work, m->resdom->g->points[which_prior]->sigmaa);
			gdl_vector_memcpy (e, work->e);
			restore_matrix (work->X, 0);
			break;
		
	}
	wm = wp = 0;
	for(p = 0; p < N; p++)
	{
		wt = gdl_vector_get (e, p);
		wm += wt*wt;
		wt = gdl_vector_get (work->y, p);
		wp += wt*wt;	
	}
	//printf ("R-SQUARE = 1 - %g / %g = %g (N = %d)\n", wm, wp, 1.0-wm/wp, N);
	/**
	 * Deprecated : weighted average residuals 
	 */
//	if (m->puradd)
//	{
//		const double w = m->puradd->w;
//		const gdl_bayreg_grid * g = m->puradd->g;
//		gdl_vector * et = gdl_vector_calloc (N);
//		for(wm = 0, p = 0; p < g->size; p++)
//		{
//			bf  = gdl_bayreg_single_puradd_bf (work, g->points[p]->sigmaa);
//			wm += bf*g->weight[p];
//			for(i = 0; i < N; i++)
//			{
//				//fprintf (stderr, "ADD e %d w=%g %g %g)\n", i, bf*g->weight[p], gdl_vector_get(work->e, i), gdl_vector_get(work->y, i));
//				gdl_vector_set (et, i, gdl_vector_get(et, i) + bf*g->weight[p]*gdl_vector_get(work->e, i));
//			}
//		}
//		for(i = 0; i < N; i++)
//		{
//			gdl_vector_set (e, i, (w)*(gdl_vector_get(et, i)/wm));
//		}
//		gdl_vector_free (et);
//		wt += w;
//	}
//	if (m->adddom)
//	{
//		gdl_boolean ok = can_fit_adddom (work->X);
//		if (ok || (!m->puradd && !ok))
//		{
//			const gdl_bayreg_grid * g = m->adddom->g;
//			const double w = m->adddom->w; 
//			gdl_vector * et = gdl_vector_calloc (N);
//			for(wm = 0, p = 0; p < g->size; p++)
//			{
//				bf = gdl_bayreg_single_adddom_bf (work, g->points[p]->sigmaa, g->points[p]->sigmad);
//				wm += bf*g->weight[p];
//				for(i = 0; i < N; i++)
//				{
//					//fprintf (stderr, "ADDOM e %d w=%g %g (%g)\n", i, bf*g->weight[p], gdl_vector_get(work->e, i), gdl_vector_get(work->y, i));
//					gdl_vector_set (et, i, gdl_vector_get(et, i) + bf*g->weight[p]*gdl_vector_get(work->e, i));
//				}
//			}
//			for(i = 0; i < N; i++)
//			{
//				gdl_vector_set (e, i, gdl_vector_get(e, i)+(w)*(gdl_vector_get(et, i)/wm));
//			}
//			wt  += w;
//			gdl_vector_free (et);
//		}
//	}
//	if (m->resdom)
//	{
//		const gdl_bayreg_grid * g = m->resdom->g;
//		size_t can_fit = move_matrix_to_recessive_model (work->X, 0);
//		if (can_fit)
//		{
//			const double w    = m->resdom->w;
//			const double pres = m->resdom->pres;
//			const double pdom = m->resdom->pdom;
//			const gdl_bayreg_grid * g = m->resdom->g;
//			gdl_vector * et = gdl_vector_calloc (N);
//			for(wm = 0, p = 0; p < g->size; p++)
//			{
//				bf   = gdl_bayreg_single_puradd_bf (work, g->points[p]->sigmaa);
//				wm  += g->weight[p]*bf;
//				for(i = 0; i < N; i++)
//				{
//					//fprintf (stderr, "RES e %d w=%g %g (%g)\n", i, bf*g->weight[p], gdl_vector_get(work->e, i), gdl_vector_get(work->y, i));
//					gdl_vector_set (et, i, gdl_vector_get(et, i) + bf*g->weight[p]*gdl_vector_get(work->e, i));
//				}
//			}
//			for(i = 0; i < N; i++)
//			{
//				gdl_vector_set (e, i, gdl_vector_get(e, i)+(w*pres)*(gdl_vector_get(et, i)/(wm*(pres+pdom))));
//			}
//			gdl_vector_free (et);
//			move_matrix_to_dominant_model (work->X, 0);
//			et = gdl_vector_calloc (N);
//			for(wm = 0, p = 0; p < g->size; p++)
//			{
//				bf   = gdl_bayreg_single_puradd_bf (work, g->points[p]->sigmaa);
//				wm  += g->weight[p]*bf;
//				for(i = 0; i < N; i++)
//				{
//					//fprintf (stderr, "DOM e %d w=%g %g (%g)\n", i, bf*g->weight[p], gdl_vector_get(work->e, i), gdl_vector_get(work->y, i));
//					gdl_vector_set (et, i, gdl_vector_get(et, i) + bf*g->weight[p]*gdl_vector_get(work->e, i));
//				}			
//			}
//			for(i = 0; i < N; i++)
//			{
//				gdl_vector_set (e, i, gdl_vector_get(e, i)+(w*pres)*(gdl_vector_get(et, i)/(wm*(pres+pdom))));
//			}
//			gdl_vector_free (et);
//			wt  += w;
//		}
//		restore_matrix (work->X, 0);
//	}
//	if (wt > 0)
//	{
//		for(i = 0; i < N; i++)
//		{
//			gdl_vector_set (e, i, gdl_vector_get(e, i)/wt);
//			//fprintf (stderr, "e %d %g %g\n", i, gdl_vector_get(e, i), gdl_vector_get(work->y, i));
//		}
//	}
	return e;
}


double
gdl_bayreg_model_average_bf (const gdl_bayreg_model * m, const double bf[], const gdl_boolean rm_negative)
{
	size_t p,i=0;
	double x, tbf = 0, bfm, wm, wt=0;
	
	if (m->puradd)
	{
		const double w = m->puradd->w;
		const gdl_bayreg_grid * g = m->puradd->g;
		for(bfm = wm = 0, p = 0; p < g->size; p++, i++)
		{
			wm  += g->weight[p];
			if (!rm_negative || (rm_negative && bf[i] > 1.0))
			{
				bfm += g->weight[p]*bf[i];
			}			
		}
		bfm /= wm;
		tbf  += w*bfm;
		wt   += w;
	}
	if (m->adddom)
	{
		const gdl_bayreg_grid * g = m->adddom->g;
		if (bf[i] != GDL_NEGINF)
		{
			const double w = m->adddom->w; 
			for(bfm = wm = 0, p = 0; p < g->size; p++, i++)
			{
				wm  += g->weight[p];
				if (!rm_negative || (rm_negative && bf[i] > 1.0))
				{
					bfm += g->weight[p]*bf[i];
				}			
			}
			bfm /= wm;
			tbf  += w*bfm;
			wt  += w;
		}
		else
		{
			i += g->size;
		}
	}
	if (m->resdom)
	{
		if (bf[i] != GDL_NEGINF)
		{
			const double w    = m->resdom->w;
			const double pres = m->resdom->pres;
			const double pdom = m->resdom->pdom;
			const gdl_bayreg_grid * g = m->resdom->g;
			double bff = 0;
			for(bfm = wm = 0, p = 0; p < g->size; p++, i++)
			{
				wm  += g->weight[p];
				if (!rm_negative || (rm_negative && bf[i] > 1.0))
				{
					bfm += g->weight[p]*bf[i];
				}			
			}
			bfm /= wm;
			bff += pres*bfm;
			for(bfm = 0, p = 0; p < g->size; p++, i++)
			{
				if (!rm_negative || (rm_negative && bf[i] > 1.0))
				{
					bfm += g->weight[p]*bf[i];
				}			
			}
			bfm /= wm;
			bff += pdom*bfm;
			bff /= (pres+pdom);
			tbf  += w*bff;
			wt  += w;
		}
	}
	return (wt) ? tbf/wt : GDL_DBL_EPSILON;
}

gdl_bayreg_bf_zip *
gdl_bayreg_model_zip_bf (const gdl_bayreg_model * m, const double bf[])
{
	size_t p,i=0,j=0;
	double x, tbf = 0, bfm, wm, wt=0;
	gdl_bayreg_bf_zip * z;
	
	z = GDL_CALLOC (gdl_bayreg_bf_zip, 1);
	
	if (m->puradd)
	{
		const gdl_bayreg_grid * g = m->puradd->g;
		for(p = 0; p < g->size; p++, i++)
		{
			if(bf[i] > 1.0)
				(z->n)++;		
		}
	}
	if (m->adddom)
	{
		const gdl_bayreg_grid * g = m->adddom->g;
		for(p = 0; p < g->size; p++, i++)
		{
			if (bf[i] > 1.0)
				(z->n)++;			
		}
	}
	if (m->resdom)
	{
		const gdl_bayreg_grid * g = m->resdom->g;
		for (p = 0; p < g->size; p++, i++)
		{
			if (bf[i] > 1.0)
				(z->n)++;
		}
		for(p = 0; p < g->size; p++, i++)
		{
			if (bf[i] > 1.0)
				(z->n)++;
		}
	}
	
	if (z->n)
	{
		z->m  = GDL_CALLOC (size_t, z->n);
		z->w  = GDL_CALLOC (size_t, z->n);
		z->bf = GDL_CALLOC (double, z->n);
		i = j = 0;
		if (m->puradd)
		{
			const gdl_bayreg_grid * g = m->puradd->g;
			for(p = 0; p < g->size; p++, i++)
			{
				if(bf[i] > 1.0)
				{
					z->bf[j] = bf[i];
					z->m[j]  = 0;
					z->w[j]  = p;
					j++;
				}		
			}
		}
		if (m->adddom)
		{
			const gdl_bayreg_grid * g = m->adddom->g;
			for(p = 0; p < g->size; p++, i++)
			{
				if (bf[i] > 1.0)
				{
					z->bf[j] = bf[i];
					z->m[j]  = 1;
					z->w[j]  = p;
					j++;
				}			
			}
		}
		if (m->resdom)
		{
			const gdl_bayreg_grid * g = m->resdom->g;
			for (p = 0; p < g->size; p++, i++)
			{
				if (bf[i] > 1.0)
				{
					z->bf[j] = bf[i];
					z->m[j]  = 2;
					z->w[j]  = p;
					j++;
				}
			}
			for(p = 0; p < g->size; p++, i++)
			{
				if (bf[i] > 1.0)
				{
					z->bf[j] = bf[i];
					z->m[j]  = 3;
					z->w[j]  = p;
					j++;
				}
			}
		}
	}
	
	return z;
}

void
gdl_bayreg_model_average_zip_w (const gdl_bayreg_model * m,
                                double *w0,
                                double *w1,
                                double *w2,
                                double *w3,
                                double *wt)
{
	size_t p;
	
	*w0 = *w1 = *w2 = *w3 = *wt = 0;
	
	if (m->puradd)
	{
		*wt += m->puradd->w;
		const gdl_bayreg_grid * g = m->puradd->g;
		for(p = 0; p < g->size; p++)
			*w0 += g->weight[p];
	}
	if (m->adddom)
	{
		*wt += m->adddom->w;
		const gdl_bayreg_grid * g = m->adddom->g;
		for(p = 0; p < g->size; p++)
			*w1 += g->weight[p];
	}
	if (m->resdom)
	{
		*wt += m->resdom->w;
		*w3 += m->resdom->pres;
		*w3 += m->resdom->pdom;
		const gdl_bayreg_grid * g = m->resdom->g;
		for(p = 0; p < g->size; p++)
			*w2 += g->weight[p];
	}
}                                

double
gdl_bayreg_model_average_zip_bf (const gdl_bayreg_model * m,
                                 const gdl_bayreg_bf_zip * zip,
                                 const double w0,
                                 const double w1,
                                 const double w2,
                                 const double w3,
                                 const double wt)
{
	size_t i;
	double bf0=0,bf1=0,bf2=0,bf3=0;
	
	for(i = 0; i < zip->n; i++)
	{
		switch(zip->m[i])
		{
			case 0:
				if (m->puradd)
				{
					bf0 += (m->puradd->g->weight[zip->w[i]]*zip->bf[i]*m->puradd->w)/(w0*wt);
				}
				break;
			case 1:
				if (m->adddom)
				{
					bf1 += (m->adddom->g->weight[zip->w[i]]*zip->bf[i]*m->adddom->w)/(w1*wt);
				}
				break;
			case 2:
				if (m->resdom)
				{
					bf2 += (m->resdom->g->weight[zip->w[i]]*zip->bf[i]*m->resdom->w*m->resdom->pres)/(w2*w3*wt);
				}
				break;
			case 3:
				if (m->resdom)
				{
					bf3 += (m->resdom->g->weight[zip->w[i]]*zip->bf[i]*m->resdom->w*m->resdom->pdom)/(w2*w3*wt);
				}
				break;
		}	
	}
	
	return (bf0+bf1+bf2+bf3);
}

gdl_bayreg_model_theta *
gdl_bayreg_model_parameters (const gdl_bayreg_model * m)
{
	size_t p,i=0,nm=0;
	gdl_bayreg_model_theta * t;
	
	t = GDL_CALLOC (gdl_bayreg_model_theta, 1);
	t->size = 1;
	
	if (m->puradd) {t->size += 1;nm++;}
	if (m->adddom)	{t->size += 1;nm++;}
	if (m->resdom)	{t->size += 2;nm++;}
	
	t->values = GDL_MALLOC (double **, t->size);
	t->nt     = GDL_MALLOC (size_t, t->size);
	t->names  = GDL_MALLOC (gdl_string *, t->size);
	
	t->values[0] = GDL_MALLOC (double *, nm);
	t->nt[0]     = nm;
	t->names[0]  = gdl_string_sprintf ("model type (weight)");
	t->nvalue    = nm;
	i  = 1;
	nm = 0;
	if (m->puradd)
	{
		const gdl_bayreg_grid * g = m->puradd->g;
		t->values[0][nm++] = &(m->puradd->w);
		t->values[i] = GDL_MALLOC (double *, g->size);
		t->nt[i]     = g->size;
		t->names[i]  = gdl_string_sprintf ("additive model (sigma)");
		for(p = 0; p < g->size; p++) 
		{
			t->nvalue++;
			t->values[i][p] =	&(g->weight[p]);
		}
		i++;
	}
	if (m->adddom)
	{
		const gdl_bayreg_grid * g = m->adddom->g;
		t->values[0][nm++] = &(m->adddom->w);
		t->values[i] = GDL_MALLOC (double *, g->size);
		t->nt[i]     = g->size;
		t->names[i]  = gdl_string_sprintf ("additive+dominance model (sigma)");
		for(p = 0; p < g->size; p++) 
		{
			t->nvalue++;
			t->values[i][p] =	&(g->weight[p]);
		}
		i++;	
	}
	if (m->resdom)
	{
		const gdl_bayreg_grid * g = m->resdom->g;
		t->values[0][nm++] = &(m->resdom->w);
		t->values[i] = GDL_MALLOC (double *, 2);
		t->nt[i]     = 2;
		t->names[i]  = gdl_string_sprintf ("recessive+dominance model (weight)");
		t->values[i][0] = &(m->resdom->pres);
		t->values[i][1] = &(m->resdom->pdom);
		t->nvalue+=2;
		i++;
		t->values[i] = GDL_MALLOC (double *, g->size);
		t->nt[i]     = g->size;
		t->names[i]  = gdl_string_sprintf ("recsessive+dominance model (sigma)");
		for(p = 0; p < g->size; p++) 
		{
			t->nvalue++;
			t->values[i][p] =	&(g->weight[p]);
		}
		i++;
	}
	return t;	
}

double
gdl_bayreg_model_best_zip_bf (const gdl_bayreg_model * m,
                              const gdl_bayreg_bf_zip * zip,
                              size_t * which,
                              const double w0,
                              const double w1,
                              const double w2,
                              const double w3,
                              const double wt)
{
	size_t i, imax = 0;
	double bfi, max = 0;
	
	for(i = 0; i < zip->n; i++)
	{
		switch(zip->m[i])
		{
			case 0:
				if (m->puradd)
				{
					 bfi = (m->puradd->g->weight[zip->w[i]]*zip->bf[i]*m->puradd->w)/(w0*wt);
					 if (bfi > max)
					 {
					 	max  = bfi;
					 	imax = i;
					 }
				}
				break;
			case 1:
				if (m->adddom)
				{
					bfi = (m->adddom->g->weight[zip->w[i]]*zip->bf[i]*m->adddom->w)/(w1*wt);
					if (bfi > max)
					{
					 	max  = bfi;
					 	imax = i;
					}
				}
				break;
			case 2:
				if (m->resdom)
				{
					bfi = (m->resdom->g->weight[zip->w[i]]*zip->bf[i]*m->resdom->w*m->resdom->pres)/(w2*w3*wt);
					if (bfi > max)
					{
					 	max  = bfi;
					 	imax = i;
					}
				}
				break;
			case 3:
				if (m->resdom)
				{
					bfi = (m->resdom->g->weight[zip->w[i]]*zip->bf[i]*m->resdom->w*m->resdom->pdom)/(w2*w3*wt);
					if (bfi > max)
					{
					 	max  = bfi;
					 	imax = i;
					}
				}
				break;
		}	
	}
	*which = imax;
	return max;
}
