/*  
 * 	sheet/gdl_bits.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_bit_vector.h>
#include <gdl/gdl_check_range.h>

gdl_bit_vector *
gdl_bit_vector_alloc (size_t n)
{
	gdl_bit_vector * g;
	
	g = GDL_MALLOC (gdl_bit_vector, 1);
	
	g->size = n;
	g->slot = GDL_BITNSLOTS(n);
	g->data = GDL_CALLOC (gdl_bit_slot, g->slot);
	
	return g;
}

gdl_bit_vector *
gdl_bit_vector_clone (const gdl_bit_vector * b)
{
	if (b)
	{
		gdl_bit_vector * c;
		
		c = gdl_bit_vector_alloc (b->size);
		
		memcpy (c->data, b->data, sizeof(gdl_bit_slot)*b->slot);
		
		return c;
	}
	return 0;
}

void
gdl_bit_vector_free (gdl_bit_vector * b)
{
	if (b)
	{
		GDL_FREE (b->data);
		GDL_FREE (b);	
	}
}

void
gdl_bit_vector_set (gdl_bit_vector * b, size_t i)
{
	if (gdl_check_range)
    {
      if (i >= b->size)         /* size_t is unsigned, can't be negative */
        {
           GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
        }
    }
	GDL_BITSET (b->data, i);
}

size_t
gdl_bit_vector_get (const gdl_bit_vector * b, size_t i)
{
	if (gdl_check_range)
    {
      if (i >= b->size)         /* size_t is unsigned, can't be negative */
        {
           GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
        }
    }
   return (GDL_BITTEST (b->data, i)) ? 1 : 0;
}

void
gdl_bit_vector_clear (gdl_bit_vector * b, size_t i)
{
	if (gdl_check_range)
    {
      if (i >= b->size)         /* size_t is unsigned, can't be negative */
        {
           GDL_ERROR_VOID ("index out of range", GDL_EINVAL);
        }
    }
	GDL_BITCLEAR (b->data, i);	
}

gdl_bit_slot *
gdl_bit_vector_slot (gdl_bit_vector * b, size_t i)
{
	if (gdl_check_range)
    {
      if (i >= b->slot)         /* size_t is unsigned, can't be negative */
        {
           GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
        }
    }
   return b->data+i;
}

size_t
gdl_bit_vector_cmp (const gdl_bit_vector * b1, const gdl_bit_vector * b2)
{
	if (b1->size==b2->size)
	{
		size_t i,j;
		for (i=0;i<b1->slot;i++)
		{
			if (b1->data[i]!=b2->data[i])
				break;
		}
		return (i<b1->slot) ? 1 : 0;
	}
	return 1;	
}

gdl_bit_vector *
gdl_bit_vector_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t size;
		gdl_bit_vector * b;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		 	
		b = gdl_bit_vector_alloc (size);
		
		status = fread (b->data, sizeof(gdl_bit_slot), b->slot, stream);
		GDL_FREAD_STATUS (status, b->slot);
		
		return b;
	}
	
	return 0;
}

int
gdl_bit_vector_fwrite (FILE * stream, gdl_bit_vector * b)
{
	if (stream && b)
	{
		int status;
		
		status = fwrite (&b->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fwrite (b->data, sizeof(gdl_bit_slot), b->slot, stream);
		GDL_FREAD_STATUS (status, b->slot);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}
