/* dico/dico.c
 * 
 * Copyright (C) 2008 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#include <sys/types.h> 
#include <sys/stat.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h> 
#include <gdl/gdl_dictionary.h>

gdl_dictionary *
gdl_dictionary_alloc (void)
{
	gdl_dictionary * d;
	
	d        = GDL_CALLOC (gdl_dictionary, 1);
	d->index = gdl_hashtable_alloc (gdl_interface_uint, 0);
	
	return d;
}

void 
gdl_dictionary_free (gdl_dictionary * d)
{
	if (d)
	{
		if (d->size)
		{
			size_t i;
			for(i = 0; i < d->size; i++)
			{
				gdl_hashtable_free (d->entries[i]);
				gdl_hashtable_free (d->reverse[i]);
			}	
			GDL_FREE (d->entries);
			GDL_FREE (d->reverse);
		}
		gdl_hashtable_free (d->index);
		GDL_FREE (d);	
	}	
}

int
gdl_dictionary_add (gdl_dictionary * d, const gdl_string * entry)
{
	size_t * idx = gdl_hashtable_lookup (d->index, entry);
	
	if (idx)
	{
		return *idx;
	}
	
	idx = GDL_MALLOC (size_t, 1);
	
	if (!d->size)
	{
		d->entries = GDL_MALLOC (gdl_hashtable * , 1);
		d->reverse = GDL_MALLOC (gdl_hashtable * , 1);
	}
	else
	{
		gdl_hashtable ** tmp = GDL_MALLOC (gdl_hashtable * , d->size+1);
		memcpy (tmp, d->entries, sizeof(gdl_hashtable *)*d->size);
		GDL_FREE (d->entries);
		d->entries=tmp;
		tmp = GDL_MALLOC (gdl_hashtable * , d->size+1);
		memcpy (tmp, d->reverse, sizeof(gdl_hashtable *)*d->size);
		GDL_FREE (d->reverse);
		d->reverse=tmp;
	}
	
	*idx=d->size;
	gdl_hashtable_add (d->index, entry, idx, 1);
	d->entries[d->size] = gdl_hashtable_alloc(gdl_interface_uint, 0);
	d->reverse[d->size] = gdl_hashtable_alloc(gdl_string_interface, 0);
	(d->size)++;
	
	return *idx;
}

int
gdl_dictionary_populate (gdl_dictionary * d, const gdl_string * entry, const gdl_string * def)
{
	size_t * idx = gdl_hashtable_lookup (d->index, entry);
	if (idx)
	{
		gdl_hashtable * table = d->entries[*idx];
		size_t * idx2 = gdl_hashtable_lookup (table, def);
		
		if (idx2)
		{
			return * idx2;
		}
		
		idx2  = GDL_MALLOC (size_t, 1);
		*idx2 = gdl_hashtable_size (table);
		gdl_hashtable_add (table, def, idx2, 1);
		
		// populate the reverse index
		{
			gdl_string * key = gdl_string_sprintf ("%d", *idx2); 
			table = d->reverse[*idx];
			gdl_hashtable_add (table, key, gdl_string_clone (def), 1);
			gdl_string_free (key);
		}
		
		return *idx2;
	}
	else
	{
		return -1; 
	}
}

const gdl_string *
gdl_dictionary_get (const gdl_dictionary * d, const gdl_string * entry, const size_t def_idx)
{
	size_t * idx = gdl_hashtable_lookup (d->index, entry);
	if (idx)
	{
		gdl_string * key      = gdl_string_sprintf ("%d", def_idx); 
		gdl_hashtable * table = d->reverse[*idx];
		gdl_string * def      = gdl_hashtable_lookup (table, key);
		gdl_string_free (key);
		return def;
	}
	return 0;
}

const int
gdl_dictionary_lookup (const gdl_dictionary * d, const gdl_string * entry, const gdl_string * def)
{
	size_t * idx = gdl_hashtable_lookup (d->index, entry);
	if (idx)
	{
		gdl_hashtable * table = d->entries[*idx];
		size_t * def_idx      = gdl_hashtable_lookup (table, def);
		if (def_idx)
			return *def_idx;
	}
	return -1;
}

size_t
gdl_dictionary_size (const gdl_dictionary * d, const gdl_string * entry)
{
	size_t * idx = gdl_hashtable_lookup (d->index, entry);
	if (idx)
	{
		gdl_hashtable * table = d->entries[*idx];
		return gdl_hashtable_size (table);
	}
	return 0;	
}

gdl_dictionary *
gdl_dictionary_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_dictionary * d;
		
		d = gdl_dictionary_alloc();
		
		status = fread (&d->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = gdl_hashtable_fread (stream, d->index);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		d->entries = GDL_MALLOC (gdl_hashtable *, d->size);
		d->reverse = GDL_MALLOC (gdl_hashtable *, d->size);
		for(i = 0; i < d->size; i++)
		{
			d->entries[i] = gdl_hashtable_alloc (gdl_interface_uint, 0);
			d->reverse[i] = gdl_hashtable_alloc (gdl_string_interface, 0);
			status = gdl_hashtable_fread (stream, d->entries[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
			status = gdl_hashtable_fread (stream, d->reverse[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		return d;
	}
	return 0;	
}

int
gdl_dictionary_fwrite (FILE * stream, const gdl_dictionary * d)
{
	if (stream && d)
	{
		int status;
		size_t i;
		
		status = fwrite (&d->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_hashtable_fwrite (stream, d->index);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		for(i = 0; i < d->size; i++)
		{
			status = gdl_hashtable_fwrite (stream, d->entries[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			status = gdl_hashtable_fwrite (stream, d->reverse[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}
