/*
 *  eqtl/bayes.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_sort_double.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_bayesian_regression.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>
#include <gdl/gdl_eqtl_bayes.h>

#define VECTOR_SET(X,i,y)(*(X->data+i*X->stride)=y)
#define VECTOR_GET(X,i)(*(X->data+i*X->stride))
#define MATRIX_SET(X,i,j,y)(*(X->data+(i*X->tda+j))=y)
#define MATRIX_GET(X,i,j)(*(X->data+(i*X->tda+j)))

size_t
gdl_eqtl_bayes_snp_init (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, const size_t snp, const size_t probe, double popf[], double * f)
{
	size_t i, j, k, kk, l, p, npop;
	
	bayes->y->size = bayes->X->size1 = chrom->nindiv;
	
	// first, how many populations ?
	for (npop = i = 0; i < chrom->npop; i++)
		if (gdl_snp_chromosome_is_polymorphic (chrom->snp_data, i, snp))
			npop++;
	
	bayes->P = npop;
	
	*f = 0;
	for (kk = k = i = 0; i < chrom->npop; i++)
	{
		popf[i] = 0;
		if (gdl_snp_chromosome_is_polymorphic (chrom->snp_data, i, snp))
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++, kk++)
			{
				VECTOR_SET (bayes->y, k, gdl_matrix_get (bayes->Y, kk, probe));
				switch(gdl_snp_chromosome_get_genotype (chrom->snp_data, i, j, snp))
				{
					case 2:
						MATRIX_SET(bayes->X, k, bayes->P, 2);
						MATRIX_SET(bayes->X, k, (bayes->P+1), 0);
						popf[i]+=1;
						//gdl_matrix_set (bayes->X, k, bayes->P, 2);
						//gdl_matrix_set (bayes->X, k, bayes->P+1, 0);
						break;
					case 1:
						MATRIX_SET (bayes->X, k, bayes->P, 1);
						MATRIX_SET (bayes->X, k, (bayes->P+1), 1);
						popf[i]+=0.5;
//						gdl_matrix_set (bayes->X, k, bayes->P, 1);
//						gdl_matrix_set (bayes->X, k, bayes->P+1, 1);
						break;
					case 0:
						MATRIX_SET (bayes->X, k, bayes->P, 0);
						MATRIX_SET (bayes->X, k, (bayes->P+1), 0);
//						gdl_matrix_set (bayes->X, k, bayes->P, 0);
//						gdl_matrix_set (bayes->X, k, bayes->P+1, 0);
						break;	
				}
				for (l = 1; l < bayes->P; l++)
				{
					MATRIX_SET (bayes->X, k, l, 0);
				}
				if (i < bayes->P-1)
				{
					MATRIX_SET (bayes->X, k, (i+1), 1);
				}
			}
			*f += popf[i];
			popf[i] /= chrom->pop_sizes[i];
		}
		else kk += chrom->pop_sizes[i];
	}
	
	bayes->y->size = bayes->X->size1 = k;
	
	*f /= k;
	
	return npop;
}

size_t
gdl_eqtl_bayes_snp_init_qnorm (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, const size_t snp, const size_t probe, double popf[], double * f, size_t * rm_individual[])
{
	size_t i, j, k, kk, l, p, npop, N = chrom->nindiv;
	// first, how many individuals ?
	if (rm_individual)
	{
		for(i = 0; i < chrom->npop; i++)
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++)
			{
				N -= rm_individual[i][j];
			}	
		}
	}
	bayes->y->size = bayes->X->size1 = N;
	// then, how many populations ?
	for (npop = i = 0; i < chrom->npop; i++)
	{
		if (gdl_snp_chromosome_is_polymorphic (chrom->snp_data, i, snp))
		{
			if (rm_individual)
			{
				for(j = 0; j < chrom->pop_sizes[i]; j++)
				{
					if (!rm_individual[i][j])
						break;
				}
				if (j < chrom->pop_sizes[i])
				{
					npop++;
				}
			}
			else
			{
				npop++;
			}
		}
	}
	if (npop)
	{
		bayes->P = npop;
		*f = 0;
		for (kk = k = i = 0; i < chrom->npop; i++)
		{
			popf[i] = 0;
			if (gdl_snp_chromosome_is_polymorphic (chrom->snp_data, i, snp))
			{
				for(j = 0; j < chrom->pop_sizes[i]; j++, kk++)
				{
					if (rm_individual && rm_individual[i][j])
					{
						continue;
					}
					VECTOR_SET (bayes->y, k, gdl_matrix_get (bayes->Y, kk, probe));
					switch(gdl_snp_chromosome_get_genotype (chrom->snp_data, i, j, snp))
					{
						case 2:
							MATRIX_SET(bayes->X, k, 1, 2);
							MATRIX_SET(bayes->X, k, 2, 0);
							popf[i]+=1;
							break;
						case 1:
							MATRIX_SET (bayes->X, k, 1, 1);
							MATRIX_SET (bayes->X, k, 2, 1);
							popf[i]+=0.5;
							break;
						case 0:
							MATRIX_SET (bayes->X, k, 1, 0);
							MATRIX_SET (bayes->X, k, 2, 0);
							break;	
					}
					k++;
					//printf ("%g %1.0f\n", VECTOR_GET (bayes->y, k), MATRIX_GET (bayes->X, k, 1));
				}
				*f += popf[i];
				popf[i] /= chrom->pop_sizes[i];
			}
			else kk += chrom->pop_sizes[i];
		}
		bayes->y->size = bayes->X->size1 = k;
		*f /= k;
	}
	
	return npop;
}

void
gdl_eqtl_bayes_gene_init (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, const gdl_boolean qnorm)
{
	size_t i, j, k, p, * rk = 0;
	double pr, * x = 0;

	bayes->np = block->size;
		
	gdl_matrix_free (bayes->Y);
	GDL_FREE (bayes->pidx);
	
	bayes->Y    = gdl_matrix_alloc (chrom->nindiv, bayes->np);
	bayes->pidx = GDL_MALLOC (size_t, bayes->np);
	
	for (p = 0; p < block->size; p++)
	{
		bayes->pidx[p] = p;
		for (k = i = 0; i < chrom->npop; i++)
		{
			const size_t n = chrom->pop_sizes[i];
			if (qnorm)
			{
				x  = GDL_CALLOC (double, n);
				rk = GDL_CALLOC (size_t, n);
				memcpy (x, block->probes[p]->data[i], sizeof(double)*n);
				gdl_sort_index (rk, x, 1, n);
				for (j = 0; j < n; j++)
				{
					pr = ((double)(j+1))/((double)(n+1));
					x[rk[j]] = gdl_ran_ugaussian_quantile (pr);
				}
				for(j = 0; j < n; j++, k++)
				{
					MATRIX_SET (bayes->Y, k, p, x[j]);
				}
				GDL_FREE (x);
				GDL_FREE (rk);
			}
			else
				for(j = 0; j < n; j++, k++)
					MATRIX_SET (bayes->Y, k, p, block->probes[p]->data[i][j]);
		}
	}
	
}

#include "../linalg/givens.c"
#include "../linalg/svdstep.c"

static int
gdl_eqtl_bayes_svd (gdl_matrix * A, gdl_matrix * V, gdl_vector * S, gdl_vector * work)
{
	size_t a, b, i, j;
  
  const size_t M = A->size1;
  const size_t N = A->size2;
  const size_t K = GDL_MIN (M, N);

  {
    gdl_vector_view f = gdl_vector_subvector (work, 0, K - 1);
    
    /* bidiagonalize matrix A, unpack A into U S V */
    
    gdl_linalg_bidiag_decomp (A, S, &f.vector);
    gdl_linalg_bidiag_unpack2 (A, S, &f.vector, V);
    
    /* apply reduction steps to B=(S,Sd) */
    
    chop_small_elements (S, &f.vector);
    
    /* Progressively reduce the matrix until it is diagonal */
    
    b = N - 1;
    
    while (b > 0)
      {
        double fbm1 = VECTOR_GET ((&f.vector), b - 1);

        if (fbm1 == 0.0 || gdl_isnan (fbm1))
          {
            b--;
            continue;
          }
        
        /* Find the largest unreduced block (a,b) starting from b
           and working backwards */
        
        a = b - 1;
        
        while (a > 0)
          {
            double fam1 = VECTOR_GET ((&f.vector), a - 1);

            if (fam1 == 0.0 || gdl_isnan (fam1))
              {
                break;
              }
            
            a--;
          }
        
        {
          const size_t n_block = b - a + 1;
          gdl_vector_view S_block = gdl_vector_subvector (S, a, n_block);
          gdl_vector_view f_block = gdl_vector_subvector (&f.vector, a, n_block - 1);
          
          gdl_matrix_view U_block =
            gdl_matrix_submatrix (A, 0, a, A->size1, n_block);
          gdl_matrix_view V_block =
            gdl_matrix_submatrix (V, 0, a, V->size1, n_block);
          
          qrstep (&S_block.vector, &f_block.vector, &U_block.matrix, &V_block.matrix);
          
          /* remove any small off-diagonal elements */
          
          chop_small_elements (&S_block.vector, &f_block.vector);
        }
      }
  }
  /* Make singular values positive by reflections if necessary */
  
  for (j = 0; j < K; j++)
    {
      double Sj = VECTOR_GET (S, j);
      
      if (Sj < 0.0)
        {
          for (i = 0; i < N; i++)
            {
              double Vij = MATRIX_GET (V, i, j);
              MATRIX_SET (V, i, j, (-Vij));
            }
          
          VECTOR_SET (S, j, -Sj);
        }
    }
  
//  /* Sort singular values into decreasing order */
//  
//  for (i = 0; i < K; i++)
//    {
//      double S_max = VECTOR_GET (S, i);
//      size_t i_max = i;
//      
//      for (j = i + 1; j < K; j++)
//        {
//          double Sj = VECTOR_GET (S, j);
//          
//          if (Sj > S_max)
//            {
//              S_max = Sj;
//              i_max = j;
//            }
//        }
//      
//      if (i_max != i)
//        {
//          /* swap eigenvalues */
//          gdl_vector_swap_elements (S, i, i_max);
//          
//          /* swap eigenvectors */
//          gdl_matrix_swap_columns (A, i, i_max);
//          gdl_matrix_swap_columns (V, i, i_max);
//        }
//    }
  
  return GDL_SUCCESS;	
}

int
gdl_eqtl_bayes_solve (gdl_bayreg_workspace * bayes)
{
   const size_t n = bayes->invOmega->size1;
   const size_t p = bayes->invOmega->size2;

   size_t i, j, p_eff;
   double alpha0;

   /* Balance the columns of the matrix A */

   //gdl_linalg_balance_columns (bayes->invOmega, bayes->D);

   /* Decompose A into U S Q^T */
   
   gdl_eqtl_bayes_svd (bayes->invOmega, bayes->Q, bayes->S, bayes->xt);
   
   /* Solve y = A c for c */

   gdl_blas_dgemv (CblasTrans, 1.0, bayes->invOmega, bayes->yy, 0.0, bayes->xt);

   //alpha0     = gdl_vector_get (bayes->S, 0);
   bayes->det = 1;
   
  for (j = 0; j < p; j++)
    {
      gdl_vector_view column = gdl_matrix_column (bayes->Q, j);
      double alpha = VECTOR_GET(bayes->S, j);
//		if (alpha <= GDL_DBL_EPSILON * alpha0) {
//        alpha = 0.0;
//      } else {
      bayes->det *= alpha;//*gdl_vector_get (bayes->D, j);
      alpha = 1.0 / alpha;
      //}
      gdl_vector_scale (&column.vector, alpha);
    }

   //gdl_vector_set_zero (bayes->b);

   gdl_blas_dgemv (CblasNoTrans, 1.0, bayes->Q, bayes->xt, 0.0, bayes->b);

   /* Unscale the balancing factors */

   //gdl_vector_div (bayes->b, bayes->D);

   return GDL_SUCCESS;
}

//double
//gdl_eqtl_bayes_compute_LRT (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, const double * popf, double * r2, double * p)
//{
//	size_t i, j, kx, ky;
//	double a, l0, l1, chi, e, sig0=0, sig1=0, xx=0, xy=0;
//	
//	for (kx = ky = i = 0; i < chrom->npop; i++)
//	{
//		if (!popf[i]) 
//		{
//			ky += chrom->pop_sizes[i];
//			continue;
//		}
//		for(j = 0; j < chrom->pop_sizes[i]; j++, kx++, ky++)
//		{
//			xx += (MATRIX_GET(bayes->X, kx, 1)-popf[i])*(MATRIX_GET(bayes->X, kx, 1)-popf[i]);
//			xy += (MATRIX_GET(bayes->X, kx, 1)-popf[i])*VECTOR_GET(bayes->y, ky);
//			sig0 += VECTOR_GET(bayes->y, ky)*VECTOR_GET(bayes->y, ky);
//		}
//	}
//	a=xy/xx;
//	for (kx = ky = i = 0; i < chrom->npop; i++)
//	{
//		if (!popf[i])
//		{
//			ky += chrom->pop_sizes[i];
//			continue;
//		}
//		for(j = 0; j < chrom->pop_sizes[i]; j++, kx++, ky++)
//		{
//			 e = VECTOR_GET(bayes->y, ky)-a*(MATRIX_GET(bayes->X, kx, 1)-popf[i]);
//			 sig1 += e*e;
//		}
//	}
//	chi = (log(sig0)-log(sig1))*kx;
//	chi = (chi >= 0) ? chi : 0; // in case of some numerical errors if sig0=sig1...
//	*p=gdl_sf_gamma_inc_Q (0.5,0.5*chi);
//	*r2=1-sig1/sig0;
//	
//	return chi;
//}

// Just for QNORM 
double
gdl_eqtl_bayes_compute_LRT (gdl_bayreg_workspace * bayes,
                            gdl_eqtl_chromosome * chrom,
                            size_t * rm_individuals[],
                            const double * popf,
                            double * r2,
                            double * p)
{
	size_t i, j, nn;
	double a, l0, l1, chi, e, f, g, sig0=0, sig1=0, x=0, y=0, xx=0, xy=0, yy=0;
	
	for (nn = i = 0; i < chrom->npop; i++)
	{
		if (!popf[i]) 
		{
			continue;
		}
		for(j = 0; j < chrom->pop_sizes[i]; j++)
		{
			if (rm_individuals && rm_individuals[i][j])
			{
				continue;
			}
			e   = MATRIX_GET(bayes->X, nn, 1);
			x  += e;
			xx += e*e;
			f   = VECTOR_GET(bayes->y, nn);
			y  += f;
			xy += e*f;
			nn++;
		}
	}
	a=(nn*xy-x*y)/(nn*xx-x*x);
	y/=nn;
	x/=nn;
	for (nn = i = 0; i < chrom->npop; i++)
	{
		if (!popf[i])
		{
			continue;
		}
		for(j = 0; j < chrom->pop_sizes[i]; j++)
		{
			if (rm_individuals && rm_individuals[i][j])
			{
				continue;
			}
			e = MATRIX_GET(bayes->X, nn, 1);
			f = VECTOR_GET(bayes->y, nn);
			sig0 += (f-y)*(f-y);
			g = (f-y)-a*(e-x);
			sig1 += g*g;
			nn++;
		}
	}
	
	chi = (sig0-sig1)*(nn - 2)/sig1;
	chi = (chi >= 0) ? chi : 0;
	*p  = gdl_sf_beta_inc (0.5*(nn - 2), 0.5,  (nn - 2)/((nn - 2) + chi));
	*r2 = 1-sig1/sig0;
	
	//printf ("%d %g %g %g\n", kx, sig0, sig1, -log(*p)/log(10));
	
	return chi;
}

double
gdl_eqtl_bayes_compute_factor (gdl_bayreg_workspace * bayes,
                               const double sigmaa,
                               const double sigmad)
{
	const size_t N = bayes->X->size1;
	
	size_t i, j;
	double e0, e1, t, s, s2, u, v, bf;
	
	gdl_matrix_free (bayes->invOmega);
	
	bayes->invOmega = gdl_matrix_calloc (3, 3);
	
	MATRIX_SET (bayes->invOmega, 1, 1, 1/(sigmaa*sigmaa));
	MATRIX_SET (bayes->invOmega, 2, 2, 1/(sigmad*sigmad));

	bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = bayes->X->size2 = 3;
	
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->X, bayes->X, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->X, bayes->y, 0, bayes->yy);
	
	gdl_eqtl_bayes_solve (bayes);
	
	e1 = s = s2 = 0;
	for (i = 0; i < N; i++)
	{
		t   = VECTOR_GET (bayes->y, i);
		s2 += t*t;
		s  += t;
		v   = t;
		for (j = 0; j < 3; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->X, i, j);
		}
		e1 += v*t;
	}
	e0 = s2-s*s/N;
	
	bf  = -0.5*log(bayes->det);
	bf += 0.5*log(N);
	bf -= log (sigmaa);
	bf -= log (sigmad);
	bf -= 0.5*N*(log(e1)-log(e0));
	
	return bf/log(10);
}

int
gdl_eqtl_bayes_residuals (gdl_bayreg_model     * model,
                          gdl_bayreg_workspace *  work,
                          gdl_eqtl_chromosome  * chrom,
                          gdl_eqtl_probe       * probe, 
                          const size_t sx, 
                          const size_t px)
{
	size_t i, j, k, kk;
	double f, * popf;
	gdl_vector * e;
	
	popf = GDL_MALLOC (double, chrom->npop);
	
	gdl_eqtl_bayes_snp_init_qnorm (work, chrom, sx, px, popf, &f, 0);
	e = gdl_bayreg_model_single_residual (model, work);
	for (k = kk = i = 0; i < chrom->npop; i++)
	{
		if (popf[i] == 0.0)
		{
			k += chrom->pop_sizes[i];
			continue;
		}
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++, kk++)
		{
			//fprintf (stderr, "%d %d %d %g", i, j, k, gdl_matrix_get (work->Y, k, px));
			gdl_matrix_set (work->Y, k, px, gdl_vector_get (e, kk));
			//fprintf (stderr, " %g\n", gdl_matrix_get (work->Y, k, px));
		}		
	}	
	gdl_vector_free (e); 
	GDL_FREE (popf);
}

double
gdl_eqtl_bayes_compute_factor_pop (gdl_bayreg_workspace * bayes,
                                const double sigmaa,
                                const double sigmad,
                                const double sigmap)
{
	const size_t N = bayes->X->size1;
	
	size_t i, j, k;
	double e0, e1, det0, t, v, bf, x;
	
	gdl_matrix_free (bayes->invOmega);
	
	bayes->invOmega = gdl_matrix_calloc (bayes->P, bayes->P);
	
	if (sigmap)
	{
		for (i = 1; i < bayes->P; i++)
			MATRIX_SET (bayes->invOmega, i, i, 1/(sigmap*sigmap));
	}
	
	bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = bayes->X->size2 = bayes->P;
	
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->X, bayes->X, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->X, bayes->y, 0, bayes->yy);
	
	gdl_eqtl_bayes_solve (bayes);
	
	e0 = 0;
	for (i = 0; i < N; i++)
	{
		t  = VECTOR_GET (bayes->y, i);
		v  = t;
		for (j = 0; j < bayes->P; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->X, i, j);
		}
		e0 += v*t;
	}
	
	det0 = bayes->det;
	
	bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = bayes->X->size2 = bayes->P+2;
	
	gdl_matrix_free (bayes->invOmega);
	bayes->invOmega = gdl_matrix_calloc (bayes->P+2, bayes->P+2);
	if (sigmap)
	{
		for (i = 1; i < bayes->P; i++)
			MATRIX_SET (bayes->invOmega, i, i, 1/(sigmap*sigmap));
	}
	MATRIX_SET (bayes->invOmega, bayes->P, bayes->P, 1/(sigmaa*sigmaa));
	MATRIX_SET (bayes->invOmega, (bayes->P+1), (bayes->P+1), 1/(sigmad*sigmad));
    
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->X, bayes->X, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->X, bayes->y, 0, bayes->yy);
	
	gdl_eqtl_bayes_solve (bayes);
	
	e1 = 0;
	for (i = 0; i < N; i++)
	{
		t = VECTOR_GET (bayes->y, i);
		v = t;
		for (j = 0; j < bayes->P+2; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->X, i, j);
		}
		e1 += v*t;
	}
	
	bf  = -0.5*log(bayes->det);
	bf +=  0.5*log(det0);
	bf -= log (sigmaa);
	bf -= log (sigmad);
	bf -= 0.5*N*(log(e1)-log(e0));
	
	return bf/log(10);
}

//gdl_eqtl_bayes_chromosome_result *
//gdl_eqtl_chromosome_cis_bayes (gdl_eqtl_chromosome * chrom,
//                               const gdl_bayreg_grid * grid,
//                               const double threshold,
//                               const gdl_boolean qnorm)
//{
//	const size_t G = chrom->ngene;
//	const size_t P = chrom->ploidy;
//	const size_t N = chrom->nindiv;
//	const size_t POP = chrom->npop;
//	size_t i, j, k, l, p, g, m, n, maxg;
//	gdl_genex_gene * gene;
//	gdl_genex_block * block;
//	gdl_snp ** snps;
//	double max, bf, u, a, d, * popf, f;
//	gdl_bayreg_workspace * bayes;
//	gdl_eqtl_bayes_chromosome_result * rchrom;
//	gdl_eqtl_bayes_gene_result * rgene;
//	gdl_eqtl_bayes_block_result * rblock;
//	gdl_eqtl_bayes_probe_result * rprobe;
//	gdl_eqtl_bayes_snp_result * rsnp;
//	
//	snps = chrom->snp_data->chrom->snps;
//	
//	bayes = gdl_eqtl_bayes_alloc (N, POP);
//	
//	popf = GDL_CALLOC (double, POP);
//	
//	if (chrom->logger)
//	{
//		fprintf (chrom->logger, ">%s |ngene=%d|nsnp=%d|npop=%d|nindiv=%d\n", chrom->name, G, chrom->snp_data->chrom->size, POP, N);
//		fflush(chrom->logger);
//	}
//	
//	rchrom = gdl_eqtl_bayes_chromosome_result_alloc (G);
//	
//	for (i = 0; i < G; i++)
//	{
//		gene = chrom->genes[i];
//		
//		rgene = gdl_eqtl_bayes_gene_result_alloc (gene->size);
//		
//		for (j = 0; j < gene->size; j++)
//		{
//			block = gene->blocks[j];
//			
//			rblock = gdl_eqtl_bayes_block_result_alloc (block->size, block->snp_up, block->snp_down);
//			
//			for (p = 0; p < block->size; p++)
//			{
//				rblock->probes[p] = gdl_eqtl_bayes_probe_result_alloc (block->snp_down-block->snp_up+1);
//			}
//			
//			//gdl_genex_block_clean_record_snp (block);
//			
//			gdl_eqtl_bayes_gene_init (bayes, chrom, block, qnorm);
//			
//			for (l = block->snp_up; l <= block->snp_down; l++)
//			{
//				//if (snps[l]->ignore == 'y') continue;
//				
//				u = a = d = 0;
//				
//				for (p = 0; p < bayes->np; p++)
//				{
//					rprobe = rblock->probes[bayes->pidx[p]];
//					
//					rsnp = gdl_eqtl_bayes_snp_result_alloc (grid->size);
//					
//					if (gdl_eqtl_bayes_snp_init (bayes, chrom, l, p, popf, &f)>1)
//					{
//						for (g = 0; g < grid->size; g++)
//						{
//							rsnp->bf[g] = gdl_eqtl_bayes_compute_factor (bayes, grid->points[g]->sigmaa, grid->points[g]->sigmad);
////							for (m = 0; m < bayes->P; m++)
////							{
////								u += grid->weight[g]*gdl_vector_get (bayes->b, m);
////							}
//							rsnp->a[g] = gdl_vector_get (bayes->b, bayes->P);
//							rsnp->d[g] = gdl_vector_get (bayes->b, bayes->P+1);
//						}
//					}
//					else
//					{
//						for (g = 0; g < grid->size; g++)
//						{
//							rsnp->bf[g] = gdl_eqtl_bayes_compute_factor (bayes, grid->points[g]->sigmaa, grid->points[g]->sigmad);
//							//u  += grid->weight[g]*gdl_vector_get (bayes->b, 0);	
//							rsnp->a[g] = gdl_vector_get (bayes->b, 1);
//							rsnp->d[g] = gdl_vector_get (bayes->b, 2);
//						}	
//					}
//					
//					rprobe->snps[l-block->snp_up] = rsnp;
//					
//				} // probe				
//				
//			} // snp
//			
//			rgene->blocks[j] = rblock;
//			
//		} // block 
//		
//		rchrom->genes[i] = rgene;
//		
//		if (chrom->logger)
//		{
//			gdl_progress_bar (chrom->logger, 25, i, G);
//		}
//	} // gene
//	
//	GDL_FREE (popf);
//	
//	gdl_eqtl_bayes_free (bayes);
//	
//	return rchrom;
//}

gdl_eqtl_bayes_gene_result *
gdl_eqtl_gene_cis_bayes (gdl_eqtl_chromosome * chrom,
                         gdl_eqtl_gene * gene,
                         gdl_bayreg_workspace * bayes,
                         gdl_bayreg_model * model,
                         const gdl_boolean qnorm,
                         size_t * rm_individual[],
                         double popf[],
                         size_t npop)
{
	size_t j, p, l, g, msize;
	double f, pval, r2;
	gdl_eqtl_block * block;
	gdl_eqtl_probe * probe;
	gdl_eqtl_bayes_gene_result * rgene;
	gdl_eqtl_bayes_block_result * rblock;
	gdl_eqtl_bayes_probe_result * rprobe;
	gdl_eqtl_bayes_snp_result * rsnp;
	
	rgene = gdl_eqtl_bayes_gene_result_alloc (gene->size);
	msize = gdl_bayreg_model_bf_storage_size (model);
		
	for (j = 0; j < gene->size; j++)
	{
		block  = gene->blocks[j];
		rblock = gdl_eqtl_bayes_block_result_alloc (block->size, block->snp_up, block->snp_down);
		for (p = 0; p < block->size; p++)
		{
			rblock->probes[p] = gdl_eqtl_bayes_probe_result_alloc (block->snp_down-block->snp_up+1);
		}
		gdl_eqtl_bayes_gene_init (bayes, chrom, block, qnorm);
		for (l = block->snp_up; l <= block->snp_down; l++)
		{
			for (p = 0; p < bayes->np; p++)
			{
				rprobe = rblock->probes[bayes->pidx[p]];
				rsnp   = gdl_eqtl_bayes_snp_result_alloc (msize);
				npop   = gdl_eqtl_bayes_snp_init_qnorm (bayes, chrom, l, p, popf, &f, rm_individual);
//				for (g = 0; g < grid->size; g++)
//				{
					//rsnp->bf[g] = gdl_eqtl_bayes_compute_factor (bayes, grid->points[g]->sigmaa, grid->points[g]->sigmad);
				//}
				gdl_bayreg_model_single_bf (model, bayes, rsnp->bf);
				gdl_eqtl_bayes_compute_LRT (bayes, chrom, rm_individual, popf, &r2, &pval);
				rsnp->pval = pval;
				rprobe->snps[l-block->snp_up] = rsnp;
			} // probe				
		} // snp
		rgene->blocks[j] = rblock;
	} // block 
	
	return rgene;
}                               

gdl_eqtl_bayes_chromosome_result *
gdl_eqtl_chromosome_cis_bayes (gdl_eqtl_chromosome * chrom,
                               gdl_bayreg_model * model,
                               const gdl_boolean qnorm,
                               size_t * rm_individual[],
                               const gdl_feature_table * gene_set)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, g, m, n, maxg, npop, msize;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double max, bf, u, a, d, * popf, f, r2, pval;
	gdl_bayreg_workspace * bayes;
	gdl_eqtl_bayes_chromosome_result * rchrom;
	gdl_eqtl_bayes_gene_result * rgene;
	gdl_eqtl_bayes_block_result * rblock;
	gdl_eqtl_bayes_probe_result * rprobe;
	gdl_eqtl_bayes_snp_result * rsnp;
	
	snps = chrom->snp_data->chrom->snps;
	
	msize = gdl_bayreg_model_bf_storage_size (model);
	
	bayes = gdl_bayreg_workspace_alloc (N, POP);
	
	popf = GDL_CALLOC (double, POP);
	
	if (chrom->logger)
	{
		fprintf (chrom->logger, ">%s |ngene=%d|nsnp=%d|npop=%d|nindiv=%d|\n", chrom->name, G, chrom->snp_data->chrom->size, POP, N);
		fflush(chrom->logger);
	}
	
	rchrom = gdl_eqtl_bayes_chromosome_result_alloc (G);
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		if (gene_set && !gdl_feature_table_is_inside (gene_set, chrom->name, gene->name))
		{
			rchrom->genes[i] = gdl_eqtl_bayes_gene_result_alloc (0);
			continue;
		}
		else
		{
			rgene = gdl_eqtl_bayes_gene_result_alloc (gene->size);	
		}
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			rblock = gdl_eqtl_bayes_block_result_alloc (block->size, block->snp_up, block->snp_down);
			
			for (p = 0; p < block->size; p++)
			{
				rblock->probes[p] = gdl_eqtl_bayes_probe_result_alloc (block->snp_down-block->snp_up+1);
			}
			
			gdl_eqtl_bayes_gene_init (bayes, chrom, block, qnorm);
			
			for (l = block->snp_up; l <= block->snp_down; l++)
			{
				//if (snps[l]->ignore == 'y') continue;
				
				u = a = d = 0;
				
				for (p = 0; p < bayes->np; p++)
				{
					rprobe = rblock->probes[bayes->pidx[p]];
					
					rsnp   = gdl_eqtl_bayes_snp_result_alloc (msize);
					
					npop   = gdl_eqtl_bayes_snp_init_qnorm (bayes, chrom, l, p, popf, &f, rm_individual);
					
					if (npop)
					{
						gdl_bayreg_model_single_bf (model, bayes, rsnp->bf);
						gdl_eqtl_bayes_compute_LRT (bayes, chrom, rm_individual, popf, &r2, &pval);
						rsnp->pval = pval;
						rprobe->snps[l-block->snp_up] = rsnp;
					}
					else
					{
						gdl_eqtl_bayes_snp_result_free (rsnp);
						rprobe->snps[l-block->snp_up] = 0;
					}
					
				} // probe				
				
			} // snp
			
			rgene->blocks[j] = rblock;
			
		} // block 
		
		rchrom->genes[i] = rgene;
		
		if (chrom->logger)
		{
			gdl_progress_bar (chrom->logger, 25, i, G);
		}
	} // gene
	
	GDL_FREE (popf);
	
	gdl_bayreg_workspace_free (bayes);
	
	return rchrom;
}

static void
gdl_eqtl_chromosome_cis_bayes_permut_block_signal (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, const gdl_rng * rng, const size_t within)
{
	size_t i, j, k, l, p;
	double t;
	
	for (p = 0; p < bayes->np; p++)
	{
		if (within)
		{
			for (k = i = 0; i < chrom->npop; i++)
			{
				for(j = chrom->pop_sizes[i]-1; j > 0; j--)
				{
					l = gdl_rng_uniform_int (rng, j+1);
					t = MATRIX_GET(bayes->Y, k+l, p);
					MATRIX_SET(bayes->Y, k+l, p, MATRIX_GET(bayes->Y, k+j, p));
					MATRIX_SET(bayes->Y, k+j, p, t);
				}
				k += chrom->pop_sizes[i];
			}
		}
		else
		{
			for(j = chrom->nindiv-1; j > 0; j--)
			{
				l = (j + 1) * gdl_rng_uniform (rng);
				t = MATRIX_GET(bayes->Y, l, p);
				MATRIX_SET(bayes->Y, l, p, MATRIX_GET(bayes->Y, j, p));
				MATRIX_SET(bayes->Y, j, p, t);
			}	
		}
	}
}
 
int
gdl_eqtl_chromosome_cis_bayes_permut (gdl_eqtl_chromosome * chrom,
                                      const gdl_bayreg_grid * grid,
                                      const size_t permut,
                                      const gdl_rng * rng,
                                      FILE * stream)
                                      //gdl_eqtl_bayes_permut * storage)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, g, m, n, b, maxg, npop;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double max, bf, logbf, gwt, u, a, d, * popf, f, r2, pval;
	gdl_bayreg_workspace * bayes;
	
	snps = chrom->snp_data->chrom->snps;
	
	bayes = gdl_bayreg_workspace_alloc (N, POP);
	
	popf = GDL_CALLOC (double, POP);
	
	if (chrom->logger)
	{
		fprintf (chrom->logger, ">%s |ngene=%d|nsnp=%d|npop=%d|nindiv=%d|\n", chrom->name, G, chrom->snp_data->chrom->size, POP, N);
		fflush(chrom->logger);
	}
	
	gwt = 0;
	for (g = 0; g < grid->size; g++)
		gwt += grid->weight[g];
		
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			gdl_eqtl_bayes_gene_init (bayes, chrom, block, 1);
			
			for (b = 0; b < permut; b++)
			{
				gdl_eqtl_chromosome_cis_bayes_permut_block_signal (bayes, chrom, rng, 0);	
			
				for (l = block->snp_up; l <= block->snp_down; l++)
				{
					u = a = d = 0;
					
					for (p = 0; p < bayes->np; p++)
					{
						npop = gdl_eqtl_bayes_snp_init_qnorm (bayes, chrom, l, p, popf, &f, 0);
						
						bf = 0;
						for (g = 0; g < grid->size; g++)
						{
							logbf = gdl_eqtl_bayes_compute_factor (bayes, grid->points[g]->sigmaa, grid->points[g]->sigmad);
							bf    += grid->weight[g]*pow(10, logbf);
						}
						bf /= gwt;
						
						//gdl_eqtl_bayes_compute_LRT (bayes, chrom, popf, &r2, &pval);
						//logbf = log(bf)/log(10);
						//gdl_eqtl_bayes_permut_add (storage, f, logbf, -log(pval)/log(10));
						
						fprintf (stream, "%e\n", bf);
						
					} // probe				
					
				} // snp
				
			} // permut
			
		} // block 
		
		if (chrom->logger)
		{
			gdl_progress_bar (chrom->logger, 25, i, G);
		}
	} // gene
	
	GDL_FREE (popf);
	
	gdl_bayreg_workspace_free (bayes);
}

int
gdl_eqtl_chromosome_cis_freq_permut (gdl_eqtl_chromosome * chrom,
                                     const size_t permut,
                                     const gdl_rng * rng,
                                     const gdl_boolean best_snp,
                                     FILE * stream)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, g, m, n, b, maxg, npop, nmax;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double max, bf, logbf, gwt, u, a, d, * popf, f, r2, pval;
	gdl_bayreg_workspace * bayes;
	
	snps = chrom->snp_data->chrom->snps;
	
	bayes = gdl_bayreg_workspace_alloc (N, POP);
	
	popf = GDL_CALLOC (double, POP);
	
	if (chrom->logger)
	{
		fprintf (chrom->logger, ">%s |ngene=%d|nsnp=%d|npop=%d|nindiv=%d|\n", chrom->name, G, chrom->snp_data->chrom->size, POP, N);
		fflush(chrom->logger);
	}
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			gdl_eqtl_bayes_gene_init (bayes, chrom, block, 1);
			
			for (b = 0; b < permut; b++)
			{
				gdl_eqtl_chromosome_cis_bayes_permut_block_signal (bayes, chrom, rng, 0);	
				
				for (p = 0; p < bayes->np; p++)
				{
					if (block->probes[p]->ignore == 'y')
						continue; 
					
					max  = 1.0;
					nmax = 0;
					
					for (l = block->snp_up; l <= block->snp_down; l++)
					{
						u = a = d = 0;
						
						npop = gdl_eqtl_bayes_snp_init_qnorm (bayes, chrom, l, p, popf, &f, 0);
						
						gdl_eqtl_bayes_compute_LRT (bayes, chrom, 0, popf, &r2, &pval);
						
						if (best_snp)
						{
							if (pval < max)
							{
								max = pval;
								nmax= 1;
							}
							else if (pval == max)
							{
								nmax++;	
							}
						}
						else
						{
							fprintf (stream, "%e\n", pval);
							fflush(stream);
						}
					} // snp
					if (best_snp)
					{
						fprintf (stream, "%d %e\n", nmax, max);
						fflush(stream);	
					}
				} // probe
				
			} // permut
			
		} // block 
		
		if (chrom->logger)
		{
			gdl_progress_bar (chrom->logger, 25, i, G);
		}
	} // gene
	
	GDL_FREE (popf);
	
	gdl_bayreg_workspace_free (bayes);
}

//void
//gdl_eqtl_chromosome_cis_bayes_permut (gdl_eqtl_chromosome * chrom, const gdl_bayreg_grid * grid, const double threshold, const gdl_boolean qnorm, const size_t permut, const gdl_rng * rng)
//{
//	const size_t G = chrom->ngene;
//	const size_t P = chrom->ploidy;
//	const size_t N = chrom->nindiv;
//	const size_t POP = chrom->npop;
//	size_t i, j, k, l, p, g, m, n, b, maxg;
//	gdl_genex_gene * gene;
//	gdl_genex_block * block;
//	gdl_snp ** snps;
//	double max, bf, u, a, d, * popf, f, pv, pv_permut, rsq;
//	gdl_bayreg_workspace * bayes;
//	
//	snps = chrom->snp_data->chrom->snps;
//	
//	bayes = gdl_eqtl_bayes_alloc (N, POP);
//	
//	popf = GDL_CALLOC (double, POP);
//	
//	for (i = 0; i < G; i++)
//	{
//		gene = chrom->genes[i];
//		
//		if (gene->ignore == 'y') continue;
//		
//		for (j = 0; j < gene->size; j++)
//		{
//			block = gene->blocks[j];
//			
//			if (block->ignore == 'y') continue;
//			
//			//gdl_genex_block_clean_record_snp (block);
//			
//			gdl_eqtl_bayes_gene_init (bayes, chrom, block, qnorm);
//			
//			gdl_genex_block_init_permut_storage (block);
//			
//			for (l = block->snp_up; l <= block->snp_down; l++)
//			{
//				if (snps[l]->ignore == 'y') continue;
//				
//				u = a = d = 0;
//				
//				for (p = 0; p < bayes->np; p++)
//				{
//					if (gdl_eqtl_bayes_snp_init (bayes, chrom, l, p, popf, &f)>1)
//					{
//						for (bf = 0, g = 0; g < grid->size; g++)
//						{
//							bf += grid->weight[g]*gdl_eqtl_bayes_compute_factor2 (bayes, grid->sigmaa[g], 0.25*grid->sigmaa[g], grid->sigmap, &rsq);
//							for (m = 0; m < bayes->P; m++)
//							{
//								u += grid->weight[g]*gdl_vector_get (bayes->b, m);
//							}
//							a  += grid->weight[g]*gdl_vector_get (bayes->b, bayes->P);
//							d  += grid->weight[g]*gdl_vector_get (bayes->b, bayes->P+1);
//						}
//					}
//					else
//					{
//						for (bf = 0, g = 0; g < grid->size; g++)
//							bf += grid->weight[g]*gdl_eqtl_bayes_compute_factor (bayes, grid->sigmaa[g], 0.25*grid->sigmaa[g], &rsq);
//					}
//					
//					gdl_genex_block_set_obs_pval (block, bayes->pidx[p], l, -bf);
//					
//				} // probes
//				
//			} // block snp
//			
//			for (b = 0; b < permut; b++)
//			{
//				gdl_eqtl_chromosome_cis_bayes_permut_block_signal (bayes, chrom, rng);	
//				
//				for (l = block->snp_up; l <= block->snp_down; l++)
//				{
//					if (snps[l]->ignore == 'y') continue;
//					
//					u = a = d = 0;
//					
//					for (p = 0; p < bayes->np; p++)
//					{
//						if (gdl_eqtl_bayes_snp_init (bayes, chrom, l, p, popf, &f)>1)
//						{
//							for (bf = 0, g = 0; g < grid->size; g++)
//								bf += grid->weight[g]*gdl_eqtl_bayes_compute_factor2 (bayes, grid->sigmaa[g], 0.25*grid->sigmaa[g], grid->sigmap, &rsq);
//						}
//						else
//						{
//							for (bf = 0, g = 0; g < grid->size; g++)
//								bf += grid->weight[g]*gdl_eqtl_bayes_compute_factor (bayes, grid->sigmaa[g], 0.25*grid->sigmaa[g], &rsq);
//						}
//						gdl_genex_block_set_permut_pval (block, bayes->pidx[p], l, -bf);
//						
//					} // probes
//					
//				} // block snp
//				
//			} // permut
//			
//			for (l = block->snp_up; l <= block->snp_down; l++)
//			{
//				if (snps[l]->ignore == 'y') continue;
//				
//				for (p = 0; p < bayes->np; p++)
//				{
//					if (chrom->logger)
//					{
//						pv = gdl_genex_block_get_obs_pval (block, bayes->pidx[p], l);
//						pv_permut = gdl_genex_block_get_permut_pval (block, bayes->pidx[p], l, permut);
//						fprintf (stdout, "%s %s %d %s %d %d %c %s %d %d %1.5f",chrom->name, snps[l]->rs, snps[l]->position, gene->name, block->start, block->end, block->strand, block->probes[bayes->pidx[p]]->name, block->probes[bayes->pidx[p]]->start, block->probes[bayes->pidx[p]]->end, f/(double)P);
//						for (m = 0; m < chrom->npop; m++)
//						{
//							fprintf (stdout, " %1.5f", popf[m]/(double)P);
//						}
//						fprintf (stdout, " %e %e\n", -pv, -log(pv_permut)/log(10));
//					}
//				}
//			}
//			
//			gdl_genex_block_clean_permut_storage (block);
//			
//		} // block
//		
//	} // gene
//	
//	GDL_FREE (popf);
//	
//	gdl_eqtl_bayes_free (bayes);
//}
