/*
 *  eqtl/bin.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_eqtl_bayes.h>

gdl_eqtl_bayes_bin *
gdl_eqtl_bayes_bin_alloc (const size_t idx)
{
	gdl_eqtl_bayes_bin * b;
	
	b = GDL_CALLOC (gdl_eqtl_bayes_bin, 1);
	b->idx = idx;
	
	return b;	
}

void
gdl_eqtl_bayes_bin_free (gdl_eqtl_bayes_bin * b)
{
	if (b)
	{
		size_t i, j;
		for (i = 0; i < b->nchrom; i++)
		{
			for (j = 0; j < b->snp_sizes[i]; j++)
			{
				gdl_eqtl_bayes_snp_record_free (b->snps[i][j]);
			}
		}
		GDL_FREE (b->snps);
		GDL_FREE (b->snp_sizes);
		gdl_hashtable_free (b->snp_table);
		GDL_FREE (b);
	}
}

void
gdl_eqtl_bayes_bin_init (gdl_eqtl_bayes_bin * b, const size_t nchrom, const gdl_rng * rng)
{
	size_t i;
	gdl_hashtable_itr * itr;
	
	if (b->snp_table)
	{
		size_t * n;
		
		b->nchrom = nchrom;
		b->snps   = GDL_CALLOC (gdl_eqtl_bayes_snp_record **, nchrom);
		n = b->snp_sizes = GDL_CALLOC (size_t, nchrom);
		
		itr = gdl_hashtable_iterator (b->snp_table);
		do
		{
		 	gdl_eqtl_bayes_snp_record * r = (gdl_eqtl_bayes_snp_record *) gdl_hashtable_iterator_value (itr);
		 	n[r->cidx]++;
		} while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
		
		for (b->nfunc = i = 0; i < nchrom; i++)
		{
			if (n[i])
			{
				b->snps[i] = GDL_MALLOC (gdl_eqtl_bayes_snp_record *, n[i]);
			}
			b->nfunc += n[i];
			n[i]=0;
		}
		
		b->proba = gdl_ran_beta (rng, 1+b->nfunc, 1+b->nsnp-b->nfunc);
		
		itr = gdl_hashtable_iterator (b->snp_table);
		do
		{
		 	gdl_eqtl_bayes_snp_record * r = (gdl_eqtl_bayes_snp_record *) gdl_hashtable_iterator_value (itr);
		 	b->snps[r->cidx][n[r->cidx]] = r;
		 	r->gamma=b->proba;
		 	n[r->cidx]++;
		} while (gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
				
		gdl_hashtable_free (b->snp_table);
		b->snp_table=0;
	}
	else
	{
		b->proba = gdl_ran_beta (rng, 1, 1+b->nsnp);
	}
}

gdl_eqtl_bayes_snp_record *
gdl_eqtl_bayes_bin_add_functional_snp (gdl_eqtl_bayes_bin * b, const size_t ngrid, const size_t chromidx, const size_t snpidx, const gdl_snp * snp)
{
	if (!b->snp_table)
	{
		b->nfunc = 0;
		b->snp_table = gdl_hashtable_alloc (gdl_hash_default, 0);
	}
	gdl_string * key;
	gdl_eqtl_bayes_snp_record * rec = gdl_eqtl_bayes_snp_record_alloc (ngrid);
	
	rec->cidx = chromidx;
	rec->sidx = snpidx;
	rec->snp  = gdl_snp_clone (snp);
	rec->bin  = b;
	
	key = gdl_string_sprintf ("%d-%d", chromidx, snpidx);
	
	gdl_hashtable_update (b->snp_table, key, rec, 0);
	
	gdl_string_free (key);
	
	return rec;
}

int
gdl_eqtl_bayes_bin_fread (FILE * stream, gdl_eqtl_bayes_bin * b)
{
	if (stream && b)
	{
		int status;
		size_t i, j;
		
		status = fread (&b->idx, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->ngrid, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->nsnp, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->nfunc, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->nchrom, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->nzfunc, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->proba, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->length, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->gamma, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->gene, sizeof(gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		b->snp_sizes = GDL_MALLOC (size_t, b->nchrom);
		status = fread (b->snp_sizes, sizeof(size_t), b->nchrom, stream);
		GDL_FREAD_STATUS (status, b->nchrom);
		
		b->snps = GDL_CALLOC (gdl_eqtl_bayes_snp_record **, b->nchrom);
		for (i = 0; i < b->nchrom; i++)
		{
			if (b->snp_sizes[i])
			{
				b->snps[i] = GDL_MALLOC (gdl_eqtl_bayes_snp_record *, b->snp_sizes[i]);
				for (j = 0; j < b->snp_sizes[i]; j++)
				{
					b->snps[i][j] = gdl_eqtl_bayes_snp_record_fread (stream, b->ngrid);
					GDL_FREAD_STATUS (b->snps[i][j]!=0, 1);
				}
			}
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

int
gdl_eqtl_bayes_bin_fwrite (FILE * stream, const gdl_eqtl_bayes_bin * b)
{
	if (stream && b)
	{
		int status;
		size_t i, j;
		
		status = fwrite (&b->idx, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->ngrid, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->nsnp, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->nfunc, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->nchrom, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->nzfunc, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->proba, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->length, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->gamma, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->gene, sizeof(gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		status = fwrite (b->snp_sizes, sizeof(size_t), b->nchrom, stream);
		GDL_FWRITE_STATUS (status, b->nchrom);
		
		for (i = 0; i < b->nchrom; i++)
		{
			if (b->snp_sizes[i])
			{
				for (j = 0; j < b->snp_sizes[i]; j++)
				{
					status = gdl_eqtl_bayes_snp_record_fwrite (stream, b->snps[i][j], b->ngrid);
					GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				}
			}
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}
