/*
 *  eqtl/gdl_eqtl_bayes.h 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#ifndef __GDL_EQTL_BAYES_H__
#define __GDL_EQTL_BAYES_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_snp.h>
#include <gdl/gdl_multifit.h>
#include <gdl/gdl_multifit_nlin.h>
#include <gdl/gdl_bayesian_regression.h>
#include <gdl/gdl_eqtl_chromosome.h>

__BEGIN_DECLS

double gdl_eqtl_bayes_compute_factor (gdl_bayreg_workspace * bayes, const double sigmaa, const double sigmad);
double gdl_eqtl_bayes_compute_factor_pop (gdl_bayreg_workspace * bayes, const double sigmaa, const double sigmad, const double sigmap);
int gdl_eqtl_bayes_solve (gdl_bayreg_workspace * bayes);
void gdl_eqtl_bayes_gene_init (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, const gdl_boolean qnorm);
size_t gdl_eqtl_bayes_snp_init (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, const size_t snp, const size_t probe, double popf[], double * f);

typedef struct
{
	double * bf;
	double pval;
} gdl_eqtl_bayes_snp_result;

gdl_eqtl_bayes_snp_result * gdl_eqtl_bayes_snp_result_alloc (const size_t grid_size);
void gdl_eqtl_bayes_snp_result_free (gdl_eqtl_bayes_snp_result * r);
gdl_eqtl_bayes_snp_result * gdl_eqtl_bayes_snp_result_fread (FILE * stream, const size_t grid_size);
int gdl_eqtl_bayes_snp_result_fwrite (FILE * stream, const gdl_eqtl_bayes_snp_result * r, const size_t grid_size);

typedef struct
{
	size_t size;
	gdl_eqtl_bayes_snp_result ** snps;
} gdl_eqtl_bayes_probe_result;

gdl_eqtl_bayes_probe_result * gdl_eqtl_bayes_probe_result_alloc (const size_t size);
void gdl_eqtl_bayes_probe_result_free (gdl_eqtl_bayes_probe_result * r);
gdl_eqtl_bayes_probe_result * gdl_eqtl_bayes_probe_result_fread (FILE * stream, const size_t grid_size);
int gdl_eqtl_bayes_probe_result_fwrite (FILE * stream, const gdl_eqtl_bayes_probe_result * r, const size_t grid_size);

typedef struct
{
	size_t size;
	long snp_from;
	long snp_to;
	gdl_eqtl_bayes_probe_result ** probes;
} gdl_eqtl_bayes_block_result;

gdl_eqtl_bayes_block_result * gdl_eqtl_bayes_block_result_alloc (const size_t size, const long snp_from, const long snp_to);
void gdl_eqtl_bayes_block_result_free (gdl_eqtl_bayes_block_result * r);
gdl_eqtl_bayes_block_result * gdl_eqtl_bayes_block_result_fread (FILE * stream, const size_t grid_size);
int gdl_eqtl_bayes_block_result_fwrite (FILE * stream, const gdl_eqtl_bayes_block_result * r, const size_t grid_size);

typedef struct
{
	size_t size;
	gdl_eqtl_bayes_block_result ** blocks;
} gdl_eqtl_bayes_gene_result;

gdl_eqtl_bayes_gene_result * gdl_eqtl_bayes_gene_result_alloc (const size_t size);
void gdl_eqtl_bayes_gene_result_free (gdl_eqtl_bayes_gene_result * r);
gdl_eqtl_bayes_gene_result * gdl_eqtl_bayes_gene_result_fread (FILE * stream, const size_t grid_size);
int gdl_eqtl_bayes_gene_result_fwrite (FILE * stream, const gdl_eqtl_bayes_gene_result * r, const size_t grid_size);

typedef struct
{
	size_t size;
	gdl_eqtl_bayes_gene_result ** genes;
} gdl_eqtl_bayes_chromosome_result;

gdl_eqtl_bayes_chromosome_result * gdl_eqtl_bayes_chromosome_result_alloc (const size_t size);
void gdl_eqtl_bayes_chromosome_result_free (gdl_eqtl_bayes_chromosome_result * r);
gdl_eqtl_bayes_chromosome_result * gdl_eqtl_bayes_chromosome_result_fread (FILE * stream, const size_t grid_size);
int gdl_eqtl_bayes_chromosome_result_fwrite (FILE * stream, const gdl_eqtl_bayes_chromosome_result * r, const size_t grid_size);

typedef struct
{
	size_t size;
	gdl_string * dir;
	gdl_bayreg_model * model;
	gdl_string ** chroms;
} gdl_eqtl_bayes_genome_result;

gdl_eqtl_bayes_genome_result * gdl_eqtl_bayes_genome_result_alloc (const gdl_string * dir, const size_t size, const gdl_bayreg_model * model);
void gdl_eqtl_bayes_genome_result_free (gdl_eqtl_bayes_genome_result * r);

int gdl_eqtl_bayes_genome_result_set (gdl_eqtl_bayes_genome_result * g, size_t i, const gdl_string * name, const gdl_eqtl_bayes_chromosome_result * chrom);
gdl_eqtl_bayes_chromosome_result * gdl_eqtl_bayes_genome_result_get (const gdl_eqtl_bayes_genome_result * g, const size_t i);

gdl_eqtl_bayes_genome_result * gdl_eqtl_bayes_genome_result_fread (FILE * stream);
int gdl_eqtl_bayes_genome_result_fwrite (FILE * stream, const gdl_eqtl_bayes_genome_result * r);

//gdl_eqtl_bayes_chromosome_result * gdl_eqtl_chromosome_cis_bayes (gdl_eqtl_chromosome * chrom, const gdl_bayreg_model * model, const double threshold, const gdl_boolean qnorm);
gdl_eqtl_bayes_chromosome_result * gdl_eqtl_chromosome_cis_bayes (gdl_eqtl_chromosome * chrom, gdl_bayreg_model * model, const gdl_boolean qnorm, size_t * rm_individual[], const gdl_feature_table * gene_set);

typedef struct
{
	size_t nf_bin;  // number of bins for the SNP MAF
	size_t nbf_bin; // number of bins for the Bayes Factor (log10)
	size_t npv_bin; // number of bins for the P-values (log10)
	double log_max;
	double log_min;
	size_t ***counts;
} gdl_eqtl_bayes_permut;

gdl_eqtl_bayes_permut * gdl_eqtl_bayes_permut_alloc (const size_t nf_bin, const size_t nbf_bin, const size_t npv_bin, const double log_min, const double log_max);
void gdl_eqtl_bayes_permut_free (gdl_eqtl_bayes_permut * p);
int gdl_eqtl_bayes_permut_fprintf (FILE * stream, const gdl_eqtl_bayes_permut * p);
void gdl_eqtl_bayes_permut_add (gdl_eqtl_bayes_permut * p, const double snp_maf, const double snp_logbf, const double snp_logpv); 

int
gdl_eqtl_chromosome_cis_bayes_permut (gdl_eqtl_chromosome * chrom,
                                      const gdl_bayreg_grid * grid,
                                      const size_t permut,
                                      const gdl_rng * rng,
                                      FILE * stream);
                                      //gdl_eqtl_bayes_permut * storage);

int
gdl_eqtl_chromosome_cis_freq_permut (gdl_eqtl_chromosome * chrom,
                                     const size_t permut,
                                     const gdl_rng * rng,
                                     const gdl_boolean best_snp,
                                     FILE * stream);

size_t gdl_eqtl_bayes_snp_init_qnorm (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, const size_t snp, const size_t probe, double popf[], double * f, size_t * rm_individual[]);
void gdl_eqtl_bayes_gene_init (gdl_bayreg_workspace * bayes, gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, const gdl_boolean qnorm);
int gdl_eqtl_bayes_residuals (gdl_bayreg_model * model, gdl_bayreg_workspace *  work, gdl_eqtl_chromosome  * chrom, gdl_eqtl_probe * probe, const size_t sx, const size_t px);

__END_DECLS

#endif
