/*
 *  eqtl/gdl_eqtl_genome.h 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#ifndef __GDL_EQTL_GENOME_H__
#define __GDL_EQTL_GENOME_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>
#include <gdl/gdl_eqtl_bayes.h>

__BEGIN_DECLS

typedef struct
{
	size_t chrom_idx;
	size_t gene_idx;
} gdl_eqtl_gene_entry;

typedef struct
{
	size_t npop;
	size_t * pop_sizes;
	size_t * cumpop_sizes;
	size_t nsample;
	size_t nvar;
	size_t nfactor;
	size_t nvector;
	gdl_string ** pop_names;
	gdl_string ** sample_names;
	gdl_string ** var_names;
	gdl_string *** levels;
	unsigned char * var_types;
	size_t * level_by_factor;
	size_t ** factors;
	double ** vectors;
	size_t * popx;
} gdl_eqtl_sample_info;

gdl_eqtl_sample_info * gdl_eqtl_sample_info_alloc (const size_t npop, const size_t pop_sizes[]);
void gdl_eqtl_sample_info_free (gdl_eqtl_sample_info * i);
int gdl_eqtl_sample_info_fscanf (FILE * stream, gdl_eqtl_sample_info * i); 
int gdl_eqtl_sample_info_fwrite (FILE * stream, const gdl_eqtl_sample_info * i);
gdl_eqtl_sample_info * gdl_eqtl_sample_info_fread (FILE * stream);
double * gdl_eqtl_sample_info_get_var (const gdl_eqtl_sample_info * p, const gdl_string * name); 
size_t * gdl_eqtl_sample_info_get_factor (const gdl_eqtl_sample_info * p, const gdl_string * name, size_t *nz);

typedef struct
{
	size_t nchrom;
	size_t ngene;
	gdl_string         * dbdir;
	gdl_string        ** chroms;
	gdl_snp_annot_dico * dico;
	gdl_hashtable      * genes;
	gdl_eqtl_sample_info * sample_info;
	FILE * logger;
} gdl_eqtl_genome;

gdl_eqtl_genome * gdl_eqtl_genome_alloc (const gdl_string * config_file, 
                                         const gdl_string * probe_dir,
                                         const gdl_string * annot_dir,
                                         const gdl_string * refseq_dir,
                                         const gdl_string * gmap_dir,
                                         const gdl_string * dbdir,
                                         gdl_snp_data_format * format,
                                         gdl_boolean cnv_data);
void gdl_eqtl_genome_free (gdl_eqtl_genome * v);

size_t gdl_eqtl_genome_size (const gdl_eqtl_genome * g);
size_t gdl_eqtl_genome_probe_size (const gdl_eqtl_genome * g);
size_t gdl_eqtl_genome_snp_size (const gdl_eqtl_genome * g);
size_t gdl_eqtl_genome_select_most_variable_probe (gdl_eqtl_genome * v, const double prop, gdl_boolean extreme_outlier);
size_t gdl_eqtl_genome_select_snp (gdl_eqtl_genome * v, const double fmin, const double gmin, const gdl_boolean poly_pop);
void gdl_eqtl_genome_collapse_gene (gdl_eqtl_genome * v);
const gdl_eqtl_gene_entry * gdl_eqtl_genome_search_gene (const gdl_eqtl_genome * v, const gdl_string * name);

void gdl_eqtl_chromosome_trans_test (gdl_eqtl_genome * genome, gdl_eqtl_chromosome * chrom, const size_t c, const double pmin, const long cis_window);

gdl_eqtl_chromosome * gdl_eqtl_genome_get (const gdl_eqtl_genome * g, size_t i);
gdl_eqtl_chromosome * gdl_eqtl_genome_lookup (const gdl_eqtl_genome * g, const gdl_string * name);
int gdl_eqtl_genome_set (const gdl_eqtl_genome * g, size_t i, gdl_eqtl_chromosome * c);
gdl_eqtl_genome * gdl_eqtl_genome_fread (FILE * stream);
int gdl_eqtl_genome_fwrite (FILE * stream, const gdl_eqtl_genome * v);
FILE * gdl_eqtl_genome_set_logger (gdl_eqtl_genome * g, FILE * stream);

void gdl_eqtl_genome_meta_probeset (gdl_eqtl_genome * v, const size_t meta_type);

__END_DECLS

#endif
