/*
 *  eqtl/gdl_eqtl_interaction.h 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#ifndef __GDL_EQTL_INTERACTION_H__
#define __GDL_EQTL_INTERACTION_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_snp.h>
#include <gdl/gdl_multifit.h>
#include <gdl/gdl_multifit_nlin.h>
#include <gdl/gdl_eqtl_chromosome.h>

__BEGIN_DECLS

typedef struct
{
	double * a;     // effect
	double * tstat; // t-stat
	double r2;
	double p0;      // p-value of H1:H0
	double p1;      // p-value of H2:H1
} gdl_eqtl_interac_snp;

gdl_eqtl_interac_snp * gdl_eqtl_interac_snp_alloc (const size_t factor_size);
void gdl_eqtl_interac_snp_free (gdl_eqtl_interac_snp * r);
gdl_eqtl_interac_snp * gdl_eqtl_interac_snp_fread (FILE * stream, const size_t factor_size);
int gdl_eqtl_interac_snp_fwrite (FILE * stream, const gdl_eqtl_interac_snp * r, const size_t factor_size);

typedef struct
{
	size_t size;
	gdl_eqtl_interac_snp ** snps;
} gdl_eqtl_interac_probe;

gdl_eqtl_interac_probe * gdl_eqtl_interac_probe_alloc (const size_t size);
void gdl_eqtl_interac_probe_free (gdl_eqtl_interac_probe * r);
gdl_eqtl_interac_probe * gdl_eqtl_interac_probe_fread (FILE * stream, const size_t factor_size);
int gdl_eqtl_interac_probe_fwrite (FILE * stream, const gdl_eqtl_interac_probe * r, const size_t factor_size);

typedef struct
{
	size_t size;
	long snp_from;
	long snp_to;
	gdl_eqtl_interac_probe ** probes;
} gdl_eqtl_interac_block;

gdl_eqtl_interac_block * gdl_eqtl_interac_block_alloc (const size_t size, const long snp_from, const long snp_to);
void gdl_eqtl_interac_block_free (gdl_eqtl_interac_block * r);
gdl_eqtl_interac_block * gdl_eqtl_interac_block_fread (FILE * stream, const size_t factor_size);
int gdl_eqtl_interac_block_fwrite (FILE * stream, const gdl_eqtl_interac_block * r, const size_t factor_size);

typedef struct
{
	size_t size;
	gdl_eqtl_interac_block ** blocks;
} gdl_eqtl_interac_gene;

gdl_eqtl_interac_gene * gdl_eqtl_interac_gene_alloc (const size_t size);
void gdl_eqtl_interac_gene_free (gdl_eqtl_interac_gene * r);
gdl_eqtl_interac_gene * gdl_eqtl_interac_gene_fread (FILE * stream, const size_t factor_size);
int gdl_eqtl_interac_gene_fwrite (FILE * stream, const gdl_eqtl_interac_gene * r, const size_t factor_size);

typedef struct
{
	size_t size;
	gdl_eqtl_interac_gene ** genes;
} gdl_eqtl_interac_chromosome;

gdl_eqtl_interac_chromosome * gdl_eqtl_interac_chromosome_alloc (const size_t size);
void gdl_eqtl_interac_chromosome_free (gdl_eqtl_interac_chromosome * r);
gdl_eqtl_interac_chromosome * gdl_eqtl_interac_chromosome_fread (FILE * stream, const size_t factor_size);
int gdl_eqtl_interac_chromosome_fwrite (FILE * stream, const gdl_eqtl_interac_chromosome * r, const size_t factor_size);

typedef struct
{
	size_t size;
	size_t nz;
	gdl_string * dir;
	gdl_string ** chroms;
} gdl_eqtl_interac_genome;

gdl_eqtl_interac_genome * gdl_eqtl_interac_genome_alloc (const gdl_string * dir, const size_t size, const size_t NZ);
void gdl_eqtl_interac_genome_free (gdl_eqtl_interac_genome * r);

int gdl_eqtl_interac_genome_set (gdl_eqtl_interac_genome * g, size_t i, const gdl_string * name, const gdl_eqtl_interac_chromosome * chrom);
gdl_eqtl_interac_chromosome * gdl_eqtl_interac_genome_get (const gdl_eqtl_interac_genome * g, const size_t i);

gdl_eqtl_interac_genome * gdl_eqtl_interac_genome_fread (FILE * stream);
int gdl_eqtl_interac_genome_fwrite (FILE * stream, const gdl_eqtl_interac_genome * r);

// WORKSPACE

gdl_eqtl_interac_chromosome *
gdl_eqtl_chromosome_interaction (gdl_eqtl_chromosome * chrom,
                                 const size_t Z[],
                                 const size_t nz,
	                              const gdl_boolean qnorm,
	                              const gdl_boolean mnorm,
	                              const gdl_boolean anova,
	                              const gdl_feature_table * gene_set);

int
gdl_eqtl_chromosome_interaction_permut (gdl_eqtl_chromosome * chrom,
                                       const size_t pZ[], // permutated Z-vector
		                                 const size_t Z[],
		                                 const size_t nz,
			                              const gdl_boolean qnorm,
			                              const gdl_boolean mnorm,
			                              const gdl_boolean anova,
			                              const gdl_feature_table * gene_set,
			                              const double cutoff,
			                              const size_t best,
			                              FILE * stream);	                              

__END_DECLS

#endif
