/*
 *  eqtl/interac.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>
#include <stdio.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_sort_double.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_multireg.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>
#include <gdl/gdl_eqtl_interaction.h>

#define VECTOR_SET(X,i,y)(*(X->data+i*X->stride)=y)
#define VECTOR_GET(X,i)(*(X->data+i*X->stride))
#define MATRIX_SET(X,i,j,y)(*(X->data+(i*X->tda+j))=y)
#define MATRIX_GET(X,i,j)(*(X->data+(i*X->tda+j)))

static void
_interac_geneExp2Vector (const gdl_eqtl_chromosome * chrom,
			                const gdl_eqtl_probe * probe,
			                const gdl_boolean qnorm,
			                const gdl_boolean mnorm,
			                const size_t z[],
			                const size_t nz,
			                double y[])
{
	size_t i, j, k, p, iz, * rk = 0, * ix;
	double pr, * x = 0;

	for (k = i = 0; i < chrom->npop; i++)
	{
		const size_t n = chrom->pop_sizes[i];
		if (qnorm)
		{
//			ix = GDL_CALLOC (size_t, nz);
//			x  = GDL_MATRIX_ALLOC (double, nz, n);
//			rk = GDL_MATRIX_ALLOC (size_t, nz, n);
//			for (j = 0; j < n; j++)
//			{	
//				x[z[k+j]][ix[z[k+j]]++]=probe->data[i][j];
//				//printf ("%d %d %d z=%d ix=%d\n", i, j, k, z[k+j], ix[z[k+j]]);
//			}
//			for (iz = 0; iz < nz; iz++)
//			{
//				const size_t nn = ix[iz];
//				gdl_sort_index (rk[iz], x[iz], 1, nn);
//				for (ix[iz] = j = 0; j < n; j++)
//				{
//					if (z[k+j] != iz) continue;
//					pr = ((double)(ix[iz]+0.5))/((double)nn);
//					size_t rkx = rk[z[k+j]][ix[iz]];
//					x[z[k+j]][rkx] = gdl_ran_ugaussian_quantile (pr);
//					//printf ("%d %d %d rkx=%d ix=%d (%d)\n", i, j, k, rkx, ix[iz], nn);
//					ix[iz]++;
//				}
//			}
//			// reset ix
//			for (iz = 0; iz < nz; iz++) ix[iz]=0;
//			// copy
//			for (j = 0; j < n; j++, k++)
//			{	
//				y[k] = x[z[k]][ix[z[k]]++];
//			}
//			GDL_FREE (ix);
//			GDL_MATRIX_FREE (x,nz);
//			GDL_MATRIX_FREE (rk,nz);
			x  = GDL_CALLOC (double, n);
			rk = GDL_CALLOC (size_t, n);
			memcpy (x, probe->data[i], sizeof(double)*n);
			gdl_sort_index (rk, x, 1, n);
			for (j = 0; j < n; j++)
			{
				pr = ((double)(j+1))/((double)(n+1));
				x[rk[j]] = gdl_ran_ugaussian_quantile (pr);
			}
			for(j = 0; j < n; j++, k++)
			{
				y[k]=x[j];
			}
			GDL_FREE (x);
			GDL_FREE (rk);
		}
		else
		{
			memcpy(&y[k], probe->data[i], sizeof(double)*n);
			k+=n;
		}
	}	
	if (mnorm && nz==2) 	           
	{
		// regress y ~ mu + factor + e and y* = e
		// This assumes that nz==2
		double a, e, f, xm=0, ym=0, xx=0, xy=0, yy=0;
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
			{
				e   = z[k];
				xm  += e;
				xx  += e*e;
				f   = y[k];
				ym  += f;
				xy += e*f;
			}
		}
		a=(k*xy-xm*ym)/(k*xx-xm*xm);
		ym/=k;
		xm/=k;
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
			{
				 e = z[k];
				 f = y[k];
				 y[k] = (f-ym)-a*(e-xm); // residual
			}
		}
	}	
}

static size_t
_interact_SNPGeno2Vector (const gdl_eqtl_chromosome * chrom,
                          const size_t snp,
                          double X[])
{
	size_t i, j, k, n = 0;
	
	for (k = i = 0; i < chrom->npop; i++)
	{
		if (gdl_snp_chromosome_is_polymorphic (chrom->snp_data, i, snp))
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++, n++)
			{
				X[k] = (double)gdl_snp_chromosome_get_genotype (chrom->snp_data, i, j, snp);
			}
		}
		else 
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
			{
				X[k] = 3.0;
			}
		}	
	}
	
	return n;
}

// compute the effects and 
// return the residual
static double
_interac_effect (const double y[],
                 const double x[],
                 const size_t z[],
                 const size_t n,
                 const double y_sum,
                 const double xz[],
                 const double yz[],
                 const size_t nz,
                 const size_t nn,
                 double a[],
                 double t[])
{
	if (nz == 2)
	{
		size_t i;
		double mu,det,y_bar,sig=0,e;
		double * A = GDL_MALLOC (double, 6);
		
		A[0] = xz[2]*xz[3];
		A[1] = -xz[0]*xz[3];
		A[2] = -xz[1]*xz[2];
		A[3] = nn*xz[3]-xz[1]*xz[1];
		A[4] = xz[0]*xz[1];
		A[5] = nn*xz[2]-xz[0]*xz[0];
		det = nn*A[0]+xz[0]*A[1]+xz[1]*A[2];
		if (det <= GDL_DBL_MIN)
		{
			return GDL_NAN;
		}
		det  = 1.0/det;
		mu   = det*(A[0]*y_sum+A[1]*yz[0]+A[2]*yz[1]);
		a[0] = det*(A[1]*y_sum+A[3]*yz[0]+A[4]*yz[1]);
		a[1] = det*(A[2]*y_sum+A[4]*yz[0]+A[5]*yz[1]);
		
		for(i = 0; i < n; i++)
		{
			if (x[i] == 3.0)
				continue;
			y_bar = mu + ((z[i]) ? a[1]*x[i] : a[0]*x[i]);
			e     = y_bar - y[i];
			sig  += e*e;
		}
		t[0]  = a[0]/sqrt(sig*det*A[3]/(nn-3));
		t[1]  = a[1]/sqrt(sig*det*A[5]/(nn-3));
		// convert in p-value
		t[0]  = gdl_sf_beta_inc (0.5*(nn - 3), 0.5,  (nn - 3)/((nn - 3) + t[0]*t[0]));
		t[1]  = gdl_sf_beta_inc (0.5*(nn - 3), 0.5,  (nn - 3)/((nn - 3) + t[1]*t[1]));
		
		if (sig == 0)
		{
			for(i = 0; i < n; i++)
			{
				printf ("%g = %1.0f * %d\n", y[i], x[i], z[i]);
			}	
		}
		//printf ("Intercept %g\n", mu);
		//printf ("A0        %g %g\n", a[0], t[0]);
		//printf ("A1        %g %g\n", a[1], t[1]);
		
		return sig;
	}
	else
	{
		// TODO
	}
}

static int
_interac_Perform (const gdl_eqtl_chromosome * chrom,
			         const double Y[],
			         const double X[],
			         const size_t Z[],
			         const size_t nz,
			         gdl_eqtl_interac_snp * rsnp)
{
	size_t i, j, k, n = 0;
	double sig2, a, p, r2, l0, l1, chi, e, f, g, sig0=0, sig1=0, x=0, y=0, xx=0, xy=0, yy=0;
	double * xz = GDL_CALLOC (double, nz*2);
	double * yz = GDL_CALLOC (double, nz);
	
	for (n = k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			if (X[k] == 3.0) 
			   continue;
			e   = X[k];
			x  += e;
			xx += e*e;
			f   = Y[k];
			y  += f;
			xy += e*f;
			g   = Z[k];
			xz[Z[k]]    += e;
			xz[Z[k]+nz] += e*e;
			yz[Z[k]]    += e*f;
			n++;
			//printf (">%g %1.0f %d\n", Y[k], X[k], Z[k]);
		}
	}
	// Model including interaction
	sig2 = _interac_effect (Y, X, Z, k, y, xz, yz, nz, n, rsnp->a, rsnp->tstat);
	// Model without interaction
	a=(n*xy-x*y)/(n*xx-x*x);
	y/=n;
	x/=n;
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			 if (X[k]==3.0) continue;
			 e = X[k];
			 f = Y[k];
			 sig0 += (f-y)*(f-y);
			 g = (f-y)-a*(e-x);
			 sig1 += g*g;
		}
	}
	// Stats for H1:H0
	chi = (sig0-sig1)*(n - 2)/sig1;
	chi = (chi >= 0) ? chi : 0;
	rsnp->p0 = gdl_sf_beta_inc (0.5*(n - 2), 0.5,  (n - 2)/((n - 2) + chi));
	//printf ("P0 = %g\n", rsnp->p0);
	// Stats for H2:H1
	chi = (sig1-sig2)*(n - 3)/sig2;
	chi = (chi >= 0) ? chi : 0;
	rsnp->p1 = gdl_sf_beta_inc (0.5*(n - 3), 0.5,  (n - 3)/((n - 3) + chi));
	rsnp->r2 = 1-sig2/sig0;
	
	GDL_FREE (xz);
	GDL_FREE (yz);
}

static int
_interac_ANOVA (gdl_multireg_linear_workspace * workspace,
                const gdl_eqtl_chromosome * chrom,
			       const double Y[],
			       const double X[],
			       const size_t Z[],
			       const size_t pZ[],
			       const size_t nz,
			       const size_t n,
			       gdl_eqtl_interac_snp * rsnp,
			       gdl_vector * YY,
			       gdl_matrix * X0,
			       gdl_matrix * X1)
{
	size_t i, j, k, l;
	const gdl_vector * beta;

   for (k = l = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, l++)
		{
			if (X[l] == 3.0) 
			   continue;
			VECTOR_SET (YY,k,Y[l]);
			MATRIX_SET (X0,k,0,1);
			MATRIX_SET (X0,k,1,Z[l]);
			MATRIX_SET (X0,k,2,X[l]);
			MATRIX_SET (X1,k,pZ[l],X[l]);
			k++;
		}
	}
	
	YY->size  = k;
   X0->size1 = X1->size1 = k;
	
	gdl_multireg_linear_eval (workspace, X0, X1, YY);
	
	beta = gdl_multireg_linear_coefficients (workspace);
	
	rsnp->tstat[0] = VECTOR_GET (beta, 0);
	rsnp->tstat[1] = VECTOR_GET (beta, 1);
	rsnp->p1       = gdl_multireg_linear_eval_pvalue (workspace);
	
	//printf("%g %g %g\n", rsnp->tstat[0], rsnp->tstat[1], rsnp->p1);
}

gdl_eqtl_interac_chromosome *
gdl_eqtl_chromosome_interaction (gdl_eqtl_chromosome * chrom,
                                 const size_t Z[],
                                 const size_t nz,
	                              const gdl_boolean qnorm,
	                              const gdl_boolean mnorm,
	                              const gdl_boolean anova,
	                              const gdl_feature_table * gene_set)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, g, m, n, nn, maxg, npop, msize;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_genex_probe * probe;
	gdl_snp ** snps;
	double * Y, * X;
	gdl_eqtl_interac_chromosome * rchrom;
	gdl_eqtl_interac_gene * rgene;
	gdl_eqtl_interac_block * rblock;
	gdl_eqtl_interac_probe * rprobe;
	gdl_eqtl_interac_snp * rsnp;
	gdl_multireg_linear_workspace * anovaWork;
	gdl_vector * YY;
	gdl_matrix * X0, * X1;
	
	snps = chrom->snp_data->chrom->snps;
	
	if (chrom->logger)
	{
		fprintf (chrom->logger, ">%s |ngene=%d|nsnp=%d|npop=%d|nindiv=%d|\n", chrom->name, G, chrom->snp_data->chrom->size, POP, N);
		fflush(chrom->logger);
	}
	
	rchrom = gdl_eqtl_interac_chromosome_alloc (G);
	
	Y = GDL_MALLOC (double, chrom->nindiv);
	X = GDL_MALLOC (double, chrom->nindiv);
	
	if (anova)
	{
		anovaWork = gdl_multireg_linear_workspace_alloc (chrom->nindiv, 5);
		YY = gdl_vector_alloc (chrom->nindiv);
		X0 = gdl_matrix_alloc (chrom->nindiv, 3);
		X1 = gdl_matrix_alloc (chrom->nindiv, 2);	
	}
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		if (gene_set && !gdl_feature_table_is_inside (gene_set, chrom->name, gene->name))
		{
			rchrom->genes[i] = gdl_eqtl_interac_gene_alloc (0);
			continue;
		}
		else
		{
			rgene = gdl_eqtl_interac_gene_alloc (gene->size);	
		}
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			rblock = gdl_eqtl_interac_block_alloc (block->size, block->snp_up, block->snp_down);
			
			for (p = 0; p < block->size; p++)
			{
				probe = block->probes[p];
				
				rblock->probes[p] = gdl_eqtl_interac_probe_alloc (block->snp_down-block->snp_up+1);
			
				rprobe = rblock->probes[p];
				
				_interac_geneExp2Vector (chrom, probe, qnorm, mnorm, Z, nz, Y);
				
				for (l = block->snp_up; l <= block->snp_down; l++)
				{
					rsnp = gdl_eqtl_interac_snp_alloc (nz);
					
					nn = _interact_SNPGeno2Vector (chrom, l, X);
					
					if (anova)
					{
						_interac_ANOVA (anovaWork, chrom, Y, X, Z, Z, nz, nn, rsnp, YY, X0, X1);	
					}
					else
					{
						_interac_Perform (chrom, Y, X, Z, nz, rsnp);
					}
					
					rprobe->snps[l-block->snp_up] = rsnp;
				} // snp
				
			} // probe
			
			rgene->blocks[j] = rblock;
			
		} // block 
		
		rchrom->genes[i] = rgene;
		
		if (chrom->logger)
		{
			gdl_progress_bar (chrom->logger, 25, i, G);
		}
	} // gene
	
	GDL_FREE (Y);
	GDL_FREE (X);
	
	if (anova)
	{
		gdl_vector_free (YY);
		gdl_matrix_free (X0);
		gdl_matrix_free (X1);
		gdl_multireg_linear_workspace_free (anovaWork);
	}
	
	return rchrom;
}

int
gdl_eqtl_chromosome_interaction_permut (gdl_eqtl_chromosome * chrom,
													const size_t pZ[],
		                                 const size_t Z[],
		                                 const size_t nz,
			                              const gdl_boolean qnorm,
			                              const gdl_boolean mnorm,
			                              const gdl_boolean anova,
			                              const gdl_feature_table * gene_set,
			                              const double cutoff,
			                              const size_t best,
			                              FILE * stream)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, l, b, p, nbest, nn;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_genex_probe * probe;
	gdl_eqtl_interac_snp * rsnp;
	gdl_snp ** snps;
	double * Y, * X, best_pval;
	gdl_multireg_linear_workspace * anovaWork;
	gdl_vector * YY;
	gdl_matrix * X0, * X1;
	
	snps = chrom->snp_data->chrom->snps;
	
	if (chrom->logger)
	{
		fprintf (chrom->logger, ">%s |ngene=%d|nsnp=%d|npop=%d|nindiv=%d|\n", chrom->name, G, chrom->snp_data->chrom->size, POP, N);
		fflush(chrom->logger);
	}
	
	Y = GDL_MALLOC (double, chrom->nindiv);
	X = GDL_MALLOC (double, chrom->nindiv);
	
	rsnp = gdl_eqtl_interac_snp_alloc (nz);
	
	if (anova)
	{
		anovaWork = gdl_multireg_linear_workspace_alloc (chrom->nindiv, 5);
		YY = gdl_vector_alloc (chrom->nindiv);
		X0 = gdl_matrix_alloc (chrom->nindiv, 3);
		X1 = gdl_matrix_alloc (chrom->nindiv, 2);	
	}
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		if (gene_set && !gdl_feature_table_is_inside (gene_set, chrom->name, gene->name))
		{
			continue;
		}
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			for (p = 0; p < block->size; p++)
			{
				probe = block->probes[p];
					
				_interac_geneExp2Vector (chrom, probe, qnorm, mnorm, Z, nz, Y);
				
				nbest     = 0;
				best_pval = 1.0;
				
				for (l = block->snp_up; l <= block->snp_down; l++)
				{
					nn = _interact_SNPGeno2Vector (chrom, l, X);
					
					if (anova)
					{
						_interac_ANOVA (anovaWork, chrom, Y, X, Z, pZ, nz, nn, rsnp, YY, X0, X1);	
					}
					else
					{
						_interac_Perform (chrom, Y, X, pZ, nz, rsnp);
					}				
					
					if (rsnp->p1 <= cutoff)
					{
						if (!best)
						{
							fprintf (stream, "%s %d %g\n", chrom->name, i, rsnp->p1);
						}
						else if (rsnp->p1 < best_pval)
						{
							best_pval = rsnp->p1;
							nbest     = 1;
						}
						else if (rsnp->p1 == best_pval)
						{
							(nbest)++;
						}
					}			
				} // snp
				if (best && best_pval <= cutoff)
				{
					fprintf (stream, "%s %d %d %g\n", chrom->name, i, nbest, best_pval);	
				}
			} // probe
			
		} // block 
		
		if (chrom->logger)
		{
			gdl_progress_bar (chrom->logger, 25, i, G);
		}
	} // gene
	
	if (anova)
	{
		gdl_vector_free (YY);
		gdl_matrix_free (X0);
		gdl_matrix_free (X1);
		gdl_multireg_linear_workspace_free (anovaWork);
	}
	
	GDL_FREE (Y);
	GDL_FREE (X);
	gdl_eqtl_interac_snp_free (rsnp);
}

gdl_eqtl_interac_snp *
gdl_eqtl_interac_snp_alloc (const size_t grid_size)
{
	gdl_eqtl_interac_snp * r;
	
	r = GDL_CALLOC (gdl_eqtl_interac_snp, 1);
	
	r->a     = GDL_CALLOC (double, grid_size);
	r->tstat = GDL_CALLOC (double, grid_size);
	
	return r;	
}

void
gdl_eqtl_interac_snp_free (gdl_eqtl_interac_snp * r)
{
	if (r)
	{
		GDL_FREE (r->a);
		GDL_FREE (r->tstat);
		GDL_FREE (r);
	}
}

gdl_eqtl_interac_snp *
gdl_eqtl_interac_snp_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		gdl_eqtl_interac_snp * r;
		
		r = gdl_eqtl_interac_snp_alloc (grid_size);
		
		status = fread (r->a, sizeof(double), grid_size, stream);
		GDL_FREAD_STATUS (status, grid_size);
		status = fread (r->tstat, sizeof(double), grid_size, stream);
		GDL_FREAD_STATUS (status, grid_size);
		status = fread (&(r->p0), sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(r->p1), sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(r->r2), sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return r;	
	}
	return 0;
}

int
gdl_eqtl_interac_snp_fwrite (FILE * stream, const gdl_eqtl_interac_snp * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		
		status = fwrite (r->a, sizeof(double), grid_size, stream);
		GDL_FWRITE_STATUS (status, grid_size);
		status = fwrite (r->tstat, sizeof(double), grid_size, stream);
		GDL_FWRITE_STATUS (status, grid_size);
		status = fwrite (&(r->p0), sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(r->p1), sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(r->r2), sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/**
 * 
 * Probes
 * 
 */

gdl_eqtl_interac_probe *
gdl_eqtl_interac_probe_alloc (const size_t size)
{
	gdl_eqtl_interac_probe * r;
	
	r = GDL_MALLOC (gdl_eqtl_interac_probe, 1);
	
	r->size = size;
	r->snps = GDL_CALLOC (gdl_eqtl_interac_snp *, size);
	
	return r;
}

void
gdl_eqtl_interac_probe_free (gdl_eqtl_interac_probe * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_interac_snp_free (r->snps[i]);
		GDL_FREE (r->snps);
		GDL_FREE (r);
	}
}

gdl_eqtl_interac_probe *
gdl_eqtl_interac_probe_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		gdl_eqtl_interac_probe * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_interac_probe_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			r->snps[i] = gdl_eqtl_interac_snp_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->snps[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_interac_probe_fwrite (FILE * stream, const gdl_eqtl_interac_probe * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_interac_snp_fwrite (stream, r->snps[i], grid_size);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/**
 * 
 * Blocks
 * 
 */

gdl_eqtl_interac_block *
gdl_eqtl_interac_block_alloc (const size_t size, const long snp_from, const long snp_to)
{
	gdl_eqtl_interac_block * r;
	
	r = GDL_MALLOC (gdl_eqtl_interac_block, 1);
	
	r->size     = size;
	r->snp_from = snp_from;
	r->snp_to   = snp_to;
	r->probes   = GDL_CALLOC (gdl_eqtl_interac_probe *, size);
	
	return r;
}

void
gdl_eqtl_interac_block_free (gdl_eqtl_interac_block * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_interac_probe_free (r->probes[i]);
		GDL_FREE (r->probes);
		GDL_FREE (r);
	}
}

gdl_eqtl_interac_block *
gdl_eqtl_interac_block_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		long from, to;
		gdl_eqtl_interac_block * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&from, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&to, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_interac_block_alloc (size, from, to);
		
		for (i = 0; i < size; i++)
		{
			r->probes[i] = gdl_eqtl_interac_probe_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->probes[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_interac_block_fwrite (FILE * stream, const gdl_eqtl_interac_block * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->snp_from, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->snp_to, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_interac_probe_fwrite (stream, r->probes[i], grid_size);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/**
 * 
 * Genes
 * 
 */

gdl_eqtl_interac_gene *
gdl_eqtl_interac_gene_alloc (const size_t size)
{
	gdl_eqtl_interac_gene * r;
	
	r = GDL_MALLOC (gdl_eqtl_interac_gene, 1);
	
	r->size = size;
	r->blocks = GDL_CALLOC (gdl_eqtl_interac_block *, size);
	
	return r;
}

void
gdl_eqtl_interac_gene_free (gdl_eqtl_interac_gene * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_interac_block_free (r->blocks[i]);
		GDL_FREE (r->blocks);
		GDL_FREE (r);
	}
}

gdl_eqtl_interac_gene *
gdl_eqtl_interac_gene_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		gdl_eqtl_interac_gene * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_interac_gene_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			r->blocks[i] = gdl_eqtl_interac_block_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->blocks[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_interac_gene_fwrite (FILE * stream, const gdl_eqtl_interac_gene * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_interac_block_fwrite (stream, r->blocks[i], grid_size);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/*
 * Chromosome
 */
 
gdl_eqtl_interac_chromosome *
gdl_eqtl_interac_chromosome_alloc (const size_t size)
{
	gdl_eqtl_interac_chromosome * r;
	
	r = GDL_MALLOC (gdl_eqtl_interac_chromosome, 1);
	
	r->size = size;
	r->genes = GDL_CALLOC (gdl_eqtl_interac_gene *, size);
	
	return r;
}

void
gdl_eqtl_interac_chromosome_free (gdl_eqtl_interac_chromosome * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_eqtl_interac_gene_free (r->genes[i]);
		GDL_FREE (r->genes);
		GDL_FREE (r);
	}
}

gdl_eqtl_interac_chromosome *
gdl_eqtl_interac_chromosome_fread (FILE * stream, const size_t grid_size)
{
	if (stream)
	{
		int status;
		size_t i,size;
		gdl_eqtl_interac_chromosome * r;
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r = gdl_eqtl_interac_chromosome_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			r->genes[i] = gdl_eqtl_interac_gene_fread (stream, grid_size);
			GDL_FREAD_STATUS (r->genes[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_interac_chromosome_fwrite (FILE * stream, const gdl_eqtl_interac_chromosome * r, const size_t grid_size)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_eqtl_interac_gene_fwrite (stream, r->genes[i], grid_size);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

/*
 * Genome
 */
 
gdl_eqtl_interac_genome *
gdl_eqtl_interac_genome_alloc (const gdl_string * dir, const size_t size, const size_t nz)
{
	gdl_eqtl_interac_genome * r;
	
	r = GDL_MALLOC (gdl_eqtl_interac_genome, 1);
	
	r->dir    = gdl_string_clone (dir);
	r->size   = size;
	r->chroms = GDL_CALLOC (gdl_string *, size);
	r->nz     = nz;
	
	return r;
}

void
gdl_eqtl_interac_genome_free (gdl_eqtl_interac_genome * r)
{
	if (r)
	{
		size_t i;
		for (i = 0; i < r->size; i++)
			gdl_string_free (r->chroms[i]);
		GDL_FREE (r->chroms);
		gdl_string_free (r->dir);
		GDL_FREE (r);
	}
}

gdl_eqtl_interac_genome *
gdl_eqtl_interac_genome_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_eqtl_interac_genome * r;
		
		r = GDL_CALLOC (gdl_eqtl_interac_genome, 1);
		
		r->dir = gdl_string_fread (stream);
		GDL_FREAD_STATUS (r->dir!=0, 1);
		
		status = fread (&r->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&r->nz, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		r->chroms = GDL_CALLOC (gdl_string *, r->size);
		
		for (i = 0; i < r->size; i++)
		{
			r->chroms[i] = gdl_string_fread (stream);
			GDL_FREAD_STATUS (r->chroms[i]!=0, 1);
		}
		
		return r;
	}
	
	return 0;
}

int
gdl_eqtl_interac_genome_fwrite (FILE * stream, const gdl_eqtl_interac_genome * r)
{
	if (stream && r)
	{
		int status;
		size_t i;
		
		status = gdl_string_fwrite (stream, r->dir);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		status = fwrite (&r->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&r->nz, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < r->size; i++)
		{
			status = gdl_string_fwrite (stream, r->chroms[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
	
	return GDL_EINVAL;
}

int
gdl_eqtl_interac_genome_set (gdl_eqtl_interac_genome * g, size_t i, const gdl_string * name, const gdl_eqtl_interac_chromosome * chrom)
{
	FILE * stream;
	gdl_string * filename = gdl_string_sprintf ("%s/%s.res", g->dir, name);
	
	gdl_string_free (g->chroms[i]);
	
	g->chroms[i] = gdl_string_clone (name);
	
	stream = gdl_fileopen (filename, "w");
	
	gdl_eqtl_interac_chromosome_fwrite (stream, chrom, g->nz);
	
	gdl_fileclose (filename, stream);
	
	gdl_string_free (filename);
}

gdl_eqtl_interac_chromosome *
gdl_eqtl_interac_genome_get (const gdl_eqtl_interac_genome * g, size_t i)
{
	FILE * stream;
	gdl_string * filename = gdl_string_sprintf ("%s/%s.res", g->dir, g->chroms[i]);
	gdl_eqtl_interac_chromosome * chrom;
	
	stream = gdl_fileopen (filename, "r");
	
	chrom = gdl_eqtl_interac_chromosome_fread (stream, g->nz);
	
	gdl_fileclose (filename, stream);
	
	gdl_string_free (filename);
	
	return chrom;	
}



