/*
 *  eqtl/perm.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>

void gdl_eqtl_chromosome_get_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq);
double ** gdl_eqtl_chromosome_cis_get_block_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, double ** var, size_t ** pidx, size_t * np);
double gdl_eqtl_chromosome_do_LRT (gdl_eqtl_chromosome * chrom, const double * x, const double * y, const double * popf, double * a, double * r2, double * p);

static void
gdl_eqtl_chromosome_cis_permut_block_signal (gdl_eqtl_chromosome * chrom, double ** Y, size_t np, const gdl_rng * rng)
{
	size_t i, j, k, l, p;
	double t;
	
	for (p = 0; p < np; p++)
	{
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = chrom->pop_sizes[i]-1; j > 0; j--)
			{
				l = (j + 1) * gdl_rng_uniform (rng);
				t = Y[p][k+l];
				Y[p][k+l] = Y[p][k+j];
				Y[p][k+j] = t;
			}
			k += chrom->pop_sizes[i];
		}
	}	
}

void
gdl_eqtl_chromosome_cis_permut (gdl_eqtl_chromosome * chrom, const double freqmin, const size_t permut_size, const gdl_rng * rng)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, m, p, b, np, * pidx;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double * popf, f, chi, t, pv, pv_permut, r2, * x, ** Y, * var;
	
	snps = chrom->snp_data->chrom->snps;
	
	x = GDL_MALLOC (double, N);
	popf = GDL_MALLOC (double, POP);
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		if (gene->ignore == 'y') continue;
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			if (block->ignore == 'y') continue;
			
			Y = gdl_eqtl_chromosome_cis_get_block_signal (chrom, block, &var, &pidx, &np);
			
			gdl_genex_block_init_permut_storage (block);
			
			for (l = block->snp_up; l <= block->snp_down; l++)
			{
				if (snps[l]->ignore == 'y') continue;
				
				gdl_eqtl_chromosome_get_snp_vector (chrom, l, x, popf, &f);
					
				for (p = 0; p < np; p++)
				{
					chi = gdl_eqtl_chromosome_do_LRT (chrom, x, Y[p], popf, &t, &r2, &pv);
					gdl_genex_block_set_obs_pval (block, pidx[p], l, pv);
				}
			}
			
			for (b = 0; b < permut_size; b++)
			{
				
				gdl_eqtl_chromosome_cis_permut_block_signal (chrom, Y, np, rng);
				
				for (l = block->snp_up; l <= block->snp_down; l++)
				{
					if (snps[l]->ignore == 'y') continue;
					
					gdl_eqtl_chromosome_get_snp_vector (chrom, l, x, popf, &f);
					
					if (f < freqmin*P) continue; // exclude rare SNP
					// permut vector
					for (p = 0; p < np; p++)
					{
						chi = gdl_eqtl_chromosome_do_LRT (chrom, x, Y[p], popf, &t, &r2, &pv);
						gdl_genex_block_set_permut_pval (block, pidx[p], l, pv);
					}
				}
				
			}
			
			for (l = block->snp_up; l <= block->snp_down; l++)
			{
				if (snps[l]->ignore == 'y') continue;
				
				gdl_eqtl_chromosome_get_snp_vector (chrom, l, x, popf, &f);
				
				for (p = 0; p < np; p++)
				{
					if (chrom->logger)
					{
						pv = gdl_genex_block_get_obs_pval (block, pidx[p], l);
						pv_permut = gdl_genex_block_get_permut_pval (block, pidx[p], l, permut_size);
						fprintf (chrom->logger, "%s %s %d %s %d %d %d %d %c %s %d %d %1.5f",chrom->name,snps[l]->rs,snps[l]->position,gene->name,gene->id,j+1,block->start,block->end,block->strand,block->probes[pidx[p]]->name, block->probes[pidx[p]]->start, block->probes[pidx[p]]->end, f/(double)P);
						for (m = 0; m < chrom->npop; m++)
						{
							fprintf (chrom->logger, " %1.5f", popf[m]/(double)P);
						}
						fprintf (chrom->logger, " %g %g %g %e %e\n", chi, t, r2, pv, pv_permut);
					}
				}
			}
			
			gdl_genex_block_clean_permut_storage (block);
			
			GDL_MATRIX_FREE (Y, np);
			GDL_FREE (var);
			GDL_FREE (pidx);
		}
	}
}
