/*
 *  eqtl/reg.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>

static void
_gdl_eqtl_chromosome_cis_test_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq)
{
	size_t i, j, k, p;
	*freq = 0;
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(pop_freq[i] = 0, j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			switch(gdl_snp_chromosome_get_genotype (chrom->snp_data, i, j, snp))
			{
				case 2:
					x[k]=2;
					pop_freq[i]+=2;
					break;
				case 1:
					x[k]=1;
					pop_freq[i]+=1;
					break;
				case 0:
					x[k]=0;
					break;	
			}
		}
		*freq += pop_freq[i];
		pop_freq[i] /= chrom->pop_sizes[i];
	}
	*freq/=chrom->nindiv;
}

static double ** 
_gdl_eqtl_chromosome_cis_test_block_centred_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, double * var)
{
	size_t i, j, k, p;
	double ** Y, s, m;
	
	Y = GDL_MATRIX_ALLOC (double, block->size, chrom->nindiv+1);
	
	for (p = 0; p < block->size; p++)
	{
		var[p] = 0;
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
			{
				Y[p][k]  = block->probes[p]->data[i][j];
				Y[p][k] -= block->probes[p]->mean[i];
				var[p]  += Y[p][k]*Y[p][k];
			}
			//var[p] += (chrom->pop_sizes[i]-1)*block->probes[p]->var[i]/chrom->nindiv;
		}
		var[p] /= chrom->nindiv; 
	}
	
	return Y;
}

//double 
//_get_alpha (const size_t N, double f, double * x, double * y, double * t, double * p)
//{
//	 size_t i;
//	 double a,xy=0,xx=0,e=0;
//	 
//	 for (i = 0; i < N; i++)
//	 {
//	 	 xy+=(x[i]-f)*(y[i+1]-y[0]);
//	 	 xx+=(x[i]-f)*(x[i]-f);
//	 }
//	 a = xy/xx;
//	 for (i = 0; i < N; i++)
//	 {
//	 	 e+=(y[i+1]-y[0]-a*x[i])*(y[i+1]-y[0]-a*x[i]);
//	 }
//	 e /= (N-1);
//	 e /= xx; 
//	 *t = a/sqrt(e);
//	 *p = gdl_sf_beta_inc (0.5*(N-2),0.5,(N-2)/((N-2)+(*t)*(*t)));
//	 
//	 return a;
//}

double
_gdl_eqtl_chromosome_cis_test_perform (gdl_eqtl_chromosome * chrom, const double * x, const double * y, const double sig0, const double * popf, double * a, double * r2, double * p)
{
	size_t i, j, k;
	double l0, l1, chi, e, sig1=0, xx=0, xy=0;
	
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			xx += (x[k]-popf[i])*(x[k]-popf[i]);
			xy += (x[k]-popf[i])*y[k];
		}
	}
	*a=xy/xx;
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			 e = y[k]-(*a)*(x[k]-popf[i]);
			 sig1 += e*e;
		}
	}
	sig1 /= chrom->nindiv;
	chi = (log(sig0)-log(sig1))*chrom->nindiv;
	*p=gdl_sf_gamma_inc_Q (0.5,0.5*chi);
	*r2=1-sig1/sig0;
	return chi;
}

void
gdl_eqtl_chromosome_cis_test (gdl_eqtl_chromosome * chrom, gdl_snp_annot_dico * dico, const double freqmin, const double pmax)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, N1;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double * popf, f, chi, t, pv, r2, * x, ** Y, * var;
	
	snps = chrom->snp_data->chrom->snps;
	
	x = GDL_MALLOC (double, N);
	popf = GDL_MALLOC (double, POP);
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			var   = GDL_MALLOC (double, block->size);
			Y     = _gdl_eqtl_chromosome_cis_test_block_centred_signal (chrom, block, var);
			for (l = block->snp_from; l <= block->snp_to; l++)
			{
				if (snps[l]->polypop=='y')
				{
					_gdl_eqtl_chromosome_cis_test_snp_vector (chrom, l, x, popf, &f);
					if (f < freqmin*P) continue; // exclude rare SNP
					for (p = 0; p < block->size; p++)
					{
						chi = _gdl_eqtl_chromosome_cis_test_perform (chrom, x, Y[p], var[p], popf, &t, &r2, &pv);
						if (chrom->logger && pv < pmax)
						{
							fprintf (chrom->logger, "%s %s %d %d %d %d %s %d %d %s %d %1.f %g %g %g %e\n",  chrom->name,
																	                                               gene->name,
																	                                               gene->id,
																	                                               j+1,
																	                                               block->start,
																	                                               block->end,
																	                                               block->probes[p]->name,
																	                                               block->probes[p]->start,
																	                                               block->probes[p]->end,
																	                                               snps[l]->rs,
																	                                               snps[l]->position,
																	                                               f/P,
																	                                               chi,
																	                                               t,
																	                                               r2,
																	                                               pv);
							gdl_snp_annot_dico_add_count (dico, snps[l]);																	                                               
						}
					}
				}
			}
			GDL_MATRIX_FREE (Y, block->size);
			GDL_FREE (var);
		}
	}
}

void
_gdl_eqtl_chromosome_cis_permut_block_signal (gdl_eqtl_chromosome * chrom, gdl_genex_block * block, double ** Y, const gdl_rng * rng)
{
	size_t i, j, k, l, p;
	double t;
	
	for (p = 0; p < block->size; p++)
	{
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = chrom->pop_sizes[i]-1; j > 0; j--)
			{
				l = (j + 1) * gdl_rng_uniform (rng);
				t = Y[p][k+l];
				Y[p][k+l] = Y[p][k+j];
				Y[p][k+j] = t;
			}
			k += chrom->pop_sizes[i];
		}
	}	
}

void
gdl_eqtl_chromosome_cis_permut (gdl_eqtl_chromosome * chrom, const double freqmin, const size_t permut_size, const gdl_rng * rng)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, b, N1;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double * popf, f, chi, t, pv, pv_permut, r2, * x, ** Y, * var;
	
	snps = chrom->snp_data->chrom->snps;
	
	x = GDL_MALLOC (double, N);
	popf = GDL_MALLOC (double, POP);
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			var   = GDL_MALLOC (double, block->size);
			Y     = _gdl_eqtl_chromosome_cis_test_block_centred_signal (chrom, block, var);
			
			gdl_genex_block_init_permut_storage (block);
			
			for (l = block->snp_from; l <= block->snp_to; l++)
			{
				if (snps[l]->polypop=='y')
				{
					_gdl_eqtl_chromosome_cis_test_snp_vector (chrom, l, x, popf, &f);
					if (f < freqmin*P) continue; // exclude rare SNP
					for (p = 0; p < block->size; p++)
					{
						chi = _gdl_eqtl_chromosome_cis_test_perform (chrom, x, Y[p], var[p], popf, &t, &r2, &pv);
						gdl_genex_block_set_obs_pval (block, p, l, pv);
					}
				}
			}
			
			for (b = 0; b < permut_size; b++)
			{
				_gdl_eqtl_chromosome_cis_permut_block_signal (chrom, block, Y, rng);
				for (l = block->snp_from; l <= block->snp_to; l++)
				{
					if (snps[l]->polypop=='y')
					{
						_gdl_eqtl_chromosome_cis_test_snp_vector (chrom, l, x, popf, &f);
						if (f < freqmin*P) continue; // exclude rare SNP
						// permut vector
						for (p = 0; p < block->size; p++)
						{
							chi = _gdl_eqtl_chromosome_cis_test_perform (chrom, x, Y[p], var[p], popf, &t, &r2, &pv);
							gdl_genex_block_set_permut_pval (block, p, l, pv);
						}
					}
				}
			}
			
			for (l = block->snp_from; l <= block->snp_to; l++)
			{
				if (snps[l]->polypop=='y')
				{
					_gdl_eqtl_chromosome_cis_test_snp_vector (chrom, l, x, popf, &f);
					if (f < freqmin*P) continue; // exclude rare SNP
					for (p = 0; p < block->size; p++)
					{
						if (chrom->logger)
						{
							pv = gdl_genex_block_get_obs_pval (block, p, l);
							pv_permut = gdl_genex_block_get_permut_pval (block, p, l, permut_size);
							fprintf (chrom->logger, "%s %s %d %d %d %d %s %d %d %s %d %1.f %e %e\n",  chrom->name,
																	                                               gene->name,
																	                                               gene->id,
																	                                               j+1,
																	                                               block->start,
																	                                               block->end,
																	                                               block->probes[p]->name,
																	                                               block->probes[p]->start,
																	                                               block->probes[p]->end,
																	                                               snps[l]->rs,
																	                                               snps[l]->position,
																	                                               f/P,
																	                                               pv,
																	                                               pv_permut);
						}
					}
				}
			}
			
			gdl_genex_block_clean_permut_storage (block);
			GDL_MATRIX_FREE (Y, block->size);
			GDL_FREE (var);
		}
	}
}	
