/* gdl/gdl_errno.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_ERRNO_H__
#define __GDL_ERRNO_H__

#include <gdl/gdl_common.h>

__BEGIN_DECLS

enum { 
  GDL_SUCCESS  = 0, 
  GDL_FAILURE  = -1,
  GDL_CONTINUE = -2,  /* iteration has not converged */
  GDL_EDOM     = 1,   /* input domain error, e.g sqrt(-1) */
  GDL_ERANGE   = 2,   /* output range error, e.g. exp(1e100) */
  GDL_EFAULT   = 3,   /* invalid pointer */
  GDL_EINVAL   = 4,   /* invalid argument supplied by user */
  GDL_EFAILED  = 5,   /* generic failure */
  GDL_EFACTOR  = 6,   /* factorization failed */
  GDL_ESANITY  = 7,   /* sanity check failed - shouldn't happen */
  GDL_ENOMEM   = 8,   /* malloc failed */
  GDL_EBADFUNC = 9,   /* problem with user-supplied function */
  GDL_ERUNAWAY = 10,  /* iterative process is out of control */
  GDL_EMAXITER = 11,  /* exceeded max number of iterations */
  GDL_EZERODIV = 12,  /* tried to divide by zero */
  GDL_EBADTOL  = 13,  /* user specified an invalid tolerance */
  GDL_ETOL     = 14,  /* failed to reach the specified tolerance */
  GDL_EUNDRFLW = 15,  /* underflow */
  GDL_EOVRFLW  = 16,  /* overflow  */
  GDL_ELOSS    = 17,  /* loss of accuracy */
  GDL_EROUND   = 18,  /* failed because of roundoff error */
  GDL_EBADLEN  = 19,  /* matrix, vector lengths are not conformant */
  GDL_ENOTSQR  = 20,  /* matrix not square */
  GDL_ESING    = 21,  /* apparent singularity detected */
  GDL_EDIVERGE = 22,  /* integral or series is divergent */
  GDL_EUNSUP   = 23,  /* requested feature is not supported by the hardware */
  GDL_EUNIMPL  = 24,  /* requested feature not (yet) implemented */
  GDL_ECACHE   = 25,  /* cache limit exceeded */
  GDL_ETABLE   = 26,  /* table limit exceeded */
  GDL_ENOPROG  = 27,  /* iteration is not making progress towards solution */
  GDL_ENOPROGJ = 28,  /* jacobian evaluations are not improving the solution */
  GDL_ETOLF    = 29,  /* cannot reach the specified tolerance in F */
  GDL_ETOLX    = 30,  /* cannot reach the specified tolerance in X */
  GDL_ETOLG    = 31,  /* cannot reach the specified tolerance in gradient */
  GDL_EOF      = 32,  /* end of file */
  GDL_DEFVAL   = 33   /* use default value when no value has been given by the user */
} ;

void gdl_error (const char * reason, const char * file, int line,
                int gdl_errno);

void gdl_warning (const char * reason, const char * file, int line,
                int gdl_errno);

void gdl_stream_printf (const char *label, const char *file,
                        int line, const char *reason);

const char * gdl_strerror (const int gdl_errno);

typedef void gdl_error_handler_t (const char * reason, const char * file,
                                  int line, int gdl_errno);

typedef void gdl_warning_handler_t (const char * reason, const char * file,
                                  int line, int gdl_errno);                                  

typedef void gdl_stream_handler_t (const char * label, const char * file,
                                   int line, const char * reason);

gdl_error_handler_t * 
gdl_set_error_handler (gdl_error_handler_t * new_handler);

gdl_error_handler_t *
gdl_set_error_handler_off (void);

gdl_warning_handler_t * 
gdl_set_warning_handler (gdl_warning_handler_t * new_handler);

gdl_warning_handler_t *
gdl_set_warning_handler_off (void);

gdl_stream_handler_t * 
gdl_set_stream_handler (gdl_stream_handler_t * new_handler);

FILE * gdl_set_stream (FILE * new_stream);

/* GDL_WARNING: call the warning handler, and return the warning code */

#define GDL_WARNING(reason, gdl_errno) \
       gdl_warning (reason, __FILE__, __LINE__, gdl_errno) ; \
       

/* GDL_ERROR: call the error handler, and return the error code */

#define GDL_ERROR(reason, gdl_errno) \
       do { \
       gdl_error (reason, __FILE__, __LINE__, gdl_errno) ; \
       return gdl_errno ; \
       } while (0)

/* GDL_ERROR_VAL: call the error handler, and return the given value */

#define GDL_ERROR_VAL(reason, gdl_errno, value) \
       do { \
       gdl_error (reason, __FILE__, __LINE__, gdl_errno) ; \
       return value ; \
       } while (0)

/* GDL_ERROR_VOID: call the error handler, and then return
   (for void functions which still need to generate an error) */

#define GDL_ERROR_VOID(reason, gdl_errno) \
       do { \
       gdl_error (reason, __FILE__, __LINE__, gdl_errno) ; \
       return ; \
       } while (0)

/* GDL_ERROR_NULL suitable for out-of-memory conditions */

#define GDL_ERROR_NULL(reason, gdl_errno) GDL_ERROR_VAL(reason, gdl_errno, 0)

/* Sometimes you have several status results returned from
 * function calls and you want to combine them in some sensible
 * way. You cannot produce a "total" status condition, but you can
 * pick one from a set of conditions based on an implied hierarchy.
 *
 * In other words:
 *    you have: status_a, status_b, ...
 *    you want: status = (status_a if it is bad, or status_b if it is bad,...)
 *
 * In this example you consider status_a to be more important and
 * it is checked first, followed by the others in the order specified.
 *
 * Here are some dumb macros to do this.
 */
#define GDL_ERROR_SELECT_2(a,b)       ((a) != GDL_SUCCESS ? (a) : ((b) != GDL_SUCCESS ? (b) : GDL_SUCCESS))
#define GDL_ERROR_SELECT_3(a,b,c)     ((a) != GDL_SUCCESS ? (a) : GDL_ERROR_SELECT_2(b,c))
#define GDL_ERROR_SELECT_4(a,b,c,d)   ((a) != GDL_SUCCESS ? (a) : GDL_ERROR_SELECT_3(b,c,d))
#define GDL_ERROR_SELECT_5(a,b,c,d,e) ((a) != GDL_SUCCESS ? (a) : GDL_ERROR_SELECT_4(b,c,d,e))

#define GDL_STATUS_UPDATE(sp, s) do { if ((s) != GDL_SUCCESS) *(sp) = (s);} while(0)
/**
 * Here, a simple macro to check the output of fread().
 */
#define GDL_FREAD_STATUS(s, n) if (s != n) \
    { GDL_ERROR_VOID ("fread failed", GDL_EFAILED);}
/**
 * Here, a simple macro to check the output of fwrite().
 */
#define GDL_FWRITE_STATUS(s, n) if (s != n) \
    { GDL_ERROR_VOID ("fwrite failed", GDL_EFAILED);}

__END_DECLS

#endif /* __GDL_ERRNO_H__ */
