/*  
 *  fmatrix/fmatrix.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_fview.h>
//#include <gdl/gdl_glabels.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_fmatrix.h>

struct _gdl_fmatrix
{
	size_t size1;
	size_t size2;
	size_t nf;
	size_t * nc;
	size_t * tnc;
	gdl_matrix * data;
	const gdl_fview         * fview;
	const gdl_mask          * mask;
	const gdl_fview_wrapper * fwrap;
	gdl_boolean last_level;
};

static void
gdl_fmatrix_factor_init (gdl_fmatrix * fm, size_t i, size_t j, const gdl_gvalues * x, gdl_boolean last_level)
{
	size_t k, jj;
	
	if (x)
	{
		for (k = 0; k < x->size; k++)
		{
			if ((x->values[k]->idx < fm->nc[j]) || last_level)
			{
				jj = fm->tnc[j] + x->values[k]->idx;
				gdl_matrix_set (fm->data, i, jj, x->values[k]->value);
			}
		}
	}
	else
	{
		gdl_fmatrix_factor_set_missing (fm, i, j);
	}
} 

static int
gdl_fmatrix_init (gdl_fmatrix * fm, const gdl_fview * fview, const gdl_mask * mask, gdl_boolean last_level)
{
	size_t i, j, k, l, nf, nn;
	gdl_factor * factor;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	nn = GDL_FVIEW_ACCESSION_SIZE (fview, mask);
	nf = GDL_FVIEW_FACTOR_SIZE (fview, mask);
	
	fm->size1  = nn;
	fm->nf     = nf;
	fm->nc     = GDL_MALLOC (size_t, nf);
	fm->tnc    = GDL_MALLOC (size_t, nf);
	fm->tnc[0] = 0;
	
	for (i = 0; i < nf; i++)
	{
		factor     = GDL_FVIEW_GET_FACTOR (fview, mask, i);
		fm->nc[i]  = gdl_factor_size (factor);
		if (gdl_factor_get_type (factor) != gdl_factor_continuous && fm->nc[i] <= 1)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Factor %s is not informative", gdl_entity_get_name (factor)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (gdl_factor_get_type (factor) != gdl_factor_continuous && !last_level)
		{
			(fm->nc[i])--;
		}
		if (i)
		{
			fm->tnc[i] = fm->tnc[i-1] + fm->nc[i-1];
		}
	}
	
	fm->size2 = fm->tnc[i-1] + fm->nc[i-1];
	
	fm->data  = gdl_matrix_calloc (fm->size1, fm->size2);
	
	gbuf = GDL_FVIEW_GET_NEW (fview, mask);
	
	for (i = 0; i < nn; i++)
	{
		for (j = 0; j < nf; j++)
		{
			GDL_FVIEW_GET_VALUE (fview, mask, i, j, gbuf);
			gdl_fmatrix_factor_init (fm, i, j, gdl_gvalues_get_gvalues (gbuf), last_level);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static int
gdl_fmatrix_init_wrapper (gdl_fmatrix * fm, const gdl_fview_wrapper * fwrap, gdl_boolean last_level)
{
	size_t i, j, k, l, nf, nn;
	gdl_factor * factor;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	nn = gdl_fview_wrapper_accession_size (fwrap);
	nf = gdl_fview_wrapper_factor_size (fwrap);
	
	fm->size1  = nn;
	fm->nf     = nf;
	fm->nc     = GDL_MALLOC (size_t, nf);
	fm->tnc    = GDL_MALLOC (size_t, nf);
	fm->tnc[0] = 0;
	
	for (i = 0; i < nf; i++)
	{
		factor     = gdl_fview_wrapper_get_factor (fwrap, i);
		fm->nc[i]  = gdl_factor_size (factor);
		if (gdl_factor_get_type (factor) != gdl_factor_continuous && fm->nc[i] <= 1)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Factor %s is not informative", gdl_entity_get_name (factor)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (gdl_factor_get_type (factor) != gdl_factor_continuous && !last_level)
		{
			(fm->nc[i])--;
		}
		if (i)
		{
			fm->tnc[i] = fm->tnc[i-1] + fm->nc[i-1];
		}
	}
	
	fm->size2 = fm->tnc[i-1] + fm->nc[i-1];
	
	fm->data  = gdl_matrix_calloc (fm->size1, fm->size2);
	
	gbuf = gdl_fview_wrapper_get_new (fwrap);
	
	for (i = 0; i < nn; i++)
	{
		for (j = 0; j < nf; j++)
		{
			gdl_fview_wrapper_get_value (fwrap, i, j, gbuf);
			gdl_fmatrix_factor_init (fm, i, j, gdl_gvalues_get_gvalues (gbuf), last_level);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static int
gdl_fmatrix_init_wrapper_mask (gdl_fmatrix * fm, const gdl_fview_wrapper * fwrap, const gdl_mask * mask, gdl_boolean last_level)
{
	size_t i, j, k, l, nf, nn;
	gdl_factor * factor;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	nn = gdl_mask_size (mask, GDL_ACCESSION);
	if (!nn)
	{
		nn = gdl_fview_wrapper_accession_size (fwrap);
	}
	nf = gdl_mask_size (mask, GDL_FACTOR);
	if (!nf)
	{
		nf = gdl_fview_wrapper_factor_size (fwrap);
	}
	
	fm->size1  = nn;
	fm->nf     = nf;
	fm->nc     = GDL_MALLOC (size_t, nf);
	fm->tnc    = GDL_MALLOC (size_t, nf);
	fm->tnc[0] = 0;
	
	for (i = 0; i < nf; i++)
	{
		factor     = gdl_fview_wrapper_get_factor (fwrap, gdl_mask_get_idx (mask, GDL_FACTOR, i));
		fm->nc[i]  = gdl_factor_size (factor);
		if (gdl_factor_get_type (factor) != gdl_factor_continuous && fm->nc[i] <= 1)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Factor %s is not informative", gdl_entity_get_name (factor)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (gdl_factor_get_type (factor) != gdl_factor_continuous && !last_level)
		{
			(fm->nc[i])--;
		}
		if (i)
		{
			fm->tnc[i] = fm->tnc[i-1] + fm->nc[i-1];
		}
	}
	
	fm->size2 = fm->tnc[i-1] + fm->nc[i-1];
	
	fm->data  = gdl_matrix_calloc (fm->size1, fm->size2);
	
	gbuf = gdl_fview_wrapper_get_new (fwrap);
	
	for (i = 0; i < nn; i++)
	{
		for (j = 0; j < nf; j++)
		{
			gdl_fview_wrapper_get_value (fwrap, gdl_mask_get_idx (mask, GDL_ACCESSION, i), gdl_mask_get_idx (mask, GDL_FACTOR, j), gbuf);
			gdl_fmatrix_factor_init (fm, i, j, gdl_gvalues_get_gvalues (gbuf), last_level);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

gdl_fmatrix *
gdl_fmatrix_alloc (const gdl_fview * fview, const gdl_mask * mask, gdl_boolean last_level)
{
	gdl_fmatrix * fm;
	
	fm = GDL_CALLOC (gdl_fmatrix, 1);
	
	fm->fview = fview;
	fm->mask  = mask;
	fm->last_level = last_level;
	
	gdl_fmatrix_init (fm, fview, mask, last_level);
	
	return fm;
}

gdl_fmatrix *
gdl_fmatrix_wrapper_alloc (const gdl_fview_wrapper * fwrap, gdl_boolean last_level)
{
	gdl_fmatrix * fm;
	
	fm = GDL_CALLOC (gdl_fmatrix, 1);
	
	fm->fwrap = fwrap;
	fm->last_level = last_level;
	
	gdl_fmatrix_init_wrapper (fm, fwrap, last_level);
	
	return fm;
	
}

gdl_fmatrix *
gdl_fmatrix_wrapper_mask_alloc (const gdl_fview_wrapper * fwrap, const gdl_mask * mask, gdl_boolean last_level)
{
	gdl_fmatrix * fm;
	
	fm = GDL_CALLOC (gdl_fmatrix, 1);
	
	fm->fwrap = fwrap;
	fm->mask  = mask;
	fm->last_level = last_level;
	
	gdl_fmatrix_init_wrapper_mask (fm, fwrap, mask, last_level);
	
	return fm;	
}

void
gdl_fmatrix_free (gdl_fmatrix * m)
{
	if (m)
	{
		GDL_FREE (m->nc);
		GDL_FREE (m->tnc);
		//gdl_glabels_free (m->labels);
		gdl_matrix_free (m->data);
		GDL_FREE (m);
	}	
}

gdl_mask *
gdl_fmatrix_get_mask (const gdl_fmatrix * m)
{
	if (m->fview && m->mask)
	{
		return gdl_mask_clone (m->mask);
	}
	else if (m->fwrap && !m->mask)
	{
		return gdl_mask_clone (gdl_fview_wrapper_mask (m->fwrap));
	}
	else if (m->fwrap && m->mask)
	{
		return gdl_mask_clone (m->mask);
	}
	
	return NULL;
}

size_t
gdl_fmatrix_row_size (const gdl_fmatrix * m)
{
	return m->size1;	
}

size_t
gdl_fmatrix_column_size (const gdl_fmatrix * m)
{
	return m->size2;
}

size_t
gdl_fmatrix_accession_size (const gdl_fmatrix * m)
{
	return m->size1;	
}

size_t
gdl_fmatrix_factor_size (const gdl_fmatrix * m)
{
	return m->nf;	
}

size_t
gdl_fmatrix_factor_column_size (const gdl_fmatrix * m, size_t i)
{
	return m->nc[i];	
}

size_t
gdl_fmatrix_factor_first_column (const gdl_fmatrix * m, size_t i)
{
	return m->tnc[i];
}

double
gdl_fmatrix_get (const gdl_fmatrix * m, size_t i, size_t j)
{
	return gdl_matrix_get (m->data, i, j);	
}

void
gdl_fmatrix_set (gdl_fmatrix * m, size_t i, size_t j, double x)
{
	gdl_matrix_set (m->data, i, j, x);
}

void
gdl_fmatrix_factor_set (gdl_fmatrix * m, size_t i, size_t j, const gdl_gvalues * x)
{
	size_t k, jj;
	
	if (x)
	{
		for (k = 0; k < x->size; k++)
		{
			if (x->values[k]->idx < m->nc[j])
			{
				jj = m->tnc[j] + x->values[k]->idx;
				gdl_matrix_set (m->data, i, jj, x->values[k]->value);
			}
		}
	}
	else
	{
		gdl_fmatrix_factor_set_missing (m, i, j);	
	}	
}

gdl_boolean
gdl_fmatrix_factor_is_missing (const gdl_fmatrix * m, size_t i, size_t j)
{
	return (gdl_isnan (gdl_matrix_get (m->data, i, m->tnc[j]))) ? gdl_true : gdl_false;
}

gdl_gvalues *
gdl_fmatrix_factor_get (const gdl_fmatrix * m, size_t i, size_t j)
{
	if (!gdl_fmatrix_factor_is_missing (m, i, j))
	{
		size_t k, n;
		gdl_gvalues * glx;
		
		for (n = 0, k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
		{
			if (gdl_matrix_get (m->data, i, k))
			{
				n++;	
			}
		}
		
		glx = gdl_gvalues_alloc (n);
		
		for (n = 0, k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
		{
			if (gdl_matrix_get (m->data, i, k))
			{
				glx->values[n]        = gdl_gvalue_alloc ();
				glx->values[n]->idx   = k-m->tnc[j];
				glx->values[n]->value = gdl_matrix_get (m->data, i, k);
				n++;
			}
		}
		
		return glx;
	}
	
	return NULL;
}

void
gdl_fmatrix_factor_set_missing (gdl_fmatrix * m, size_t i, size_t j)
{
	size_t k;
	
	for (k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
	{
		gdl_matrix_set (m->data, i, k, GDL_NAN);
	}
}

void
gdl_fmatrix_factor_set_all (gdl_fmatrix * m, size_t i, size_t j, double x)
{
	size_t k;
	
	for (k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
	{
		gdl_matrix_set (m->data, i, k, x);
	}
}

double
gdl_fmatrix_factor_column_get (gdl_fmatrix * m, size_t i, size_t j, size_t a)
{
	return gdl_matrix_get (m->data, i, m->tnc[j]+a);	
}

void
gdl_fmatrix_factor_column_set (gdl_fmatrix * m, size_t i, size_t j, size_t a, double x)
{
	gdl_matrix_set (m->data, i, m->tnc[j]+a, x);
}

const gdl_matrix *
gdl_fmatrix_get_matrix (const gdl_fmatrix * m)
{
	return m->data;	
}

gdl_flabels *
gdl_fmatrix_get_labels (const gdl_fmatrix * m)
{
	if (m->fview)
	{
		return gdl_flabels_alloc (m->fview, m->mask, m->last_level);	
	}
	else if (m->fwrap && !m->mask)
	{
		return gdl_flabels_wrapper_alloc (m->fwrap, m->last_level);
	}
	else if (m->fwrap && m->mask)
	{
		return gdl_flabels_wrapper_mask_alloc (m->fwrap, m->mask, m->last_level);
	}
}

int
gdl_fmatrix_fprintf (FILE * stream, const gdl_fmatrix * m)
{
	size_t i, j;
	
	printf ("FMATRIX [ %d , %d ]\n", m->size1, m->size2);
	
	for (i = 0; i < m->size1; i++)
	{
		for (j = 0; j < m->size2; j++)
		{
			if (!gdl_isnan (gdl_matrix_get (m->data, i, j)))
			{
				printf (" %1.4f", gdl_matrix_get (m->data, i, j));
			}
			else
			{
				printf (" NA");
			}
		}
		printf ("\n");
	}
	
	return GDL_SUCCESS;
}
