/*  
 *  fpoint/gdl_fpoint.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_nan.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gpoint.h>
#include <gdl/gdl_fpoint.h>
#include <gdl/gdl_check_range.h>

typedef struct
{
	int * idx;
	gdl_gvalues_registry * block;
} gdl_fdatapoint_gblock;

typedef struct 
{
	size_t size;
	const gdl_factor_type * type;
	void * block;
} gdl_fdatapoint_block;

struct _gdl_fdatapoints
{
	size_t n;
	size_t f;
	gdl_fdatapoint_block ** blocks;
};

static gdl_fdatapoint_gblock *
gdl_fdatapoint_gblock_alloc (size_t size)
{
	size_t i;
	gdl_fdatapoint_gblock * gb;
	
	gb = GDL_MALLOC (gdl_fdatapoint_gblock, 1);
	
	gb->idx  = GDL_MALLOC (int, size);
	for (i = 0; i < size; i++)
	{
		gb->idx[i] = -1;	
	}
	gb->block = gdl_gvalues_registry_alloc ();
	
	return gb;
}

static void
gdl_fdatapoint_gblock_free (gdl_fdatapoint_gblock * gb)
{
	if (gb)
	{
		GDL_FREE (gb->idx);
		gdl_gvalues_registry_free (gb->block);
		GDL_FREE (gb);	
	}
}

static gdl_fdatapoint_block *
gdl_fdatapoint_block_alloc (const gdl_factor_type * T, size_t n)
{
	gdl_fdatapoint_block * b;
	
	b = GDL_MALLOC (gdl_fdatapoint_block, 1);
	
	b->type = T;
	b->size = n;
	
	if (T == gdl_factor_continuous)
	{
		size_t i;
		double * y;
		
		y = GDL_MALLOC (double, n);
		
		for (i = 0; i < n; i++)
		{
			y[i] = GDL_NAN;	
		}
		
		b->block = y;
	}
	else
	{
		b->block = gdl_fdatapoint_gblock_alloc (n);
	}
	
	return b;	
}

static void
gdl_fdatapoint_block_free (gdl_fdatapoint_block * b)
{
	if (b)
	{
		if (b->type == gdl_factor_continuous)
		{
			GDL_FREE (b->block);
		}
		else
		{
			gdl_fdatapoint_gblock_free ((gdl_fdatapoint_gblock *)b->block);		
		}
		GDL_FREE (b);
	}
}

gdl_fdatapoints *
gdl_fdatapoints_alloc (size_t n, size_t f, gdl_factor ** factors)
{
	size_t i;
	gdl_fdatapoints * fp;
	
	fp = GDL_MALLOC (gdl_fdatapoints, 1);
	
	fp->n = n;
	fp->f = f;
	
	fp->blocks = GDL_MALLOC (gdl_fdatapoint_block *, f);
	
	for (i = 0; i < fp->f; i++)
	{
		fp->blocks[i] = gdl_fdatapoint_block_alloc (gdl_factor_get_type (factors[i]), n);
	}
	
	return fp;
}

void
gdl_fdatapoints_free (gdl_fdatapoints * f)
{
	if (f)
	{
		size_t i;
		for (i = 0; i < f->f; i++)
		{
			gdl_fdatapoint_block_free (f->blocks[i]);	
		}
		GDL_FREE (f->blocks);
		GDL_FREE (f);	
	}
}

static int
gdl_fdatapoint_block_set (gdl_fdatapoint_block * b, size_t j, const gdl_fdatapoint * x)
{
	if (b->type == gdl_factor_continuous)
	{
		double * y = (double *) b->block;
		
		if (x && x->size)
		{
			y[j] = x->values[0]->value;
		}
		else
		{
			y[j] = GDL_NAN;
		}
		
		return GDL_SUCCESS;
	}
	else
	{
		gdl_fdatapoint_gblock * gb = (gdl_fdatapoint_gblock *) b->block;
		
		if (x && x->size)
		{
			gb->idx[j] = gdl_gvalues_registry_add (gb->block, x);
		}
		else
		{
			gb->idx[j] = -1;
		}
		
		return GDL_SUCCESS;
	}
}

int
gdl_fdatapoints_set (gdl_fdatapoints * f, size_t i, size_t j, const gdl_fdatapoint * x)
{
	if (gdl_check_range)
	{
		if (i >= f->n || j >= f->f)
		{
		    GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
		}
	}
	return gdl_fdatapoint_block_set (f->blocks[j], i, x);	
}

static gdl_gvalues *
gdl_fdatapoint_block_get (const gdl_fdatapoint_block * b, size_t j)
{
	gdl_gvalues * x = NULL;
	
	if (b->type == gdl_factor_continuous)
	{
		double * y = (double *) b->block;
		if (!gdl_isnan (y[j]))
		{
			x            = gdl_gvalues_alloc (1);
			x->values[0] = gdl_gvalue_alloc ();
			x->values[0]->idx   = 0;
			x->values[0]->value = y[j];
		}
	}
	else
	{
		gdl_fdatapoint_gblock * gb = (gdl_fdatapoint_gblock *) b->block;
		if (gb->idx[j]>=0)
		{
			x = gdl_gvalues_clone (gdl_gvalues_registry_get (gb->block, gb->idx[j]));
		}
	}
	
	return x;
}

gdl_fdatapoint *
gdl_fdatapoints_get (const gdl_fdatapoints * f, size_t i, size_t j)
{
	if (gdl_check_range)
	{
		if (i >= f->n || j >= f->f)
		{
		    GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
		}
	}
	return gdl_fdatapoint_block_get (f->blocks[j], i);
}

static gdl_boolean
gdl_fdatapoint_block_is_missing (const gdl_fdatapoint_block * b, size_t j)
{
	gdl_boolean x = gdl_true;
	
	if (b->type == gdl_factor_continuous)
	{
		double * y = (double *) b->block;
		if (!gdl_isnan (y[j]))
		{
			x = gdl_false;
		}
	}
	else
	{
		gdl_fdatapoint_gblock * gb = (gdl_fdatapoint_gblock *) b->block;
		if (gb->idx[j]>=0)
		{
			x = gdl_false;
		}
	}
	
	return x;
}


gdl_boolean
gdl_fdatapoints_is_missing (const gdl_fdatapoints * f, size_t i, size_t j)
{
	if (gdl_check_range)
	{
		if (i >= f->n || j >= f->f)
		{
		    GDL_ERROR_VAL ("index out of range", GDL_EINVAL, 0);
		}
	}
	return gdl_fdatapoint_block_is_missing (f->blocks[j], i);
}

static gdl_fdatapoint_gblock *
gdl_fdatapoint_gblock_fread (FILE * stream, size_t size)
{
	if (stream)
	{
		int status;
		gdl_fdatapoint_gblock * gb;
		
		gb = GDL_MALLOC (gdl_fdatapoint_gblock, 1);
		
		gb->idx = GDL_MALLOC (int, size);
		status = fread (gb->idx, sizeof (int), size, stream);
		GDL_FREAD_STATUS (status, size);
		gb->block = gdl_gvalues_registry_fread (stream);
		GDL_FREAD_STATUS (gb->block!=0, 1);
		
		return gb;
	}
	
	return NULL;	
}

static gdl_fdatapoint_block *
gdl_fdatapoint_block_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_fdatapoint_block * b;
		
		b = GDL_MALLOC (gdl_fdatapoint_block, 1);
		
		b->type = gdl_factor_type_fread (stream);
		GDL_FREAD_STATUS (b->type!=0, 1);
		
		status  = fread (&(b->size), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (b->type == gdl_factor_continuous)
		{
			b->block = GDL_MALLOC (double, b->size);
			status = fread (b->block, sizeof (double), b->size, stream);
			GDL_FREAD_STATUS (status, b->size);
		}
		else
		{
			b->block = gdl_fdatapoint_gblock_fread (stream, b->size);
			GDL_FREAD_STATUS (b->block!=0, 1);
		}
		
		return b;
	}
	
	return NULL;	
}

gdl_fdatapoints *
gdl_fdatapoints_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_fdatapoints * f;
		
		f = GDL_MALLOC (gdl_fdatapoints, 1);
		
		status = fread (&(f->n), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		status = fread (&(f->f), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		f->blocks = GDL_MALLOC (gdl_fdatapoint_block *, f->f);
		
		for (i = 0; i < f->f; i++)
		{
			f->blocks[i] = gdl_fdatapoint_block_fread (stream);
			GDL_FREAD_STATUS (f->blocks[i]!=0, 1);
		}
		
		return f;
	}
	
	return NULL;
}

static int
gdl_fdatapoint_gblock_fwrite (FILE * stream, const gdl_fdatapoint_gblock * gb, const size_t size)
{
	if (stream && gb)
	{
		int status;
		
		status = fwrite (gb->idx, sizeof (int), size, stream);
		GDL_FWRITE_STATUS (status, size);
		status = gdl_gvalues_registry_fwrite (stream, gb->block);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

static int
gdl_fdatapoint_block_fwrite (FILE * stream, const gdl_fdatapoint_block * b)
{
	if (stream && b)
	{
		int status;
		
		status = gdl_factor_type_fwrite (stream, b->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = fwrite (&(b->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (b->type == gdl_factor_continuous)
		{
			status = fwrite (b->block, sizeof (double), b->size, stream);
			GDL_FWRITE_STATUS (status, b->size);
		}
		else
		{
			status = gdl_fdatapoint_gblock_fwrite (stream, b->block, b->size);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

int
gdl_fdatapoints_fwrite (FILE * stream, const gdl_fdatapoints * f)
{
	if (stream && f)
	{
		int status;
		size_t i;
		
		status = fwrite (&(f->n), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(f->f), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < f->f; i++)
		{
			status = gdl_fdatapoint_block_fwrite (stream, f->blocks[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}
